// Copyright 2015, VIXL authors
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
//   * Redistributions of source code must retain the above copyright notice,
//     this list of conditions and the following disclaimer.
//   * Redistributions in binary form must reproduce the above copyright notice,
//     this list of conditions and the following disclaimer in the documentation
//     and/or other materials provided with the distribution.
//   * Neither the name of ARM Limited nor the names of its contributors may be
//     used to endorse or promote products derived from this software without
//     specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS CONTRIBUTORS "AS IS" AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
// DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
// SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
// CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
// OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


#include <cstdio>
#include <cstring>
#include <string>

#include "test-runner.h"

#include "aarch64/disasm-aarch64.h"
#include "aarch64/macro-assembler-aarch64.h"
#include "test-disasm-aarch64.h"

namespace vixl {
namespace aarch64 {

TEST(bootstrap) {
  SETUP();

  // Instructions generated by C compiler, disassembled by objdump, and
  // reformatted to suit our disassembly style.
  COMPARE(dci(0xa9ba7bfd), "stp x29, x30, [sp, #-96]!");
  COMPARE(dci(0x910003fd), "mov x29, sp");
  COMPARE(dci(0x9100e3a0), "add x0, x29, #0x38 (56)");
  COMPARE(dci(0xb900001f), "str wzr, [x0]");
  COMPARE(dci(0x528000e1), "mov w1, #0x7");
  COMPARE(dci(0xb9001c01), "str w1, [x0, #28]");
  COMPARE(dci(0x390043a0), "strb w0, [x29, #16]");
  COMPARE(dci(0x790027a0), "strh w0, [x29, #18]");
  COMPARE(dci(0xb9400400), "ldr w0, [x0, #4]");
  COMPARE(dci(0x0b000021), "add w1, w1, w0");
  COMPARE(dci(0x531b6800), "lsl w0, w0, #5");
  COMPARE(dci(0x521e0400), "eor w0, w0, #0xc");
  COMPARE(dci(0x72af0f00), "movk w0, #0x7878, lsl #16");
  COMPARE(dci(0xd360fc00), "lsr x0, x0, #32");
  COMPARE(dci(0x13037c01), "asr w1, w0, #3");
  COMPARE(dci(0x4b000021), "sub w1, w1, w0");
  COMPARE(dci(0x2a0103e0), "mov w0, w1");
  COMPARE(dci(0x93407c00), "sxtw x0, w0");
  COMPARE(dci(0x2a000020), "orr w0, w1, w0");
  COMPARE(dci(0xa8c67bfd), "ldp x29, x30, [sp], #96");

  CLEANUP();
}


TEST(mov_mvn) {
  SETUP();

  COMPARE_MACRO(Mov(w0, Operand(0x1234)), "mov w0, #0x1234");
  COMPARE_MACRO(Mov(x1, Operand(0x1234)), "mov x1, #0x1234");
  COMPARE_MACRO(Mov(w2, Operand(w3)), "mov w2, w3");
  COMPARE_MACRO(Mov(x4, Operand(x5)), "mov x4, x5");
  COMPARE_MACRO(Mov(w6, Operand(w7, LSL, 5)), "lsl w6, w7, #5");
  COMPARE_MACRO(Mov(x8, Operand(x9, ASR, 42)), "asr x8, x9, #42");
  COMPARE_MACRO(Mov(w10, Operand(w11, UXTB)), "uxtb w10, w11");
  COMPARE_MACRO(Mov(x12, Operand(x13, UXTB, 1)), "ubfiz x12, x13, #1, #8");
  COMPARE_MACRO(Mov(w14, Operand(w15, SXTH, 2)), "sbfiz w14, w15, #2, #16");
  COMPARE_MACRO(Mov(x16, Operand(x17, SXTW, 3)), "sbfiz x16, x17, #3, #32");

  COMPARE_MACRO(Mvn(w0, Operand(0x101)), "mov w0, #0xfffffefe");
  COMPARE_MACRO(Mvn(x1, Operand(0xfff1)), "mov x1, #0xffffffffffff000e");
  COMPARE_MACRO(Mvn(w2, Operand(w3)), "mvn w2, w3");
  COMPARE_MACRO(Mvn(x4, Operand(x5)), "mvn x4, x5");
  COMPARE_MACRO(Mvn(w6, Operand(w7, LSL, 12)), "mvn w6, w7, lsl #12");
  COMPARE_MACRO(Mvn(x8, Operand(x9, ASR, 63)), "mvn x8, x9, asr #63");

  CLEANUP();
}

TEST(mvn_macro) {
  SETUP();

  // Mvn uses the destination register as a scratch if it can. This only occurs
  // when `operand.IsExtendedRegister()`.
  COMPARE_MACRO(Mvn(x0, Operand(w1, SXTW)),
                "sxtw x0, w1\n"
                "mvn x0, x0");
  COMPARE_MACRO(Mvn(x0, Operand(x0, SXTW)),
                "sxtw x0, w0\n"
                "mvn x0, x0");
}


TEST(move_immediate) {
  SETUP();

  COMPARE(movz(w0, 0x1234), "mov w0, #0x1234");
  COMPARE(movz(x1, 0xabcd0000), "mov x1, #0xabcd0000");
  COMPARE(movz(x2, 0x555500000000), "mov x2, #0x555500000000");
  COMPARE(movz(x3, 0xaaaa000000000000), "mov x3, #0xaaaa000000000000");
  COMPARE(movz(x4, 0xabcd, 16), "mov x4, #0xabcd0000");
  COMPARE(movz(x5, 0x5555, 32), "mov x5, #0x555500000000");
  COMPARE(movz(x6, 0xaaaa, 48), "mov x6, #0xaaaa000000000000");

  COMPARE(movk(w7, 0x1234), "movk w7, #0x1234");
  COMPARE(movk(x8, 0xabcd0000), "movk x8, #0xabcd, lsl #16");
  COMPARE(movk(x9, 0x555500000000), "movk x9, #0x5555, lsl #32");
  COMPARE(movk(x10, 0xaaaa000000000000), "movk x10, #0xaaaa, lsl #48");
  COMPARE(movk(w11, 0xabcd, 16), "movk w11, #0xabcd, lsl #16");
  COMPARE(movk(x12, 0x5555, 32), "movk x12, #0x5555, lsl #32");
  COMPARE(movk(x13, 0xaaaa, 48), "movk x13, #0xaaaa, lsl #48");

  COMPARE(movn(w14, 0x1234), "mov w14, #0xffffedcb");
  COMPARE(movn(x15, 0xabcd0000), "mov x15, #0xffffffff5432ffff");
  COMPARE(movn(x16, 0x555500000000), "mov x16, #0xffffaaaaffffffff");
  COMPARE(movn(x17, 0xaaaa000000000000), "mov x17, #0x5555ffffffffffff");
  COMPARE(movn(w18, 0xabcd, 16), "mov w18, #0x5432ffff");
  COMPARE(movn(x19, 0x5555, 32), "mov x19, #0xffffaaaaffffffff");
  COMPARE(movn(x20, 0xaaaa, 48), "mov x20, #0x5555ffffffffffff");

  COMPARE(mov(w14, 0x1234), "mov w14, #0x1234");
  COMPARE(mov(x15, 0xabcd0000), "mov x15, #0xabcd0000");
  COMPARE(mov(x16, 0xaaaa000000000000), "mov x16, #0xaaaa000000000000");
  COMPARE(mov(w17, 0xaaaaffff), "mov w17, #0xaaaaffff");
  COMPARE(mov(x18, 0xffffaaaaffffffff), "mov x18, #0xffffaaaaffffffff");
  COMPARE(mov(x19, 0xffffffffffffffff), "mov x19, #0xffffffffffffffff");
  COMPARE(mov(x20, 0xc001c001c001c001), "mov x20, #0xc001c001c001c001");
  COMPARE(mov(sp, 0xfefefefefefefefe), "mov sp, #0xfefefefefefefefe");

  COMPARE(movk(w21, 0), "movk w21, #0x0");
  COMPARE(movk(x22, 0, 0), "movk x22, #0x0");
  COMPARE(movk(w23, 0, 16), "movk w23, #0x0, lsl #16");
  COMPARE(movk(x24, 0, 32), "movk x24, #0x0, lsl #32");
  COMPARE(movk(x25, 0, 48), "movk x25, #0x0, lsl #48");

  COMPARE(movz(x26, 0, 48), "movz x26, #0x0");
  COMPARE(movn(x27, 0, 48), "movn x27, #0x0");
  COMPARE(movn(w28, 0xffff), "movn w28, #0xffff");

  CLEANUP();
}


TEST(move_immediate_2) {
  SETUP();

  // Move instructions expected for certain immediates. This is really a macro
  // assembler test, to ensure it generates immediates efficiently.
  COMPARE_MACRO(Mov(w0, 0), "mov w0, #0x0");
  COMPARE_MACRO(Mov(w0, 0x0000ffff), "mov w0, #0xffff");
  COMPARE_MACRO(Mov(w0, 0x00010000), "mov w0, #0x10000");
  COMPARE_MACRO(Mov(w0, 0xffff0000), "mov w0, #0xffff0000");
  COMPARE_MACRO(Mov(w0, 0x0001ffff), "mov w0, #0x1ffff");
  COMPARE_MACRO(Mov(w0, 0xffff8000), "mov w0, #0xffff8000");
  COMPARE_MACRO(Mov(w0, 0xfffffffe), "mov w0, #0xfffffffe");
  COMPARE_MACRO(Mov(w0, 0xffffffff), "mov w0, #0xffffffff");
  COMPARE_MACRO(Mov(w0, 0x00ffff00), "mov w0, #0xffff00");
  COMPARE_MACRO(Mov(w0, 0xfffe7fff), "mov w0, #0xfffe7fff");
  COMPARE_MACRO(Mov(w0, 0xfffeffff), "mov w0, #0xfffeffff");
  COMPARE_MACRO(Mov(w0, 0xffff7fff), "mov w0, #0xffff7fff");

  COMPARE_MACRO(Mov(x0, 0), "mov x0, #0x0");
  COMPARE_MACRO(Mov(x0, 0x0000ffff), "mov x0, #0xffff");
  COMPARE_MACRO(Mov(x0, 0x00010000), "mov x0, #0x10000");
  COMPARE_MACRO(Mov(x0, 0xffff0000), "mov x0, #0xffff0000");
  COMPARE_MACRO(Mov(x0, 0x0001ffff), "mov x0, #0x1ffff");
  COMPARE_MACRO(Mov(x0, 0xffff8000), "mov x0, #0xffff8000");
  COMPARE_MACRO(Mov(x0, 0xfffffffe), "mov x0, #0xfffffffe");
  COMPARE_MACRO(Mov(x0, 0xffffffff), "mov x0, #0xffffffff");
  COMPARE_MACRO(Mov(x0, 0x00ffff00), "mov x0, #0xffff00");
  COMPARE_MACRO(Mov(x0, 0xffff000000000000), "mov x0, #0xffff000000000000");
  COMPARE_MACRO(Mov(x0, 0x0000ffff00000000), "mov x0, #0xffff00000000");
  COMPARE_MACRO(Mov(x0, 0x00000000ffff0000), "mov x0, #0xffff0000");
  COMPARE_MACRO(Mov(x0, 0xffffffffffff0000), "mov x0, #0xffffffffffff0000");
  COMPARE_MACRO(Mov(x0, 0xffffffff0000ffff), "mov x0, #0xffffffff0000ffff");
  COMPARE_MACRO(Mov(x0, 0xffff0000ffffffff), "mov x0, #0xffff0000ffffffff");
  COMPARE_MACRO(Mov(x0, 0x0000ffffffffffff), "mov x0, #0xffffffffffff");
  COMPARE_MACRO(Mov(x0, 0xfffe7fffffffffff), "mov x0, #0xfffe7fffffffffff");
  COMPARE_MACRO(Mov(x0, 0xfffeffffffffffff), "mov x0, #0xfffeffffffffffff");
  COMPARE_MACRO(Mov(x0, 0xffff7fffffffffff), "mov x0, #0xffff7fffffffffff");
  COMPARE_MACRO(Mov(x0, 0xfffffffe7fffffff), "mov x0, #0xfffffffe7fffffff");
  COMPARE_MACRO(Mov(x0, 0xfffffffeffffffff), "mov x0, #0xfffffffeffffffff");
  COMPARE_MACRO(Mov(x0, 0xffffffff7fffffff), "mov x0, #0xffffffff7fffffff");
  COMPARE_MACRO(Mov(x0, 0xfffffffffffe7fff), "mov x0, #0xfffffffffffe7fff");
  COMPARE_MACRO(Mov(x0, 0xfffffffffffeffff), "mov x0, #0xfffffffffffeffff");
  COMPARE_MACRO(Mov(x0, 0xffffffffffff7fff), "mov x0, #0xffffffffffff7fff");
  COMPARE_MACRO(Mov(x0, 0xffffffffffffffff), "mov x0, #0xffffffffffffffff");

  COMPARE_MACRO(Movk(w0, 0x1234, 0), "movk w0, #0x1234");
  COMPARE_MACRO(Movk(x1, 0x2345, 0), "movk x1, #0x2345");
  COMPARE_MACRO(Movk(w2, 0x3456, 16), "movk w2, #0x3456, lsl #16");
  COMPARE_MACRO(Movk(x3, 0x4567, 16), "movk x3, #0x4567, lsl #16");
  COMPARE_MACRO(Movk(x4, 0x5678, 32), "movk x4, #0x5678, lsl #32");
  COMPARE_MACRO(Movk(x5, 0x6789, 48), "movk x5, #0x6789, lsl #48");

  CLEANUP();
}


TEST(add_immediate) {
  SETUP();

  COMPARE(add(w0, w1, Operand(0xff)), "add w0, w1, #0xff (255)");
  COMPARE(add(x2, x3, Operand(0x3ff)), "add x2, x3, #0x3ff (1023)");
  COMPARE(add(w4, w5, Operand(0xfff)), "add w4, w5, #0xfff (4095)");
  COMPARE(add(x6, x7, Operand(0x1000)), "add x6, x7, #0x1000 (4096)");
  COMPARE(add(w8, w9, Operand(0xff000)), "add w8, w9, #0xff000 (1044480)");
  COMPARE(add(x10, x11, Operand(0x3ff000)),
          "add x10, x11, #0x3ff000 (4190208)");
  COMPARE(add(w12, w13, Operand(0xfff000)),
          "add w12, w13, #0xfff000 (16773120)");
  COMPARE(adds(w14, w15, Operand(0xff)), "adds w14, w15, #0xff (255)");
  COMPARE(adds(x16, x17, Operand(0xaa000)), "adds x16, x17, #0xaa000 (696320)");

  COMPARE(cmn(w18, Operand(0xff)), "cmn w18, #0xff (255)");
  COMPARE(cmn(x19, Operand(0xff000)), "cmn x19, #0xff000 (1044480)");
  COMPARE(add(w0, wsp, Operand(0)), "mov w0, wsp");
  COMPARE(add(sp, x0, Operand(0)), "mov sp, x0");

  COMPARE(add(w1, wsp, Operand(8)), "add w1, wsp, #0x8 (8)");
  COMPARE(add(x2, sp, Operand(16)), "add x2, sp, #0x10 (16)");
  COMPARE(add(wsp, wsp, Operand(42)), "add wsp, wsp, #0x2a (42)");
  COMPARE(cmn(sp, Operand(24)), "cmn sp, #0x18 (24)");
  COMPARE(adds(wzr, wsp, Operand(9)), "cmn wsp, #0x9 (9)");

  // Instructions in the add/sub immediate space, but unallocated due to shift
  // value out of range.
  COMPARE(dci(0x11800400), "unallocated (Unallocated)");
  COMPARE(dci(0x11c00400), "unallocated (Unallocated)");

  CLEANUP();
}


TEST(sub_immediate) {
  SETUP();

  COMPARE(sub(w0, w1, Operand(0xff)), "sub w0, w1, #0xff (255)");
  COMPARE(sub(x2, x3, Operand(0x3ff)), "sub x2, x3, #0x3ff (1023)");
  COMPARE(sub(w4, w5, Operand(0xfff)), "sub w4, w5, #0xfff (4095)");
  COMPARE(sub(x6, x7, Operand(0x1000)), "sub x6, x7, #0x1000 (4096)");
  COMPARE(sub(w8, w9, Operand(0xff000)), "sub w8, w9, #0xff000 (1044480)");
  COMPARE(sub(x10, x11, Operand(0x3ff000)),
          "sub x10, x11, #0x3ff000 (4190208)");
  COMPARE(sub(w12, w13, Operand(0xfff000)),
          "sub w12, w13, #0xfff000 (16773120)");
  COMPARE(subs(w14, w15, Operand(0xff)), "subs w14, w15, #0xff (255)");
  COMPARE(subs(x16, x17, Operand(0xaa000)), "subs x16, x17, #0xaa000 (696320)");
  COMPARE(cmp(w18, Operand(0xff)), "cmp w18, #0xff (255)");
  COMPARE(cmp(x19, Operand(0xff000)), "cmp x19, #0xff000 (1044480)");

  COMPARE(sub(w1, wsp, Operand(8)), "sub w1, wsp, #0x8 (8)");
  COMPARE(sub(x2, sp, Operand(16)), "sub x2, sp, #0x10 (16)");
  COMPARE(sub(wsp, wsp, Operand(42)), "sub wsp, wsp, #0x2a (42)");
  COMPARE(cmp(sp, Operand(24)), "cmp sp, #0x18 (24)");
  COMPARE(subs(wzr, wsp, Operand(9)), "cmp wsp, #0x9 (9)");

  CLEANUP();
}


TEST(add_shifted) {
  SETUP();

  COMPARE(add(w0, w1, Operand(w2)), "add w0, w1, w2");
  COMPARE(add(x3, x4, Operand(x5)), "add x3, x4, x5");
  COMPARE(add(w6, w7, Operand(w8, LSL, 1)), "add w6, w7, w8, lsl #1");
  COMPARE(add(x9, x10, Operand(x11, LSL, 2)), "add x9, x10, x11, lsl #2");
  COMPARE(add(w12, w13, Operand(w14, LSR, 3)), "add w12, w13, w14, lsr #3");
  COMPARE(add(x15, x16, Operand(x17, LSR, 4)), "add x15, x16, x17, lsr #4");
  COMPARE(add(w18, w19, Operand(w20, ASR, 5)), "add w18, w19, w20, asr #5");
  COMPARE(add(x21, x22, Operand(x23, ASR, 6)), "add x21, x22, x23, asr #6");
  COMPARE(cmn(w24, Operand(w25)), "cmn w24, w25");
  COMPARE(cmn(x26, Operand(x27, LSL, 63)), "cmn x26, x27, lsl #63");

  COMPARE(add(x0, sp, Operand(x1)), "add x0, sp, x1");
  COMPARE(add(w2, wsp, Operand(w3)), "add w2, wsp, w3");
  COMPARE(add(x4, sp, Operand(x5, LSL, 1)), "add x4, sp, x5, lsl #1");
  COMPARE(add(x4, xzr, Operand(x5, LSL, 1)), "add x4, xzr, x5, lsl #1");
  COMPARE(add(w6, wsp, Operand(w7, LSL, 3)), "add w6, wsp, w7, lsl #3");
  COMPARE(adds(xzr, sp, Operand(x8, LSL, 4)), "cmn sp, x8, lsl #4");
  COMPARE(adds(xzr, xzr, Operand(x8, LSL, 5)), "cmn xzr, x8, lsl #5");

  CLEANUP();
}


TEST(sub_shifted) {
  SETUP();

  COMPARE(sub(w0, w1, Operand(w2)), "sub w0, w1, w2");
  COMPARE(sub(x3, x4, Operand(x5)), "sub x3, x4, x5");
  COMPARE(sub(w6, w7, Operand(w8, LSL, 1)), "sub w6, w7, w8, lsl #1");
  COMPARE(sub(x9, x10, Operand(x11, LSL, 2)), "sub x9, x10, x11, lsl #2");
  COMPARE(sub(w12, w13, Operand(w14, LSR, 3)), "sub w12, w13, w14, lsr #3");
  COMPARE(sub(x15, x16, Operand(x17, LSR, 4)), "sub x15, x16, x17, lsr #4");
  COMPARE(sub(w18, w19, Operand(w20, ASR, 5)), "sub w18, w19, w20, asr #5");
  COMPARE(sub(x21, x22, Operand(x23, ASR, 6)), "sub x21, x22, x23, asr #6");
  COMPARE(cmp(w24, Operand(w25)), "cmp w24, w25");
  COMPARE(cmp(x26, Operand(x27, LSL, 63)), "cmp x26, x27, lsl #63");
  COMPARE(neg(w28, Operand(w29)), "neg w28, w29");
  COMPARE(neg(x30, Operand(x0, LSR, 62)), "neg x30, x0, lsr #62");
  COMPARE(negs(w1, Operand(w2)), "negs w1, w2");
  COMPARE(negs(x3, Operand(x4, ASR, 61)), "negs x3, x4, asr #61");

  COMPARE(sub(x0, sp, Operand(x1)), "sub x0, sp, x1");
  COMPARE(sub(w2, wsp, Operand(w3)), "sub w2, wsp, w3");
  COMPARE(sub(x4, sp, Operand(x5, LSL, 1)), "sub x4, sp, x5, lsl #1");
  COMPARE(sub(x4, xzr, Operand(x5, LSL, 1)), "neg x4, x5, lsl #1");
  COMPARE(sub(w6, wsp, Operand(w7, LSL, 3)), "sub w6, wsp, w7, lsl #3");
  COMPARE(subs(xzr, sp, Operand(x8, LSL, 4)), "cmp sp, x8, lsl #4");
  COMPARE(subs(xzr, xzr, Operand(x8, LSL, 5)), "cmp xzr, x8, lsl #5");

  CLEANUP();
}


TEST(add_extended) {
  SETUP();

  COMPARE(add(w0, w1, Operand(w2, UXTB)), "add w0, w1, w2, uxtb");
  COMPARE(adds(x3, x4, Operand(w5, UXTB, 1)), "adds x3, x4, w5, uxtb #1");
  COMPARE(add(w6, w7, Operand(w8, UXTH, 2)), "add w6, w7, w8, uxth #2");
  COMPARE(adds(x9, x10, Operand(x11, UXTW, 3)), "adds x9, x10, w11, uxtw #3");
  COMPARE(add(x12, x13, Operand(x14, UXTX, 4)), "add x12, x13, x14, uxtx #4");
  COMPARE(adds(w15, w16, Operand(w17, SXTB, 4)), "adds w15, w16, w17, sxtb #4");
  COMPARE(add(x18, x19, Operand(x20, SXTB, 3)), "add x18, x19, w20, sxtb #3");
  COMPARE(adds(w21, w22, Operand(w23, SXTH, 2)), "adds w21, w22, w23, sxth #2");
  COMPARE(add(x24, x25, Operand(x26, SXTW, 1)), "add x24, x25, w26, sxtw #1");
  COMPARE(adds(x27, x28, Operand(x29, SXTX)), "adds x27, x28, x29, sxtx");
  COMPARE(cmn(w0, Operand(w1, UXTB, 2)), "cmn w0, w1, uxtb #2");
  COMPARE(cmn(x2, Operand(x3, SXTH, 4)), "cmn x2, w3, sxth #4");

  COMPARE(add(w0, wsp, Operand(w1, UXTB)), "add w0, wsp, w1, uxtb");
  COMPARE(add(x2, sp, Operand(x3, UXTH, 1)), "add x2, sp, w3, uxth #1");
  COMPARE(add(wsp, wsp, Operand(w4, UXTW, 2)), "add wsp, wsp, w4, lsl #2");
  COMPARE(cmn(sp, Operand(xzr, UXTX, 3)), "cmn sp, xzr, lsl #3");
  COMPARE(cmn(sp, Operand(xzr, LSL, 4)), "cmn sp, xzr, lsl #4");

  CLEANUP();
}


TEST(sub_extended) {
  SETUP();

  COMPARE(sub(w0, w1, Operand(w2, UXTB)), "sub w0, w1, w2, uxtb");
  COMPARE(subs(x3, x4, Operand(w5, UXTB, 1)), "subs x3, x4, w5, uxtb #1");
  COMPARE(sub(w6, w7, Operand(w8, UXTH, 2)), "sub w6, w7, w8, uxth #2");
  COMPARE(subs(x9, x10, Operand(x11, UXTW, 3)), "subs x9, x10, w11, uxtw #3");
  COMPARE(sub(x12, x13, Operand(x14, UXTX, 4)), "sub x12, x13, x14, uxtx #4");
  COMPARE(subs(w15, w16, Operand(w17, SXTB, 4)), "subs w15, w16, w17, sxtb #4");
  COMPARE(sub(x18, x19, Operand(x20, SXTB, 3)), "sub x18, x19, w20, sxtb #3");
  COMPARE(subs(w21, w22, Operand(w23, SXTH, 2)), "subs w21, w22, w23, sxth #2");
  COMPARE(sub(x24, x25, Operand(x26, SXTW, 1)), "sub x24, x25, w26, sxtw #1");
  COMPARE(subs(x27, x28, Operand(x29, SXTX)), "subs x27, x28, x29, sxtx");
  COMPARE(cmp(w0, Operand(w1, SXTB, 1)), "cmp w0, w1, sxtb #1");
  COMPARE(cmp(x2, Operand(x3, UXTH, 3)), "cmp x2, w3, uxth #3");

  COMPARE(sub(w0, wsp, Operand(w1, UXTB)), "sub w0, wsp, w1, uxtb");
  COMPARE(sub(x2, sp, Operand(x3, UXTH, 1)), "sub x2, sp, w3, uxth #1");
  COMPARE(sub(wsp, wsp, Operand(w4, UXTW, 2)), "sub wsp, wsp, w4, lsl #2");
  COMPARE(cmp(sp, Operand(xzr, UXTX, 3)), "cmp sp, xzr, lsl #3");
  COMPARE(cmp(sp, Operand(xzr, LSL, 4)), "cmp sp, xzr, lsl #4");

  CLEANUP();
}


TEST(adc_subc_ngc) {
  SETUP();

  COMPARE(adc(w0, w1, Operand(w2)), "adc w0, w1, w2");
  COMPARE(adc(x3, x4, Operand(x5)), "adc x3, x4, x5");
  COMPARE(adcs(w6, w7, Operand(w8)), "adcs w6, w7, w8");
  COMPARE(adcs(x9, x10, Operand(x11)), "adcs x9, x10, x11");
  COMPARE(sbc(w12, w13, Operand(w14)), "sbc w12, w13, w14");
  COMPARE(sbc(x15, x16, Operand(x17)), "sbc x15, x16, x17");
  COMPARE(sbcs(w18, w19, Operand(w20)), "sbcs w18, w19, w20");
  COMPARE(sbcs(x21, x22, Operand(x23)), "sbcs x21, x22, x23");
  COMPARE(ngc(w24, Operand(w25)), "ngc w24, w25");
  COMPARE(ngc(x26, Operand(x27)), "ngc x26, x27");
  COMPARE(ngcs(w28, Operand(w29)), "ngcs w28, w29");
  COMPARE(ngcs(x30, Operand(x0)), "ngcs x30, x0");

  CLEANUP();
}


TEST(rmif) {
  SETUP();

  COMPARE(rmif(x0, 3, ZCVFlag), "rmif x0, #3, #nZCV");
  COMPARE(rmif(x1, 63, NCFlag), "rmif x1, #63, #NzCv");

  CLEANUP();
}


TEST(setf8_setf16) {
  SETUP();

  COMPARE(setf8(w0), "setf8 w0");
  COMPARE(setf16(w1), "setf16 w1");

  CLEANUP();
}


TEST(mul_and_div) {
  SETUP();

  COMPARE(mul(w0, w1, w2), "mul w0, w1, w2");
  COMPARE(mul(x3, x4, x5), "mul x3, x4, x5");
  COMPARE(mul(w30, w0, w1), "mul w30, w0, w1");
  COMPARE(mul(x30, x0, x1), "mul x30, x0, x1");
  COMPARE(mneg(w0, w1, w2), "mneg w0, w1, w2");
  COMPARE(mneg(x3, x4, x5), "mneg x3, x4, x5");
  COMPARE(mneg(w30, w0, w1), "mneg w30, w0, w1");
  COMPARE(mneg(x30, x0, x1), "mneg x30, x0, x1");
  COMPARE(smull(x0, w0, w1), "smull x0, w0, w1");
  COMPARE(smull(x30, w30, w0), "smull x30, w30, w0");
  COMPARE(smulh(x0, x1, x2), "smulh x0, x1, x2");
  COMPARE(umulh(x0, x2, x1), "umulh x0, x2, x1");

  COMPARE(sdiv(w0, w1, w2), "sdiv w0, w1, w2");
  COMPARE(sdiv(x3, x4, x5), "sdiv x3, x4, x5");
  COMPARE(udiv(w6, w7, w8), "udiv w6, w7, w8");
  COMPARE(udiv(x9, x10, x11), "udiv x9, x10, x11");

  CLEANUP();
}


TEST(madd) {
  SETUP();

  COMPARE(madd(w0, w1, w2, w3), "madd w0, w1, w2, w3");
  COMPARE(madd(w30, w21, w22, w16), "madd w30, w21, w22, w16");
  COMPARE(madd(x0, x1, x2, x3), "madd x0, x1, x2, x3");
  COMPARE(madd(x30, x21, x22, x16), "madd x30, x21, x22, x16");

  COMPARE(smaddl(x0, w1, w2, x3), "smaddl x0, w1, w2, x3");
  COMPARE(smaddl(x30, w21, w22, x16), "smaddl x30, w21, w22, x16");
  COMPARE(umaddl(x0, w1, w2, x3), "umaddl x0, w1, w2, x3");
  COMPARE(umaddl(x30, w21, w22, x16), "umaddl x30, w21, w22, x16");
  COMPARE(umull(x0, w1, w2), "umull x0, w1, w2");
  COMPARE(umull(x30, w21, w22), "umull x30, w21, w22");

  CLEANUP();
}


TEST(msub) {
  SETUP();

  COMPARE(msub(w0, w1, w2, w3), "msub w0, w1, w2, w3");
  COMPARE(msub(w30, w21, w22, w16), "msub w30, w21, w22, w16");
  COMPARE(msub(x0, x1, x2, x3), "msub x0, x1, x2, x3");
  COMPARE(msub(x30, x21, x22, x16), "msub x30, x21, x22, x16");

  COMPARE(smsubl(x0, w1, w2, x3), "smsubl x0, w1, w2, x3");
  COMPARE(smsubl(x30, w21, w22, x16), "smsubl x30, w21, w22, x16");
  COMPARE(umsubl(x0, w1, w2, x3), "umsubl x0, w1, w2, x3");
  COMPARE(umsubl(x30, w21, w22, x16), "umsubl x30, w21, w22, x16");

  CLEANUP();
}


TEST(dp_1_source) {
  SETUP();

  COMPARE(rbit(w0, w1), "rbit w0, w1");
  COMPARE(rbit(x2, x3), "rbit x2, x3");
  COMPARE(rev16(w4, w5), "rev16 w4, w5");
  COMPARE(rev16(x6, x7), "rev16 x6, x7");
  COMPARE(rev32(x8, x9), "rev32 x8, x9");
  COMPARE(rev64(x10, x11), "rev x10, x11");
  COMPARE(rev(w12, w13), "rev w12, w13");
  COMPARE(rev(x14, x15), "rev x14, x15");
  COMPARE(clz(w16, w17), "clz w16, w17");
  COMPARE(clz(x18, x19), "clz x18, x19");
  COMPARE(cls(w20, w21), "cls w20, w21");
  COMPARE(cls(x22, x23), "cls x22, x23");
  COMPARE(pacia(x24, x25), "pacia x24, x25");
  COMPARE(pacia(x26, sp), "pacia x26, sp");
  COMPARE(pacib(x27, x28), "pacib x27, x28");
  COMPARE(pacib(x29, sp), "pacib x29, sp");
  COMPARE(pacda(x30, x0), "pacda x30, x0");
  COMPARE(pacda(x1, sp), "pacda x1, sp");
  COMPARE(pacdb(x2, x3), "pacdb x2, x3");
  COMPARE(pacdb(x4, sp), "pacdb x4, sp");
  COMPARE(paciza(x5), "paciza x5");
  COMPARE(pacizb(x6), "pacizb x6");
  COMPARE(pacdza(x7), "pacdza x7");
  COMPARE(pacdzb(x8), "pacdzb x8");
  COMPARE(autia(x9, x10), "autia x9, x10");
  COMPARE(autia(x11, sp), "autia x11, sp");
  COMPARE(autib(x12, x13), "autib x12, x13");
  COMPARE(autib(x14, sp), "autib x14, sp");
  COMPARE(autda(x15, x16), "autda x15, x16");
  COMPARE(autda(x17, sp), "autda x17, sp");
  COMPARE(autdb(x18, x19), "autdb x18, x19");
  COMPARE(autdb(x20, sp), "autdb x20, sp");
  COMPARE(autiza(x21), "autiza x21");
  COMPARE(autizb(x22), "autizb x22");
  COMPARE(autdza(x23), "autdza x23");

  CLEANUP();
}


TEST(bitfield) {
  SETUP();

  COMPARE(sxtb(w0, w1), "sxtb w0, w1");
  COMPARE(sxtb(x2, x3), "sxtb x2, w3");
  COMPARE(sxth(w4, w5), "sxth w4, w5");
  COMPARE(sxth(x6, x7), "sxth x6, w7");
  COMPARE(sxtw(x8, x9), "sxtw x8, w9");
  COMPARE(sxtb(x0, w1), "sxtb x0, w1");
  COMPARE(sxth(x2, w3), "sxth x2, w3");
  COMPARE(sxtw(x4, w5), "sxtw x4, w5");

  COMPARE(uxtb(w10, w11), "uxtb w10, w11");
  COMPARE(uxtb(x12, x13), "uxtb x12, w13");
  COMPARE(uxth(w14, w15), "uxth w14, w15");
  COMPARE(uxth(x16, x17), "uxth x16, w17");
  COMPARE(uxtw(x18, x19), "ubfx x18, x19, #0, #32");

  COMPARE(asr(w20, w21, 10), "asr w20, w21, #10");
  COMPARE(asr(x22, x23, 20), "asr x22, x23, #20");
  COMPARE(lsr(w24, w25, 10), "lsr w24, w25, #10");
  COMPARE(lsr(x26, x27, 20), "lsr x26, x27, #20");
  COMPARE(lsl(w28, w29, 10), "lsl w28, w29, #10");
  COMPARE(lsl(x30, x0, 20), "lsl x30, x0, #20");

  COMPARE(sbfiz(w1, w2, 1, 20), "sbfiz w1, w2, #1, #20");
  COMPARE(sbfiz(x3, x4, 2, 19), "sbfiz x3, x4, #2, #19");
  COMPARE(sbfx(w5, w6, 3, 18), "sbfx w5, w6, #3, #18");
  COMPARE(sbfx(x7, x8, 4, 17), "sbfx x7, x8, #4, #17");
  COMPARE(bfi(w9, w10, 5, 16), "bfi w9, w10, #5, #16");
  COMPARE(bfi(x11, x12, 6, 15), "bfi x11, x12, #6, #15");
  COMPARE(bfxil(w13, w14, 7, 14), "bfxil w13, w14, #7, #14");
  COMPARE(bfxil(x15, x16, 8, 13), "bfxil x15, x16, #8, #13");
  COMPARE(ubfiz(w17, w18, 9, 12), "ubfiz w17, w18, #9, #12");
  COMPARE(ubfiz(x19, x20, 10, 11), "ubfiz x19, x20, #10, #11");
  COMPARE(ubfx(w21, w22, 11, 10), "ubfx w21, w22, #11, #10");
  COMPARE(ubfx(x23, x24, 12, 9), "ubfx x23, x24, #12, #9");
  COMPARE(bfc(w25, 13, 8), "bfc w25, #13, #8");
  COMPARE(bfc(x26, 14, 7), "bfc x26, #14, #7");

  CLEANUP();
}


TEST(crc32b) {
  SETUP();

  COMPARE(crc32b(w0, w1, w2), "crc32b w0, w1, w2");
  COMPARE(crc32b(w0, w11, w22), "crc32b w0, w11, w22");
  COMPARE(crc32b(w10, w20, w30), "crc32b w10, w20, w30");

  CLEANUP();
}


TEST(crc32h) {
  SETUP();

  COMPARE(crc32h(w1, w2, w3), "crc32h w1, w2, w3");
  COMPARE(crc32h(w2, w13, w23), "crc32h w2, w13, w23");
  COMPARE(crc32h(w11, w12, w15), "crc32h w11, w12, w15");

  CLEANUP();
}


TEST(crc32w) {
  SETUP();

  COMPARE(crc32w(w2, w3, w4), "crc32w w2, w3, w4");
  COMPARE(crc32w(w3, w14, w24), "crc32w w3, w14, w24");
  COMPARE(crc32w(w13, w13, w16), "crc32w w13, w13, w16");

  CLEANUP();
}


TEST(crc32x) {
  SETUP();

  COMPARE(crc32x(w3, w4, x5), "crc32x w3, w4, x5");
  COMPARE(crc32x(w4, w15, x25), "crc32x w4, w15, x25");
  COMPARE(crc32x(w14, w14, x30), "crc32x w14, w14, x30");

  CLEANUP();
}


TEST(crc32cb) {
  SETUP();

  COMPARE(crc32cb(w4, w5, w6), "crc32cb w4, w5, w6");
  COMPARE(crc32cb(w5, w16, w26), "crc32cb w5, w16, w26");
  COMPARE(crc32cb(w15, w15, w5), "crc32cb w15, w15, w5");

  CLEANUP();
}


TEST(crc32ch) {
  SETUP();

  COMPARE(crc32ch(w5, w6, w7), "crc32ch w5, w6, w7");
  COMPARE(crc32ch(w6, w17, w27), "crc32ch w6, w17, w27");
  COMPARE(crc32ch(w16, w16, w2), "crc32ch w16, w16, w2");

  CLEANUP();
}


TEST(crc32cw) {
  SETUP();

  COMPARE(crc32cw(w6, w7, w8), "crc32cw w6, w7, w8");
  COMPARE(crc32cw(w7, w18, w28), "crc32cw w7, w18, w28");
  COMPARE(crc32cw(w17, w17, w3), "crc32cw w17, w17, w3");

  CLEANUP();
}


TEST(crc32cx) {
  SETUP();

  COMPARE(crc32cx(w7, w8, x9), "crc32cx w7, w8, x9");
  COMPARE(crc32cx(w8, w19, x29), "crc32cx w8, w19, x29");
  COMPARE(crc32cx(w18, w18, x4), "crc32cx w18, w18, x4");

  CLEANUP();
}


TEST(extract) {
  SETUP();

  COMPARE(extr(w0, w1, w2, 0), "extr w0, w1, w2, #0");
  COMPARE(extr(x3, x4, x5, 1), "extr x3, x4, x5, #1");
  COMPARE(extr(w6, w7, w8, 31), "extr w6, w7, w8, #31");
  COMPARE(extr(x9, x10, x11, 63), "extr x9, x10, x11, #63");
  COMPARE(extr(w12, w13, w13, 10), "ror w12, w13, #10");
  COMPARE(extr(x14, x15, x15, 42), "ror x14, x15, #42");

  CLEANUP();
}


TEST(logical_immediate) {
  SETUP();
#define RESULT_SIZE (256)

  char result[RESULT_SIZE];

  // Test immediate encoding - 64-bit destination.
  // 64-bit patterns.
  uint64_t value = 0x7fffffff;
  for (int i = 0; i < 64; i++) {
    snprintf(result, RESULT_SIZE, "and x0, x0, #0x%" PRIx64, value);
    COMPARE(and_(x0, x0, Operand(value)), result);
    value = ((value & 1) << 63) | (value >> 1);  // Rotate right 1 bit.
  }

  // 32-bit patterns.
  value = 0x00003fff00003fff;
  for (int i = 0; i < 32; i++) {
    snprintf(result, RESULT_SIZE, "and x0, x0, #0x%" PRIx64, value);
    COMPARE(and_(x0, x0, Operand(value)), result);
    value = ((value & 1) << 63) | (value >> 1);  // Rotate right 1 bit.
  }

  // 16-bit patterns.
  value = 0x001f001f001f001f;
  for (int i = 0; i < 16; i++) {
    snprintf(result, RESULT_SIZE, "and x0, x0, #0x%" PRIx64, value);
    COMPARE(and_(x0, x0, Operand(value)), result);
    value = ((value & 1) << 63) | (value >> 1);  // Rotate right 1 bit.
  }

  // 8-bit patterns.
  value = 0x0e0e0e0e0e0e0e0e;
  for (int i = 0; i < 8; i++) {
    snprintf(result, RESULT_SIZE, "and x0, x0, #0x%" PRIx64, value);
    COMPARE(and_(x0, x0, Operand(value)), result);
    value = ((value & 1) << 63) | (value >> 1);  // Rotate right 1 bit.
  }

  // 4-bit patterns.
  value = 0x6666666666666666;
  for (int i = 0; i < 4; i++) {
    snprintf(result, RESULT_SIZE, "and x0, x0, #0x%" PRIx64, value);
    COMPARE(and_(x0, x0, Operand(value)), result);
    value = ((value & 1) << 63) | (value >> 1);  // Rotate right 1 bit.
  }

  // 2-bit patterns.
  COMPARE(and_(x0, x0, Operand(0x5555555555555555)),
          "and x0, x0, #0x5555555555555555");
  COMPARE(and_(x0, x0, Operand(0xaaaaaaaaaaaaaaaa)),
          "and x0, x0, #0xaaaaaaaaaaaaaaaa");

  // Test immediate encoding - 32-bit destination.
  COMPARE(and_(w0, w0, Operand(0xff8007ff)),
          "and w0, w0, #0xff8007ff");  // 32-bit pattern.
  COMPARE(and_(w0, w0, Operand(0xf87ff87f)),
          "and w0, w0, #0xf87ff87f");  // 16-bit pattern.
  COMPARE(and_(w0, w0, Operand(0x87878787)),
          "and w0, w0, #0x87878787");  // 8-bit pattern.
  COMPARE(and_(w0, w0, Operand(0x66666666)),
          "and w0, w0, #0x66666666");  // 4-bit pattern.
  COMPARE(and_(w0, w0, Operand(0x55555555)),
          "and w0, w0, #0x55555555");  // 2-bit pattern.

  // Test other instructions.
  COMPARE(tst(w1, Operand(0x11111111)), "tst w1, #0x11111111");
  COMPARE(tst(x2, Operand(0x8888888888888888)), "tst x2, #0x8888888888888888");
  COMPARE(orr(w7, w8, Operand(0xaaaaaaaa)), "orr w7, w8, #0xaaaaaaaa");
  COMPARE(orr(x9, x10, Operand(0x5555555555555555)),
          "orr x9, x10, #0x5555555555555555");
  COMPARE(eor(w15, w16, Operand(0x00000001)), "eor w15, w16, #0x1");
  COMPARE(eor(x17, x18, Operand(0x0000000000000003)), "eor x17, x18, #0x3");
  COMPARE(ands(w23, w24, Operand(0x0000000f)), "ands w23, w24, #0xf");
  COMPARE(ands(x25, x26, Operand(0x800000000000000f)),
          "ands x25, x26, #0x800000000000000f");

  // Test inverse.
  COMPARE(bic(w3, w4, Operand(0x20202020)), "and w3, w4, #0xdfdfdfdf");
  COMPARE(bic(x5, x6, Operand(0x4040404040404040)),
          "and x5, x6, #0xbfbfbfbfbfbfbfbf");
  COMPARE(orn(w11, w12, Operand(0x40004000)), "orr w11, w12, #0xbfffbfff");
  COMPARE(orn(x13, x14, Operand(0x8181818181818181)),
          "orr x13, x14, #0x7e7e7e7e7e7e7e7e");
  COMPARE(eon(w19, w20, Operand(0x80000001)), "eor w19, w20, #0x7ffffffe");
  COMPARE(eon(x21, x22, Operand(0xc000000000000003)),
          "eor x21, x22, #0x3ffffffffffffffc");
  COMPARE(bics(w27, w28, Operand(0xfffffff7)), "ands w27, w28, #0x8");
  COMPARE(bics(x29, x0, Operand(0xfffffffeffffffff)),
          "ands x29, x0, #0x100000000");

  // Test stack pointer.
  COMPARE(and_(wsp, wzr, Operand(7)), "and wsp, wzr, #0x7");
  COMPARE(ands(xzr, xzr, Operand(7)), "tst xzr, #0x7");
  COMPARE(orr(sp, xzr, Operand(15)), "orr sp, xzr, #0xf");
  COMPARE(eor(wsp, w0, Operand(31)), "eor wsp, w0, #0x1f");

  // Test move aliases.
  COMPARE(orr(w0, wzr, Operand(0x00000780)), "orr w0, wzr, #0x780");
  COMPARE(orr(w1, wzr, Operand(0x00007800)), "orr w1, wzr, #0x7800");
  COMPARE(orr(w2, wzr, Operand(0x00078000)), "mov w2, #0x78000");
  COMPARE(orr(w3, wzr, Operand(0x00780000)), "orr w3, wzr, #0x780000");
  COMPARE(orr(w4, wzr, Operand(0x07800000)), "orr w4, wzr, #0x7800000");
  COMPARE(orr(x5, xzr, Operand(0xffffffffffffc001)),
          "orr x5, xzr, #0xffffffffffffc001");
  COMPARE(orr(x6, xzr, Operand(0xfffffffffffc001f)),
          "mov x6, #0xfffffffffffc001f");
  COMPARE(orr(x7, xzr, Operand(0xffffffffffc001ff)),
          "mov x7, #0xffffffffffc001ff");
  COMPARE(orr(x8, xzr, Operand(0xfffffffffc001fff)),
          "mov x8, #0xfffffffffc001fff");
  COMPARE(orr(x9, xzr, Operand(0xffffffffc001ffff)),
          "orr x9, xzr, #0xffffffffc001ffff");

  CLEANUP();
}


TEST(logical_shifted) {
  SETUP();

  COMPARE(and_(w0, w1, Operand(w2)), "and w0, w1, w2");
  COMPARE(and_(x3, x4, Operand(x5, LSL, 1)), "and x3, x4, x5, lsl #1");
  COMPARE(and_(w6, w7, Operand(w8, LSR, 2)), "and w6, w7, w8, lsr #2");
  COMPARE(and_(x9, x10, Operand(x11, ASR, 3)), "and x9, x10, x11, asr #3");
  COMPARE(and_(w12, w13, Operand(w14, ROR, 4)), "and w12, w13, w14, ror #4");

  COMPARE(bic(w15, w16, Operand(w17)), "bic w15, w16, w17");
  COMPARE(bic(x18, x19, Operand(x20, LSL, 5)), "bic x18, x19, x20, lsl #5");
  COMPARE(bic(w21, w22, Operand(w23, LSR, 6)), "bic w21, w22, w23, lsr #6");
  COMPARE(bic(x24, x25, Operand(x26, ASR, 7)), "bic x24, x25, x26, asr #7");
  COMPARE(bic(w27, w28, Operand(w29, ROR, 8)), "bic w27, w28, w29, ror #8");

  COMPARE(orr(w0, w1, Operand(w2)), "orr w0, w1, w2");
  COMPARE(orr(x3, x4, Operand(x5, LSL, 9)), "orr x3, x4, x5, lsl #9");
  COMPARE(orr(w6, w7, Operand(w8, LSR, 10)), "orr w6, w7, w8, lsr #10");
  COMPARE(orr(x9, x10, Operand(x11, ASR, 11)), "orr x9, x10, x11, asr #11");
  COMPARE(orr(w12, w13, Operand(w14, ROR, 12)), "orr w12, w13, w14, ror #12");

  COMPARE(orn(w15, w16, Operand(w17)), "orn w15, w16, w17");
  COMPARE(orn(x18, x19, Operand(x20, LSL, 13)), "orn x18, x19, x20, lsl #13");
  COMPARE(orn(w21, w22, Operand(w23, LSR, 14)), "orn w21, w22, w23, lsr #14");
  COMPARE(orn(x24, x25, Operand(x26, ASR, 15)), "orn x24, x25, x26, asr #15");
  COMPARE(orn(w27, w28, Operand(w29, ROR, 16)), "orn w27, w28, w29, ror #16");

  COMPARE(eor(w0, w1, Operand(w2)), "eor w0, w1, w2");
  COMPARE(eor(x3, x4, Operand(x5, LSL, 17)), "eor x3, x4, x5, lsl #17");
  COMPARE(eor(w6, w7, Operand(w8, LSR, 18)), "eor w6, w7, w8, lsr #18");
  COMPARE(eor(x9, x10, Operand(x11, ASR, 19)), "eor x9, x10, x11, asr #19");
  COMPARE(eor(w12, w13, Operand(w14, ROR, 20)), "eor w12, w13, w14, ror #20");

  COMPARE(eon(w15, w16, Operand(w17)), "eon w15, w16, w17");
  COMPARE(eon(x18, x19, Operand(x20, LSL, 21)), "eon x18, x19, x20, lsl #21");
  COMPARE(eon(w21, w22, Operand(w23, LSR, 22)), "eon w21, w22, w23, lsr #22");
  COMPARE(eon(x24, x25, Operand(x26, ASR, 23)), "eon x24, x25, x26, asr #23");
  COMPARE(eon(w27, w28, Operand(w29, ROR, 24)), "eon w27, w28, w29, ror #24");

  COMPARE(ands(w0, w1, Operand(w2)), "ands w0, w1, w2");
  COMPARE(ands(x3, x4, Operand(x5, LSL, 1)), "ands x3, x4, x5, lsl #1");
  COMPARE(ands(w6, w7, Operand(w8, LSR, 2)), "ands w6, w7, w8, lsr #2");
  COMPARE(ands(x9, x10, Operand(x11, ASR, 3)), "ands x9, x10, x11, asr #3");
  COMPARE(ands(w12, w13, Operand(w14, ROR, 4)), "ands w12, w13, w14, ror #4");

  COMPARE(bics(w15, w16, Operand(w17)), "bics w15, w16, w17");
  COMPARE(bics(x18, x19, Operand(x20, LSL, 5)), "bics x18, x19, x20, lsl #5");
  COMPARE(bics(w21, w22, Operand(w23, LSR, 6)), "bics w21, w22, w23, lsr #6");
  COMPARE(bics(x24, x25, Operand(x26, ASR, 7)), "bics x24, x25, x26, asr #7");
  COMPARE(bics(w27, w28, Operand(w29, ROR, 8)), "bics w27, w28, w29, ror #8");

  COMPARE(tst(w0, Operand(w1)), "tst w0, w1");
  COMPARE(tst(w2, Operand(w3, ROR, 10)), "tst w2, w3, ror #10");
  COMPARE(tst(x0, Operand(x1)), "tst x0, x1");
  COMPARE(tst(x2, Operand(x3, ROR, 42)), "tst x2, x3, ror #42");

  COMPARE(orn(w0, wzr, Operand(w1)), "mvn w0, w1");
  COMPARE(orn(w2, wzr, Operand(w3, ASR, 5)), "mvn w2, w3, asr #5");
  COMPARE(orn(x0, xzr, Operand(x1)), "mvn x0, x1");
  COMPARE(orn(x2, xzr, Operand(x3, ASR, 42)), "mvn x2, x3, asr #42");

  COMPARE(orr(w0, wzr, Operand(w1)), "mov w0, w1");
  COMPARE(orr(x0, xzr, Operand(x1)), "mov x0, x1");
  COMPARE(orr(w16, wzr, Operand(w17, LSL, 1)), "orr w16, wzr, w17, lsl #1");
  COMPARE(orr(x16, xzr, Operand(x17, ASR, 2)), "orr x16, xzr, x17, asr #2");

  CLEANUP();
}


TEST(dp_2_source) {
  SETUP();

  COMPARE(lslv(w0, w1, w2), "lsl w0, w1, w2");
  COMPARE(lslv(x3, x4, x5), "lsl x3, x4, x5");
  COMPARE(lsrv(w6, w7, w8), "lsr w6, w7, w8");
  COMPARE(lsrv(x9, x10, x11), "lsr x9, x10, x11");
  COMPARE(asrv(w12, w13, w14), "asr w12, w13, w14");
  COMPARE(asrv(x15, x16, x17), "asr x15, x16, x17");
  COMPARE(rorv(w18, w19, w20), "ror w18, w19, w20");
  COMPARE(rorv(x21, x22, x23), "ror x21, x22, x23");
  COMPARE(pacga(x24, x25, x26), "pacga x24, x25, x26");
  COMPARE(pacga(x27, x28, sp), "pacga x27, x28, sp");

  CLEANUP();
}


TEST(adr) {
  SETUP();

  COMPARE_PREFIX(adr(x0, INT64_C(0)), "adr x0, #+0x0");
  COMPARE_PREFIX(adr(x1, 1), "adr x1, #+0x1");
  COMPARE_PREFIX(adr(x2, -1), "adr x2, #-0x1");
  COMPARE_PREFIX(adr(x3, 4), "adr x3, #+0x4");
  COMPARE_PREFIX(adr(x4, -4), "adr x4, #-0x4");
  COMPARE_PREFIX(adr(x5, 0x000fffff), "adr x5, #+0xfffff");
  COMPARE_PREFIX(adr(x6, -0x00100000), "adr x6, #-0x100000");
  COMPARE_PREFIX(adr(xzr, INT64_C(0)), "adr xzr, #+0x0");

  CLEANUP();
}


TEST(adrp) {
  SETUP();

  COMPARE_PREFIX(adrp(x0, INT64_C(0)), "adrp x0, #+0x0");
  COMPARE_PREFIX(adrp(x1, 1), "adrp x1, #+0x1000");
  COMPARE_PREFIX(adrp(x2, -1), "adrp x2, #-0x1000");
  COMPARE_PREFIX(adrp(x3, 4), "adrp x3, #+0x4000");
  COMPARE_PREFIX(adrp(x4, -4), "adrp x4, #-0x4000");
  COMPARE_PREFIX(adrp(x5, 0x000fffff), "adrp x5, #+0xfffff000");
  COMPARE_PREFIX(adrp(x6, -0x00100000), "adrp x6, #-0x100000000");
  COMPARE_PREFIX(adrp(xzr, INT64_C(0)), "adrp xzr, #+0x0");

  CLEANUP();
}


TEST(branch) {
  SETUP();

#define INST_OFF(x) (INT64_C(x) >> kInstructionSizeLog2)
  COMPARE_PREFIX(b(INST_OFF(0x4)), "b #+0x4");
  COMPARE_PREFIX(b(INST_OFF(-0x4)), "b #-0x4");
  COMPARE_PREFIX(b(INST_OFF(0x7fffffc)), "b #+0x7fffffc");
  COMPARE_PREFIX(b(INST_OFF(-0x8000000)), "b #-0x8000000");
  COMPARE_PREFIX(b(INST_OFF(0xffffc), eq), "b.eq #+0xffffc");
  COMPARE_PREFIX(b(INST_OFF(-0x100000), mi), "b.mi #-0x100000");
  COMPARE_PREFIX(b(INST_OFF(0xffffc), al), "b.al #+0xffffc");
  COMPARE_PREFIX(b(INST_OFF(-0x100000), nv), "b.nv #-0x100000");
  COMPARE_PREFIX(bl(INST_OFF(0x4)), "bl #+0x4");
  COMPARE_PREFIX(bl(INST_OFF(-0x4)), "bl #-0x4");
  COMPARE_PREFIX(bl(INST_OFF(0xffffc)), "bl #+0xffffc");
  COMPARE_PREFIX(bl(INST_OFF(-0x100000)), "bl #-0x100000");
  COMPARE_PREFIX(cbz(w0, INST_OFF(0xffffc)), "cbz w0, #+0xffffc");
  COMPARE_PREFIX(cbz(x1, INST_OFF(-0x100000)), "cbz x1, #-0x100000");
  COMPARE_PREFIX(cbnz(w2, INST_OFF(0xffffc)), "cbnz w2, #+0xffffc");
  COMPARE_PREFIX(cbnz(x3, INST_OFF(-0x100000)), "cbnz x3, #-0x100000");
  COMPARE_PREFIX(tbz(w4, 0, INST_OFF(0x7ffc)), "tbz w4, #0, #+0x7ffc");
  COMPARE_PREFIX(tbz(x5, 63, INST_OFF(-0x8000)), "tbz x5, #63, #-0x8000");
  COMPARE_PREFIX(tbz(w6, 31, INST_OFF(0)), "tbz w6, #31, #+0x0");
  COMPARE_PREFIX(tbz(x7, 31, INST_OFF(0x4)), "tbz w7, #31, #+0x4");
  COMPARE_PREFIX(tbz(x8, 32, INST_OFF(0x8)), "tbz x8, #32, #+0x8");
  COMPARE_PREFIX(tbnz(w8, 0, INST_OFF(0x7ffc)), "tbnz w8, #0, #+0x7ffc");
  COMPARE_PREFIX(tbnz(x9, 63, INST_OFF(-0x8000)), "tbnz x9, #63, #-0x8000");
  COMPARE_PREFIX(tbnz(w10, 31, INST_OFF(0)), "tbnz w10, #31, #+0x0");
  COMPARE_PREFIX(tbnz(x11, 31, INST_OFF(0x4)), "tbnz w11, #31, #+0x4");
  COMPARE_PREFIX(tbnz(x12, 32, INST_OFF(0x8)), "tbnz x12, #32, #+0x8");
  COMPARE(br(x0), "br x0");
  COMPARE(blr(x1), "blr x1");
  COMPARE(ret(x2), "ret x2");
  COMPARE(ret(lr), "ret");

  COMPARE(braaz(x0), "braaz x0");
  COMPARE(brabz(x1), "brabz x1");
  COMPARE(blraaz(x2), "blraaz x2");
  COMPARE(blrabz(x3), "blrabz x3");
  COMPARE(retaa(), "retaa");
  COMPARE(retab(), "retab");
  COMPARE(braa(x4, x5), "braa x4, x5");
  COMPARE(braa(x6, sp), "braa x6, sp");
  COMPARE(brab(x7, x8), "brab x7, x8");
  COMPARE(brab(x9, sp), "brab x9, sp");
  COMPARE(blraa(x10, x11), "blraa x10, x11");
  COMPARE(blraa(x12, sp), "blraa x12, sp");
  COMPARE(blrab(x13, x14), "blrab x13, x14");
  COMPARE(blrab(x15, sp), "blrab x15, sp");

  CLEANUP();
}


TEST(load_store) {
  SETUP();

  COMPARE(ldr(w0, MemOperand(x1)), "ldr w0, [x1]");
  COMPARE(ldr(w2, MemOperand(x3, 4)), "ldr w2, [x3, #4]");
  COMPARE(ldr(w4, MemOperand(x5, 16380)), "ldr w4, [x5, #16380]");
  COMPARE(ldr(x6, MemOperand(x7)), "ldr x6, [x7]");
  COMPARE(ldr(x8, MemOperand(x9, 8)), "ldr x8, [x9, #8]");
  COMPARE(ldr(x10, MemOperand(x11, 32760)), "ldr x10, [x11, #32760]");
  COMPARE(str(w12, MemOperand(x13)), "str w12, [x13]");
  COMPARE(str(w14, MemOperand(x15, 4)), "str w14, [x15, #4]");
  COMPARE(str(w16, MemOperand(x17, 16380)), "str w16, [x17, #16380]");
  COMPARE(str(x18, MemOperand(x19)), "str x18, [x19]");
  COMPARE(str(x20, MemOperand(x21, 8)), "str x20, [x21, #8]");
  COMPARE(str(x22, MemOperand(x23, 32760)), "str x22, [x23, #32760]");

  COMPARE(ldr(w0, MemOperand(x1, 4, PreIndex)), "ldr w0, [x1, #4]!");
  COMPARE(ldr(w2, MemOperand(x3, 255, PreIndex)), "ldr w2, [x3, #255]!");
  COMPARE(ldr(w4, MemOperand(x5, -256, PreIndex)), "ldr w4, [x5, #-256]!");
  COMPARE(ldr(x6, MemOperand(x7, 8, PreIndex)), "ldr x6, [x7, #8]!");
  COMPARE(ldr(x8, MemOperand(x9, 255, PreIndex)), "ldr x8, [x9, #255]!");
  COMPARE(ldr(x10, MemOperand(x11, -256, PreIndex)), "ldr x10, [x11, #-256]!");
  COMPARE(str(w12, MemOperand(x13, 4, PreIndex)), "str w12, [x13, #4]!");
  COMPARE(str(w14, MemOperand(x15, 255, PreIndex)), "str w14, [x15, #255]!");
  COMPARE(str(w16, MemOperand(x17, -256, PreIndex)), "str w16, [x17, #-256]!");
  COMPARE(str(x18, MemOperand(x19, 8, PreIndex)), "str x18, [x19, #8]!");
  COMPARE(str(x20, MemOperand(x21, 255, PreIndex)), "str x20, [x21, #255]!");
  COMPARE(str(x22, MemOperand(x23, -256, PreIndex)), "str x22, [x23, #-256]!");
  COMPARE(str(x24, MemOperand(x25, 0, PreIndex)), "str x24, [x25, #0]!");
  COMPARE(str(w26, MemOperand(x27, 0, PreIndex)), "str w26, [x27, #0]!");

  COMPARE(ldr(w0, MemOperand(x1, 4, PostIndex)), "ldr w0, [x1], #4");
  COMPARE(ldr(w2, MemOperand(x3, 255, PostIndex)), "ldr w2, [x3], #255");
  COMPARE(ldr(w4, MemOperand(x5, -256, PostIndex)), "ldr w4, [x5], #-256");
  COMPARE(ldr(x6, MemOperand(x7, 8, PostIndex)), "ldr x6, [x7], #8");
  COMPARE(ldr(x8, MemOperand(x9, 255, PostIndex)), "ldr x8, [x9], #255");
  COMPARE(ldr(x10, MemOperand(x11, -256, PostIndex)), "ldr x10, [x11], #-256");
  COMPARE(str(w12, MemOperand(x13, 4, PostIndex)), "str w12, [x13], #4");
  COMPARE(str(w14, MemOperand(x15, 255, PostIndex)), "str w14, [x15], #255");
  COMPARE(str(w16, MemOperand(x17, -256, PostIndex)), "str w16, [x17], #-256");
  COMPARE(str(x18, MemOperand(x19, 8, PostIndex)), "str x18, [x19], #8");
  COMPARE(str(x20, MemOperand(x21, 255, PostIndex)), "str x20, [x21], #255");
  COMPARE(str(x22, MemOperand(x23, -256, PostIndex)), "str x22, [x23], #-256");
  COMPARE(str(x24, MemOperand(x25, 0, PostIndex)), "str x24, [x25], #0");
  COMPARE(str(w26, MemOperand(x27, 0, PostIndex)), "str w26, [x27], #0");

  COMPARE(ldr(w24, MemOperand(sp)), "ldr w24, [sp]");
  COMPARE(ldr(x25, MemOperand(sp, 8)), "ldr x25, [sp, #8]");
  COMPARE(str(w26, MemOperand(sp, 4, PreIndex)), "str w26, [sp, #4]!");
  COMPARE(str(x27, MemOperand(sp, -8, PostIndex)), "str x27, [sp], #-8");

  COMPARE(ldrsw(x0, MemOperand(x1)), "ldrsw x0, [x1]");
  COMPARE(ldrsw(x2, MemOperand(x3, 8)), "ldrsw x2, [x3, #8]");
  COMPARE(ldrsw(x4, MemOperand(x5, 42, PreIndex)), "ldrsw x4, [x5, #42]!");
  COMPARE(ldrsw(x6, MemOperand(x7, -11, PostIndex)), "ldrsw x6, [x7], #-11");

  CLEANUP();
}


TEST(load_store_regoffset) {
  SETUP();

  COMPARE(ldr(w0, MemOperand(x1, w2, UXTW)), "ldr w0, [x1, w2, uxtw]");
  COMPARE(ldr(w3, MemOperand(x4, w5, UXTW, 2)), "ldr w3, [x4, w5, uxtw #2]");
  COMPARE(ldr(w6, MemOperand(x7, x8)), "ldr w6, [x7, x8]");
  COMPARE(ldr(w9, MemOperand(x10, x11, LSL, 2)), "ldr w9, [x10, x11, lsl #2]");
  COMPARE(ldr(w12, MemOperand(x13, w14, SXTW)), "ldr w12, [x13, w14, sxtw]");
  COMPARE(ldr(w15, MemOperand(x16, w17, SXTW, 2)),
          "ldr w15, [x16, w17, sxtw #2]");
  COMPARE(ldr(w18, MemOperand(x19, x20, SXTX)), "ldr w18, [x19, x20, sxtx]");
  COMPARE(ldr(w21, MemOperand(x22, x23, SXTX, 2)),
          "ldr w21, [x22, x23, sxtx #2]");
  COMPARE(ldr(x0, MemOperand(x1, w2, UXTW)), "ldr x0, [x1, w2, uxtw]");
  COMPARE(ldr(x3, MemOperand(x4, w5, UXTW, 3)), "ldr x3, [x4, w5, uxtw #3]");
  COMPARE(ldr(x6, MemOperand(x7, x8)), "ldr x6, [x7, x8]");
  COMPARE(ldr(x9, MemOperand(x10, x11, LSL, 3)), "ldr x9, [x10, x11, lsl #3]");
  COMPARE(ldr(x12, MemOperand(x13, w14, SXTW)), "ldr x12, [x13, w14, sxtw]");
  COMPARE(ldr(x15, MemOperand(x16, w17, SXTW, 3)),
          "ldr x15, [x16, w17, sxtw #3]");
  COMPARE(ldr(x18, MemOperand(x19, x20, SXTX)), "ldr x18, [x19, x20, sxtx]");
  COMPARE(ldr(x21, MemOperand(x22, x23, SXTX, 3)),
          "ldr x21, [x22, x23, sxtx #3]");

  COMPARE(str(w0, MemOperand(x1, w2, UXTW)), "str w0, [x1, w2, uxtw]");
  COMPARE(str(w3, MemOperand(x4, w5, UXTW, 2)), "str w3, [x4, w5, uxtw #2]");
  COMPARE(str(w6, MemOperand(x7, x8)), "str w6, [x7, x8]");
  COMPARE(str(w9, MemOperand(x10, x11, LSL, 2)), "str w9, [x10, x11, lsl #2]");
  COMPARE(str(w12, MemOperand(x13, w14, SXTW)), "str w12, [x13, w14, sxtw]");
  COMPARE(str(w15, MemOperand(x16, w17, SXTW, 2)),
          "str w15, [x16, w17, sxtw #2]");
  COMPARE(str(w18, MemOperand(x19, x20, SXTX)), "str w18, [x19, x20, sxtx]");
  COMPARE(str(w21, MemOperand(x22, x23, SXTX, 2)),
          "str w21, [x22, x23, sxtx #2]");
  COMPARE(str(x0, MemOperand(x1, w2, UXTW)), "str x0, [x1, w2, uxtw]");
  COMPARE(str(x3, MemOperand(x4, w5, UXTW, 3)), "str x3, [x4, w5, uxtw #3]");
  COMPARE(str(x6, MemOperand(x7, x8)), "str x6, [x7, x8]");
  COMPARE(str(x9, MemOperand(x10, x11, LSL, 3)), "str x9, [x10, x11, lsl #3]");
  COMPARE(str(x12, MemOperand(x13, w14, SXTW)), "str x12, [x13, w14, sxtw]");
  COMPARE(str(x15, MemOperand(x16, w17, SXTW, 3)),
          "str x15, [x16, w17, sxtw #3]");
  COMPARE(str(x18, MemOperand(x19, x20, SXTX)), "str x18, [x19, x20, sxtx]");
  COMPARE(str(x21, MemOperand(x22, x23, SXTX, 3)),
          "str x21, [x22, x23, sxtx #3]");

  COMPARE(ldrb(w0, MemOperand(x1, w2, UXTW)), "ldrb w0, [x1, w2, uxtw]");
  COMPARE(ldrb(w6, MemOperand(x7, x8)), "ldrb w6, [x7, x8]");
  COMPARE(ldrb(w12, MemOperand(x13, w14, SXTW)), "ldrb w12, [x13, w14, sxtw]");
  COMPARE(ldrb(w18, MemOperand(x19, x20, SXTX)), "ldrb w18, [x19, x20, sxtx]");
  COMPARE(strb(w0, MemOperand(x1, w2, UXTW)), "strb w0, [x1, w2, uxtw]");
  COMPARE(strb(w6, MemOperand(x7, x8)), "strb w6, [x7, x8]");
  COMPARE(strb(w12, MemOperand(x13, w14, SXTW)), "strb w12, [x13, w14, sxtw]");
  COMPARE(strb(w18, MemOperand(x19, x20, SXTX)), "strb w18, [x19, x20, sxtx]");

  COMPARE(ldrh(w0, MemOperand(x1, w2, UXTW)), "ldrh w0, [x1, w2, uxtw]");
  COMPARE(ldrh(w3, MemOperand(x4, w5, UXTW, 1)), "ldrh w3, [x4, w5, uxtw #1]");
  COMPARE(ldrh(w6, MemOperand(x7, x8)), "ldrh w6, [x7, x8]");
  COMPARE(ldrh(w9, MemOperand(x10, x11, LSL, 1)),
          "ldrh w9, [x10, x11, lsl #1]");
  COMPARE(ldrh(w12, MemOperand(x13, w14, SXTW)), "ldrh w12, [x13, w14, sxtw]");
  COMPARE(ldrh(w15, MemOperand(x16, w17, SXTW, 1)),
          "ldrh w15, [x16, w17, sxtw #1]");
  COMPARE(ldrh(w18, MemOperand(x19, x20, SXTX)), "ldrh w18, [x19, x20, sxtx]");
  COMPARE(ldrh(w21, MemOperand(x22, x23, SXTX, 1)),
          "ldrh w21, [x22, x23, sxtx #1]");
  COMPARE(strh(w0, MemOperand(x1, w2, UXTW)), "strh w0, [x1, w2, uxtw]");
  COMPARE(strh(w3, MemOperand(x4, w5, UXTW, 1)), "strh w3, [x4, w5, uxtw #1]");
  COMPARE(strh(w6, MemOperand(x7, x8)), "strh w6, [x7, x8]");
  COMPARE(strh(w9, MemOperand(x10, x11, LSL, 1)),
          "strh w9, [x10, x11, lsl #1]");
  COMPARE(strh(w12, MemOperand(x13, w14, SXTW)), "strh w12, [x13, w14, sxtw]");
  COMPARE(strh(w15, MemOperand(x16, w17, SXTW, 1)),
          "strh w15, [x16, w17, sxtw #1]");
  COMPARE(strh(w18, MemOperand(x19, x20, SXTX)), "strh w18, [x19, x20, sxtx]");
  COMPARE(strh(w21, MemOperand(x22, x23, SXTX, 1)),
          "strh w21, [x22, x23, sxtx #1]");

  COMPARE(ldr(x0, MemOperand(sp, wzr, SXTW)), "ldr x0, [sp, wzr, sxtw]");
  COMPARE(str(x1, MemOperand(sp, xzr)), "str x1, [sp, xzr]");

  CLEANUP();
}


TEST(load_store_byte) {
  SETUP();

  COMPARE(ldrb(w0, MemOperand(x1)), "ldrb w0, [x1]");
  COMPARE(ldrb(x2, MemOperand(x3)), "ldrb w2, [x3]");
  COMPARE(ldrb(w4, MemOperand(x5, 4095)), "ldrb w4, [x5, #4095]");
  COMPARE(ldrb(w6, MemOperand(x7, 255, PreIndex)), "ldrb w6, [x7, #255]!");
  COMPARE(ldrb(w8, MemOperand(x9, -256, PreIndex)), "ldrb w8, [x9, #-256]!");
  COMPARE(ldrb(w10, MemOperand(x11, 255, PostIndex)), "ldrb w10, [x11], #255");
  COMPARE(ldrb(w12, MemOperand(x13, -256, PostIndex)),
          "ldrb w12, [x13], #-256");
  COMPARE(ldrb(w14, MemOperand(x15, 0, PreIndex)), "ldrb w14, [x15, #0]!");
  COMPARE(ldrb(w16, MemOperand(x17, 0, PostIndex)), "ldrb w16, [x17], #0");
  COMPARE(strb(w14, MemOperand(x15)), "strb w14, [x15]");
  COMPARE(strb(x16, MemOperand(x17)), "strb w16, [x17]");
  COMPARE(strb(w18, MemOperand(x19, 4095)), "strb w18, [x19, #4095]");
  COMPARE(strb(w20, MemOperand(x21, 255, PreIndex)), "strb w20, [x21, #255]!");
  COMPARE(strb(w22, MemOperand(x23, -256, PreIndex)),
          "strb w22, [x23, #-256]!");
  COMPARE(strb(w24, MemOperand(x25, 255, PostIndex)), "strb w24, [x25], #255");
  COMPARE(strb(w26, MemOperand(x27, -256, PostIndex)),
          "strb w26, [x27], #-256");
  COMPARE(strb(w27, MemOperand(x28, 0, PreIndex)), "strb w27, [x28, #0]!");
  COMPARE(strb(w29, MemOperand(x30, 0, PostIndex)), "strb w29, [x30], #0");
  COMPARE(ldrb(w28, MemOperand(sp, 3, PostIndex)), "ldrb w28, [sp], #3");
  COMPARE(strb(x29, MemOperand(sp, -42, PreIndex)), "strb w29, [sp, #-42]!");
  COMPARE(ldrsb(w0, MemOperand(x1)), "ldrsb w0, [x1]");
  COMPARE(ldrsb(x2, MemOperand(x3, 8)), "ldrsb x2, [x3, #8]");
  COMPARE(ldrsb(w4, MemOperand(x5, 42, PreIndex)), "ldrsb w4, [x5, #42]!");
  COMPARE(ldrsb(x6, MemOperand(x7, -11, PostIndex)), "ldrsb x6, [x7], #-11");
  COMPARE(ldrsb(w8, MemOperand(x9, 0, PreIndex)), "ldrsb w8, [x9, #0]!");
  COMPARE(ldrsb(x10, MemOperand(x11, 0, PostIndex)), "ldrsb x10, [x11], #0");

  CLEANUP();
}


TEST(load_store_half) {
  SETUP();

  COMPARE(ldrh(w0, MemOperand(x1)), "ldrh w0, [x1]");
  COMPARE(ldrh(x2, MemOperand(x3)), "ldrh w2, [x3]");
  COMPARE(ldrh(w4, MemOperand(x5, 8190)), "ldrh w4, [x5, #8190]");
  COMPARE(ldrh(w6, MemOperand(x7, 255, PreIndex)), "ldrh w6, [x7, #255]!");
  COMPARE(ldrh(w8, MemOperand(x9, -256, PreIndex)), "ldrh w8, [x9, #-256]!");
  COMPARE(ldrh(w10, MemOperand(x11, 255, PostIndex)), "ldrh w10, [x11], #255");
  COMPARE(ldrh(w12, MemOperand(x13, -256, PostIndex)),
          "ldrh w12, [x13], #-256");
  COMPARE(ldrh(w14, MemOperand(x15, 0, PreIndex)), "ldrh w14, [x15, #0]!");
  COMPARE(ldrh(w16, MemOperand(x17, 0, PostIndex)), "ldrh w16, [x17], #0");
  COMPARE(strh(w14, MemOperand(x15)), "strh w14, [x15]");
  COMPARE(strh(x16, MemOperand(x17)), "strh w16, [x17]");
  COMPARE(strh(w18, MemOperand(x19, 8190)), "strh w18, [x19, #8190]");
  COMPARE(strh(w20, MemOperand(x21, 255, PreIndex)), "strh w20, [x21, #255]!");
  COMPARE(strh(w22, MemOperand(x23, -256, PreIndex)),
          "strh w22, [x23, #-256]!");
  COMPARE(strh(w24, MemOperand(x25, 255, PostIndex)), "strh w24, [x25], #255");
  COMPARE(strh(w26, MemOperand(x27, -256, PostIndex)),
          "strh w26, [x27], #-256");
  COMPARE(strh(w27, MemOperand(x28, 0, PreIndex)), "strh w27, [x28, #0]!");
  COMPARE(strh(w29, MemOperand(x30, 0, PostIndex)), "strh w29, [x30], #0");
  COMPARE(ldrh(w28, MemOperand(sp, 3, PostIndex)), "ldrh w28, [sp], #3");
  COMPARE(strh(x29, MemOperand(sp, -42, PreIndex)), "strh w29, [sp, #-42]!");
  COMPARE(ldrh(w30, MemOperand(x0, 255)), "ldurh w30, [x0, #255]");
  COMPARE(ldrh(x1, MemOperand(x2, -256)), "ldurh w1, [x2, #-256]");
  COMPARE(strh(w3, MemOperand(x4, 255)), "sturh w3, [x4, #255]");
  COMPARE(strh(x5, MemOperand(x6, -256)), "sturh w5, [x6, #-256]");
  COMPARE(ldrsh(w0, MemOperand(x1)), "ldrsh w0, [x1]");
  COMPARE(ldrsh(w2, MemOperand(x3, 8)), "ldrsh w2, [x3, #8]");
  COMPARE(ldrsh(w4, MemOperand(x5, 42, PreIndex)), "ldrsh w4, [x5, #42]!");
  COMPARE(ldrsh(x6, MemOperand(x7, -11, PostIndex)), "ldrsh x6, [x7], #-11");
  COMPARE(ldrsh(w8, MemOperand(x9, 0, PreIndex)), "ldrsh w8, [x9, #0]!");
  COMPARE(ldrsh(x10, MemOperand(x11, 0, PostIndex)), "ldrsh x10, [x11], #0");

  CLEANUP();
}

TEST(load_store_unscaled) {
  SETUP();

  // If an unscaled-offset instruction is requested, it is used, even if the
  // offset could be encoded in a scaled-offset instruction.
  COMPARE(ldurb(w0, MemOperand(x1)), "ldurb w0, [x1]");
  COMPARE(ldurb(x2, MemOperand(x3, 1)), "ldurb w2, [x3, #1]");
  COMPARE(ldurb(w4, MemOperand(x5, 255)), "ldurb w4, [x5, #255]");
  COMPARE(sturb(w14, MemOperand(x15)), "sturb w14, [x15]");
  COMPARE(sturb(x16, MemOperand(x17, 1)), "sturb w16, [x17, #1]");
  COMPARE(sturb(w18, MemOperand(x19, 255)), "sturb w18, [x19, #255]");
  COMPARE(ldursb(w0, MemOperand(x1)), "ldursb w0, [x1]");
  COMPARE(ldursb(w2, MemOperand(x3, 1)), "ldursb w2, [x3, #1]");
  COMPARE(ldursb(x2, MemOperand(x3, 255)), "ldursb x2, [x3, #255]");

  COMPARE(ldurh(w0, MemOperand(x1)), "ldurh w0, [x1]");
  COMPARE(ldurh(x2, MemOperand(x3, 2)), "ldurh w2, [x3, #2]");
  COMPARE(ldurh(w4, MemOperand(x5, 254)), "ldurh w4, [x5, #254]");
  COMPARE(sturh(w14, MemOperand(x15)), "sturh w14, [x15]");
  COMPARE(sturh(x16, MemOperand(x17, 2)), "sturh w16, [x17, #2]");
  COMPARE(sturh(w18, MemOperand(x19, 254)), "sturh w18, [x19, #254]");
  COMPARE(ldursh(w0, MemOperand(x1)), "ldursh w0, [x1]");
  COMPARE(ldursh(w2, MemOperand(x3, 2)), "ldursh w2, [x3, #2]");
  COMPARE(ldursh(x4, MemOperand(x5, 254)), "ldursh x4, [x5, #254]");

  COMPARE(ldur(w0, MemOperand(x1)), "ldur w0, [x1]");
  COMPARE(ldur(w2, MemOperand(x3, 4)), "ldur w2, [x3, #4]");
  COMPARE(ldur(w4, MemOperand(x5, 248)), "ldur w4, [x5, #248]");
  COMPARE(stur(w12, MemOperand(x13)), "stur w12, [x13]");
  COMPARE(stur(w14, MemOperand(x15, 4)), "stur w14, [x15, #4]");
  COMPARE(stur(w16, MemOperand(x17, 248)), "stur w16, [x17, #248]");
  COMPARE(ldursw(x0, MemOperand(x1)), "ldursw x0, [x1]");
  COMPARE(ldursw(x2, MemOperand(x3, 4)), "ldursw x2, [x3, #4]");
  COMPARE(ldursw(x4, MemOperand(x5, 248)), "ldursw x4, [x5, #248]");

  COMPARE(ldur(x6, MemOperand(x7)), "ldur x6, [x7]");
  COMPARE(ldur(x8, MemOperand(x9, 8)), "ldur x8, [x9, #8]");
  COMPARE(ldur(x10, MemOperand(x11, 248)), "ldur x10, [x11, #248]");
  COMPARE(stur(x18, MemOperand(x19)), "stur x18, [x19]");
  COMPARE(stur(x20, MemOperand(x21, 8)), "stur x20, [x21, #8]");
  COMPARE(stur(x22, MemOperand(x23, 248)), "stur x22, [x23, #248]");

  COMPARE(ldur(b0, MemOperand(x1)), "ldur b0, [x1]");
  COMPARE(ldur(h2, MemOperand(x3, -1)), "ldur h2, [x3, #-1]");
  COMPARE(ldur(s4, MemOperand(x5, 2)), "ldur s4, [x5, #2]");
  COMPARE(ldur(d6, MemOperand(x7, -3)), "ldur d6, [x7, #-3]");
  COMPARE(ldur(q8, MemOperand(x9, 4)), "ldur q8, [x9, #4]");
  COMPARE(stur(b10, MemOperand(x11)), "stur b10, [x11]");
  COMPARE(stur(h12, MemOperand(x13, -1)), "stur h12, [x13, #-1]");
  COMPARE(stur(s14, MemOperand(x15, 2)), "stur s14, [x15, #2]");
  COMPARE(stur(d16, MemOperand(x17, -3)), "stur d16, [x17, #-3]");
  COMPARE(stur(q18, MemOperand(x19, 4)), "stur q18, [x19, #4]");

  // Normal loads and stores are converted to unscaled loads and stores if the
  // offset requires it.
  COMPARE(ldr(w0, MemOperand(x1, 1)), "ldur w0, [x1, #1]");
  COMPARE(ldr(w2, MemOperand(x3, -1)), "ldur w2, [x3, #-1]");
  COMPARE(ldr(w4, MemOperand(x5, 255)), "ldur w4, [x5, #255]");
  COMPARE(ldr(w6, MemOperand(x7, -256)), "ldur w6, [x7, #-256]");
  COMPARE(ldr(x8, MemOperand(x9, 1)), "ldur x8, [x9, #1]");
  COMPARE(ldr(x10, MemOperand(x11, -1)), "ldur x10, [x11, #-1]");
  COMPARE(ldr(x12, MemOperand(x13, 255)), "ldur x12, [x13, #255]");
  COMPARE(ldr(x14, MemOperand(x15, -256)), "ldur x14, [x15, #-256]");
  COMPARE(str(w16, MemOperand(x17, 1)), "stur w16, [x17, #1]");
  COMPARE(str(w18, MemOperand(x19, -1)), "stur w18, [x19, #-1]");
  COMPARE(str(w20, MemOperand(x21, 255)), "stur w20, [x21, #255]");
  COMPARE(str(w22, MemOperand(x23, -256)), "stur w22, [x23, #-256]");
  COMPARE(str(x24, MemOperand(x25, 1)), "stur x24, [x25, #1]");
  COMPARE(str(x26, MemOperand(x27, -1)), "stur x26, [x27, #-1]");
  COMPARE(str(x28, MemOperand(x29, 255)), "stur x28, [x29, #255]");
  COMPARE(str(x30, MemOperand(x0, -256)), "stur x30, [x0, #-256]");
  COMPARE(ldr(w0, MemOperand(sp, 1)), "ldur w0, [sp, #1]");
  COMPARE(str(x1, MemOperand(sp, -1)), "stur x1, [sp, #-1]");
  COMPARE(ldrb(w2, MemOperand(x3, -2)), "ldurb w2, [x3, #-2]");
  COMPARE(ldrsb(w4, MemOperand(x5, -3)), "ldursb w4, [x5, #-3]");
  COMPARE(ldrsb(x6, MemOperand(x7, -4)), "ldursb x6, [x7, #-4]");
  COMPARE(ldrh(w8, MemOperand(x9, -5)), "ldurh w8, [x9, #-5]");
  COMPARE(ldrsh(w10, MemOperand(x11, -6)), "ldursh w10, [x11, #-6]");
  COMPARE(ldrsh(x12, MemOperand(x13, -7)), "ldursh x12, [x13, #-7]");
  COMPARE(ldrsw(x14, MemOperand(x15, -8)), "ldursw x14, [x15, #-8]");

  CLEANUP();
}


TEST(load_store_unscaled_option) {
  SETUP();

  // Just like load_store_unscaled, but specify the scaling option explicitly.
  LoadStoreScalingOption options[] = {PreferUnscaledOffset,
                                      RequireUnscaledOffset};

  for (size_t i = 0; i < sizeof(options) / sizeof(options[0]); i++) {
    LoadStoreScalingOption option = options[i];

    // If an unscaled-offset instruction is requested, it is used, even if the
    // offset could be encoded in a scaled-offset instruction.
    COMPARE(ldurb(w0, MemOperand(x1), option), "ldurb w0, [x1]");
    COMPARE(ldurb(x2, MemOperand(x3, 1), option), "ldurb w2, [x3, #1]");
    COMPARE(ldurb(w4, MemOperand(x5, 255), option), "ldurb w4, [x5, #255]");
    COMPARE(sturb(w14, MemOperand(x15), option), "sturb w14, [x15]");
    COMPARE(sturb(x16, MemOperand(x17, 1), option), "sturb w16, [x17, #1]");
    COMPARE(sturb(w18, MemOperand(x19, 255), option), "sturb w18, [x19, #255]");
    COMPARE(ldursb(w0, MemOperand(x1), option), "ldursb w0, [x1]");
    COMPARE(ldursb(w2, MemOperand(x3, 1), option), "ldursb w2, [x3, #1]");
    COMPARE(ldursb(x2, MemOperand(x3, 255), option), "ldursb x2, [x3, #255]");

    COMPARE(ldurh(w0, MemOperand(x1), option), "ldurh w0, [x1]");
    COMPARE(ldurh(x2, MemOperand(x3, 2), option), "ldurh w2, [x3, #2]");
    COMPARE(ldurh(w4, MemOperand(x5, 254), option), "ldurh w4, [x5, #254]");
    COMPARE(sturh(w14, MemOperand(x15), option), "sturh w14, [x15]");
    COMPARE(sturh(x16, MemOperand(x17, 2), option), "sturh w16, [x17, #2]");
    COMPARE(sturh(w18, MemOperand(x19, 254), option), "sturh w18, [x19, #254]");
    COMPARE(ldursh(w0, MemOperand(x1), option), "ldursh w0, [x1]");
    COMPARE(ldursh(w2, MemOperand(x3, 2), option), "ldursh w2, [x3, #2]");
    COMPARE(ldursh(x4, MemOperand(x5, 254), option), "ldursh x4, [x5, #254]");

    COMPARE(ldur(w0, MemOperand(x1), option), "ldur w0, [x1]");
    COMPARE(ldur(w2, MemOperand(x3, 4), option), "ldur w2, [x3, #4]");
    COMPARE(ldur(w4, MemOperand(x5, 248), option), "ldur w4, [x5, #248]");
    COMPARE(stur(w12, MemOperand(x13), option), "stur w12, [x13]");
    COMPARE(stur(w14, MemOperand(x15, 4), option), "stur w14, [x15, #4]");
    COMPARE(stur(w16, MemOperand(x17, 248), option), "stur w16, [x17, #248]");
    COMPARE(ldursw(x0, MemOperand(x1), option), "ldursw x0, [x1]");
    COMPARE(ldursw(x2, MemOperand(x3, 4), option), "ldursw x2, [x3, #4]");
    COMPARE(ldursw(x4, MemOperand(x5, 248), option), "ldursw x4, [x5, #248]");

    COMPARE(ldur(x6, MemOperand(x7), option), "ldur x6, [x7]");
    COMPARE(ldur(x8, MemOperand(x9, 8), option), "ldur x8, [x9, #8]");
    COMPARE(ldur(x10, MemOperand(x11, 248), option), "ldur x10, [x11, #248]");
    COMPARE(stur(x18, MemOperand(x19), option), "stur x18, [x19]");
    COMPARE(stur(x20, MemOperand(x21, 8), option), "stur x20, [x21, #8]");
    COMPARE(stur(x22, MemOperand(x23, 248), option), "stur x22, [x23, #248]");

    COMPARE(ldur(b0, MemOperand(x1), option), "ldur b0, [x1]");
    COMPARE(ldur(h2, MemOperand(x3, 2), option), "ldur h2, [x3, #2]");
    COMPARE(ldur(s4, MemOperand(x5, 4), option), "ldur s4, [x5, #4]");
    COMPARE(ldur(d6, MemOperand(x7, 8), option), "ldur d6, [x7, #8]");
    COMPARE(ldur(q8, MemOperand(x9, 16), option), "ldur q8, [x9, #16]");
    COMPARE(stur(b10, MemOperand(x11), option), "stur b10, [x11]");
    COMPARE(stur(h12, MemOperand(x13, 2), option), "stur h12, [x13, #2]");
    COMPARE(stur(s14, MemOperand(x15, 4), option), "stur s14, [x15, #4]");
    COMPARE(stur(d16, MemOperand(x17, 8), option), "stur d16, [x17, #8]");
    COMPARE(stur(q18, MemOperand(x19, 16), option), "stur q18, [x19, #16]");
  }

  // Normal loads and stores are converted to unscaled loads and stores if the
  // offset requires it. PreferScaledOffset is the default for these cases, so
  // the behaviour here is the same when no option is specified.
  LoadStoreScalingOption option = PreferScaledOffset;
  COMPARE(ldr(w0, MemOperand(x1, 1), option), "ldur w0, [x1, #1]");
  COMPARE(ldr(w2, MemOperand(x3, -1), option), "ldur w2, [x3, #-1]");
  COMPARE(ldr(w4, MemOperand(x5, 255), option), "ldur w4, [x5, #255]");
  COMPARE(ldr(w6, MemOperand(x7, -256), option), "ldur w6, [x7, #-256]");
  COMPARE(ldr(x8, MemOperand(x9, 1), option), "ldur x8, [x9, #1]");
  COMPARE(ldr(x10, MemOperand(x11, -1), option), "ldur x10, [x11, #-1]");
  COMPARE(ldr(x12, MemOperand(x13, 255), option), "ldur x12, [x13, #255]");
  COMPARE(ldr(x14, MemOperand(x15, -256), option), "ldur x14, [x15, #-256]");
  COMPARE(str(w16, MemOperand(x17, 1), option), "stur w16, [x17, #1]");
  COMPARE(str(w18, MemOperand(x19, -1), option), "stur w18, [x19, #-1]");
  COMPARE(str(w20, MemOperand(x21, 255), option), "stur w20, [x21, #255]");
  COMPARE(str(w22, MemOperand(x23, -256), option), "stur w22, [x23, #-256]");
  COMPARE(str(x24, MemOperand(x25, 1), option), "stur x24, [x25, #1]");
  COMPARE(str(x26, MemOperand(x27, -1), option), "stur x26, [x27, #-1]");
  COMPARE(str(x28, MemOperand(x29, 255), option), "stur x28, [x29, #255]");
  COMPARE(str(x30, MemOperand(x0, -256), option), "stur x30, [x0, #-256]");
  COMPARE(ldr(w0, MemOperand(sp, 1), option), "ldur w0, [sp, #1]");
  COMPARE(str(x1, MemOperand(sp, -1), option), "stur x1, [sp, #-1]");
  COMPARE(ldrb(w2, MemOperand(x3, -2), option), "ldurb w2, [x3, #-2]");
  COMPARE(ldrsb(w4, MemOperand(x5, -3), option), "ldursb w4, [x5, #-3]");
  COMPARE(ldrsb(x6, MemOperand(x7, -4), option), "ldursb x6, [x7, #-4]");
  COMPARE(ldrh(w8, MemOperand(x9, -5), option), "ldurh w8, [x9, #-5]");
  COMPARE(ldrsh(w10, MemOperand(x11, -6), option), "ldursh w10, [x11, #-6]");
  COMPARE(ldrsh(x12, MemOperand(x13, -7), option), "ldursh x12, [x13, #-7]");
  COMPARE(ldrsw(x14, MemOperand(x15, -8), option), "ldursw x14, [x15, #-8]");
  COMPARE(ldr(b0, MemOperand(x1, 1), option), "ldr b0, [x1, #1]");
  COMPARE(ldr(h2, MemOperand(x3, 1), option), "ldur h2, [x3, #1]");
  COMPARE(ldr(s4, MemOperand(x5, 3), option), "ldur s4, [x5, #3]");
  COMPARE(ldr(d6, MemOperand(x7, 7), option), "ldur d6, [x7, #7]");
  COMPARE(ldr(q8, MemOperand(x9, 15), option), "ldur q8, [x9, #15]");
  COMPARE(str(b10, MemOperand(x11, 1), option), "str b10, [x11, #1]");
  COMPARE(str(h12, MemOperand(x13, 1), option), "stur h12, [x13, #1]");
  COMPARE(str(s14, MemOperand(x15, 3), option), "stur s14, [x15, #3]");
  COMPARE(str(d16, MemOperand(x17, 7), option), "stur d16, [x17, #7]");
  COMPARE(str(q18, MemOperand(x19, 15), option), "stur q18, [x19, #15]");

  CLEANUP();
}


TEST(load_store_pair) {
  SETUP();

  COMPARE(ldp(w0, w1, MemOperand(x2)), "ldp w0, w1, [x2]");
  COMPARE(ldp(x3, x4, MemOperand(x5)), "ldp x3, x4, [x5]");
  COMPARE(ldp(w6, w7, MemOperand(x8, 4)), "ldp w6, w7, [x8, #4]");
  COMPARE(ldp(x9, x10, MemOperand(x11, 8)), "ldp x9, x10, [x11, #8]");
  COMPARE(ldp(w12, w13, MemOperand(x14, 252)), "ldp w12, w13, [x14, #252]");
  COMPARE(ldp(x15, x16, MemOperand(x17, 504)), "ldp x15, x16, [x17, #504]");
  COMPARE(ldp(w18, w19, MemOperand(x20, -256)), "ldp w18, w19, [x20, #-256]");
  COMPARE(ldp(x21, x22, MemOperand(x23, -512)), "ldp x21, x22, [x23, #-512]");
  COMPARE(ldp(w24, w25, MemOperand(x26, 252, PreIndex)),
          "ldp w24, w25, [x26, #252]!");
  COMPARE(ldp(x27, x28, MemOperand(x29, 504, PreIndex)),
          "ldp x27, x28, [x29, #504]!");
  COMPARE(ldp(w30, w0, MemOperand(x1, -256, PreIndex)),
          "ldp w30, w0, [x1, #-256]!");
  COMPARE(ldp(x2, x3, MemOperand(x4, -512, PreIndex)),
          "ldp x2, x3, [x4, #-512]!");
  COMPARE(ldp(w5, w6, MemOperand(x7, 252, PostIndex)),
          "ldp w5, w6, [x7], #252");
  COMPARE(ldp(x8, x9, MemOperand(x10, 504, PostIndex)),
          "ldp x8, x9, [x10], #504");
  COMPARE(ldp(w11, w12, MemOperand(x13, -256, PostIndex)),
          "ldp w11, w12, [x13], #-256");
  COMPARE(ldp(x14, x15, MemOperand(x16, -512, PostIndex)),
          "ldp x14, x15, [x16], #-512");
  COMPARE(ldp(x0, x1, MemOperand(x2, 0, PostIndex)), "ldp x0, x1, [x2], #0");
  COMPARE(ldp(w3, w4, MemOperand(x5, 0, PreIndex)), "ldp w3, w4, [x5, #0]!");

  COMPARE(ldp(s17, s18, MemOperand(x19)), "ldp s17, s18, [x19]");
  COMPARE(ldp(s20, s21, MemOperand(x22, 252)), "ldp s20, s21, [x22, #252]");
  COMPARE(ldp(s23, s24, MemOperand(x25, -256)), "ldp s23, s24, [x25, #-256]");
  COMPARE(ldp(s26, s27, MemOperand(x28, 252, PreIndex)),
          "ldp s26, s27, [x28, #252]!");
  COMPARE(ldp(s29, s30, MemOperand(x29, -256, PreIndex)),
          "ldp s29, s30, [x29, #-256]!");
  COMPARE(ldp(s31, s0, MemOperand(x1, 252, PostIndex)),
          "ldp s31, s0, [x1], #252");
  COMPARE(ldp(s2, s3, MemOperand(x4, -256, PostIndex)),
          "ldp s2, s3, [x4], #-256");
  COMPARE(ldp(d17, d18, MemOperand(x19)), "ldp d17, d18, [x19]");
  COMPARE(ldp(d20, d21, MemOperand(x22, 504)), "ldp d20, d21, [x22, #504]");
  COMPARE(ldp(d23, d24, MemOperand(x25, -512)), "ldp d23, d24, [x25, #-512]");
  COMPARE(ldp(d26, d27, MemOperand(x28, 504, PreIndex)),
          "ldp d26, d27, [x28, #504]!");
  COMPARE(ldp(d29, d30, MemOperand(x29, -512, PreIndex)),
          "ldp d29, d30, [x29, #-512]!");
  COMPARE(ldp(d31, d0, MemOperand(x1, 504, PostIndex)),
          "ldp d31, d0, [x1], #504");
  COMPARE(ldp(d2, d3, MemOperand(x4, -512, PostIndex)),
          "ldp d2, d3, [x4], #-512");
  COMPARE(ldp(s0, s1, MemOperand(x2, 0, PostIndex)), "ldp s0, s1, [x2], #0");
  COMPARE(ldp(d3, d4, MemOperand(x5, 0, PreIndex)), "ldp d3, d4, [x5, #0]!");

  COMPARE(ldp(q5, q6, MemOperand(x7)), "ldp q5, q6, [x7]");
  COMPARE(ldp(q8, q9, MemOperand(x10, 1008)), "ldp q8, q9, [x10, #1008]");
  COMPARE(ldp(q11, q12, MemOperand(x13, -1024)), "ldp q11, q12, [x13, #-1024]");
  COMPARE(ldp(q14, q15, MemOperand(x16, 1008, PreIndex)),
          "ldp q14, q15, [x16, #1008]!");
  COMPARE(ldp(q17, q18, MemOperand(x19, -1024, PreIndex)),
          "ldp q17, q18, [x19, #-1024]!");
  COMPARE(ldp(q20, q21, MemOperand(x22, 1008, PostIndex)),
          "ldp q20, q21, [x22], #1008");
  COMPARE(ldp(q23, q24, MemOperand(x25, -1024, PostIndex)),
          "ldp q23, q24, [x25], #-1024");
  COMPARE(ldp(q6, q7, MemOperand(x8, 0, PreIndex)), "ldp q6, q7, [x8, #0]!");

  COMPARE(stp(w0, w1, MemOperand(x2)), "stp w0, w1, [x2]");
  COMPARE(stp(x3, x4, MemOperand(x5)), "stp x3, x4, [x5]");
  COMPARE(stp(w6, w7, MemOperand(x8, 4)), "stp w6, w7, [x8, #4]");
  COMPARE(stp(x9, x10, MemOperand(x11, 8)), "stp x9, x10, [x11, #8]");
  COMPARE(stp(w12, w13, MemOperand(x14, 252)), "stp w12, w13, [x14, #252]");
  COMPARE(stp(x15, x16, MemOperand(x17, 504)), "stp x15, x16, [x17, #504]");
  COMPARE(stp(w18, w19, MemOperand(x20, -256)), "stp w18, w19, [x20, #-256]");
  COMPARE(stp(x21, x22, MemOperand(x23, -512)), "stp x21, x22, [x23, #-512]");
  COMPARE(stp(w24, w25, MemOperand(x26, 252, PreIndex)),
          "stp w24, w25, [x26, #252]!");
  COMPARE(stp(x27, x28, MemOperand(x29, 504, PreIndex)),
          "stp x27, x28, [x29, #504]!");
  COMPARE(stp(w30, w0, MemOperand(x1, -256, PreIndex)),
          "stp w30, w0, [x1, #-256]!");
  COMPARE(stp(x2, x3, MemOperand(x4, -512, PreIndex)),
          "stp x2, x3, [x4, #-512]!");
  COMPARE(stp(w5, w6, MemOperand(x7, 252, PostIndex)),
          "stp w5, w6, [x7], #252");
  COMPARE(stp(x8, x9, MemOperand(x10, 504, PostIndex)),
          "stp x8, x9, [x10], #504");
  COMPARE(stp(w11, w12, MemOperand(x13, -256, PostIndex)),
          "stp w11, w12, [x13], #-256");
  COMPARE(stp(x14, x15, MemOperand(x16, -512, PostIndex)),
          "stp x14, x15, [x16], #-512");
  COMPARE(stp(x0, x1, MemOperand(x2, 0, PostIndex)), "stp x0, x1, [x2], #0");
  COMPARE(stp(w3, w4, MemOperand(x5, 0, PreIndex)), "stp w3, w4, [x5, #0]!");

  COMPARE(stp(s17, s18, MemOperand(x19)), "stp s17, s18, [x19]");
  COMPARE(stp(s20, s21, MemOperand(x22, 252)), "stp s20, s21, [x22, #252]");
  COMPARE(stp(s23, s24, MemOperand(x25, -256)), "stp s23, s24, [x25, #-256]");
  COMPARE(stp(s26, s27, MemOperand(x28, 252, PreIndex)),
          "stp s26, s27, [x28, #252]!");
  COMPARE(stp(s29, s30, MemOperand(x29, -256, PreIndex)),
          "stp s29, s30, [x29, #-256]!");
  COMPARE(stp(s31, s0, MemOperand(x1, 252, PostIndex)),
          "stp s31, s0, [x1], #252");
  COMPARE(stp(s2, s3, MemOperand(x4, -256, PostIndex)),
          "stp s2, s3, [x4], #-256");
  COMPARE(stp(d17, d18, MemOperand(x19)), "stp d17, d18, [x19]");
  COMPARE(stp(d20, d21, MemOperand(x22, 504)), "stp d20, d21, [x22, #504]");
  COMPARE(stp(d23, d24, MemOperand(x25, -512)), "stp d23, d24, [x25, #-512]");
  COMPARE(stp(d26, d27, MemOperand(x28, 504, PreIndex)),
          "stp d26, d27, [x28, #504]!");
  COMPARE(stp(d29, d30, MemOperand(x29, -512, PreIndex)),
          "stp d29, d30, [x29, #-512]!");
  COMPARE(stp(d31, d0, MemOperand(x1, 504, PostIndex)),
          "stp d31, d0, [x1], #504");
  COMPARE(stp(d2, d3, MemOperand(x4, -512, PostIndex)),
          "stp d2, d3, [x4], #-512");
  COMPARE(stp(s0, s1, MemOperand(x2, 0, PostIndex)), "stp s0, s1, [x2], #0");
  COMPARE(stp(d3, d4, MemOperand(x5, 0, PreIndex)), "stp d3, d4, [x5, #0]!");

  COMPARE(stp(q5, q6, MemOperand(x7)), "stp q5, q6, [x7]");
  COMPARE(stp(q8, q9, MemOperand(x10, 1008)), "stp q8, q9, [x10, #1008]");
  COMPARE(stp(q11, q12, MemOperand(x13, -1024)), "stp q11, q12, [x13, #-1024]");
  COMPARE(stp(q14, q15, MemOperand(x16, 1008, PreIndex)),
          "stp q14, q15, [x16, #1008]!");
  COMPARE(stp(q17, q18, MemOperand(x19, -1024, PreIndex)),
          "stp q17, q18, [x19, #-1024]!");
  COMPARE(stp(q20, q21, MemOperand(x22, 1008, PostIndex)),
          "stp q20, q21, [x22], #1008");
  COMPARE(stp(q23, q24, MemOperand(x25, -1024, PostIndex)),
          "stp q23, q24, [x25], #-1024");
  COMPARE(stp(q6, q7, MemOperand(x8, 0, PreIndex)), "stp q6, q7, [x8, #0]!");

  COMPARE(ldp(w16, w17, MemOperand(sp, 4, PostIndex)),
          "ldp w16, w17, [sp], #4");
  COMPARE(stp(x18, x19, MemOperand(sp, -8, PreIndex)),
          "stp x18, x19, [sp, #-8]!");
  COMPARE(ldp(s30, s31, MemOperand(sp, 12, PostIndex)),
          "ldp s30, s31, [sp], #12");
  COMPARE(stp(d30, d31, MemOperand(sp, -16)), "stp d30, d31, [sp, #-16]");
  COMPARE(ldp(q30, q31, MemOperand(sp, 32, PostIndex)),
          "ldp q30, q31, [sp], #32");

  COMPARE(ldpsw(x0, x1, MemOperand(x2)), "ldpsw x0, x1, [x2]");
  COMPARE(ldpsw(x3, x4, MemOperand(x5, 16)), "ldpsw x3, x4, [x5, #16]");
  COMPARE(ldpsw(x6, x7, MemOperand(x8, -32, PreIndex)),
          "ldpsw x6, x7, [x8, #-32]!");
  COMPARE(ldpsw(x9, x10, MemOperand(x11, 128, PostIndex)),
          "ldpsw x9, x10, [x11], #128");
  COMPARE(ldpsw(x0, x1, MemOperand(x10, 0, PreIndex)),
          "ldpsw x0, x1, [x10, #0]!");
  COMPARE(ldpsw(x2, x3, MemOperand(x10, 0, PostIndex)),
          "ldpsw x2, x3, [x10], #0");

  CLEANUP();
}


TEST(load_pauth) {
  SETUP();

  COMPARE(ldraa(x0, MemOperand(x1)), "ldraa x0, [x1]");
  COMPARE(ldraa(x2, MemOperand(sp)), "ldraa x2, [sp]");
  COMPARE(ldraa(x3, MemOperand(x4, 64)), "ldraa x3, [x4, #64]");
  COMPARE(ldraa(x5, MemOperand(sp, 512)), "ldraa x5, [sp, #512]");
  COMPARE(ldraa(x6, MemOperand(x7, -256)), "ldraa x6, [x7, #-256]");
  COMPARE(ldraa(x8, MemOperand(sp, -1024)), "ldraa x8, [sp, #-1024]");
  COMPARE(ldraa(x9, MemOperand(x10, 2048, PreIndex)),
          "ldraa x9, [x10, #2048]!");

  COMPARE(ldrab(x9, MemOperand(x10)), "ldrab x9, [x10]");
  COMPARE(ldrab(x11, MemOperand(sp)), "ldrab x11, [sp]");
  COMPARE(ldrab(x12, MemOperand(x13, 64)), "ldrab x12, [x13, #64]");
  COMPARE(ldrab(x14, MemOperand(sp, 512)), "ldrab x14, [sp, #512]");
  COMPARE(ldrab(x15, MemOperand(x16, -256)), "ldrab x15, [x16, #-256]");
  COMPARE(ldrab(x17, MemOperand(sp, -1024)), "ldrab x17, [sp, #-1024]");
  COMPARE(ldrab(x18, MemOperand(x19, 2048, PreIndex)),
          "ldrab x18, [x19, #2048]!");

  CLEANUP();
}


TEST(load_store_exclusive) {
  SETUP();

  COMPARE(stxrb(w0, w1, MemOperand(x2)), "stxrb w0, w1, [x2]");
  COMPARE(stxrb(x3, w4, MemOperand(sp)), "stxrb w3, w4, [sp]");
  COMPARE(stxrb(w5, x6, MemOperand(x7)), "stxrb w5, w6, [x7]");
  COMPARE(stxrb(x8, x9, MemOperand(sp)), "stxrb w8, w9, [sp]");
  COMPARE(stxrh(w10, w11, MemOperand(x12)), "stxrh w10, w11, [x12]");
  COMPARE(stxrh(x13, w14, MemOperand(sp)), "stxrh w13, w14, [sp]");
  COMPARE(stxrh(w15, x16, MemOperand(x17)), "stxrh w15, w16, [x17]");
  COMPARE(stxrh(x18, x19, MemOperand(sp)), "stxrh w18, w19, [sp]");
  COMPARE(stxr(w20, w21, MemOperand(x22)), "stxr w20, w21, [x22]");
  COMPARE(stxr(x23, w24, MemOperand(sp)), "stxr w23, w24, [sp]");
  COMPARE(stxr(w25, x26, MemOperand(x27)), "stxr w25, x26, [x27]");
  COMPARE(stxr(x28, x29, MemOperand(sp)), "stxr w28, x29, [sp]");
  COMPARE(ldxrb(w30, MemOperand(x0)), "ldxrb w30, [x0]");
  COMPARE(ldxrb(w1, MemOperand(sp)), "ldxrb w1, [sp]");
  COMPARE(ldxrb(x2, MemOperand(x3)), "ldxrb w2, [x3]");
  COMPARE(ldxrb(x4, MemOperand(sp)), "ldxrb w4, [sp]");
  COMPARE(ldxrh(w5, MemOperand(x6)), "ldxrh w5, [x6]");
  COMPARE(ldxrh(w7, MemOperand(sp)), "ldxrh w7, [sp]");
  COMPARE(ldxrh(x8, MemOperand(x9)), "ldxrh w8, [x9]");
  COMPARE(ldxrh(x10, MemOperand(sp)), "ldxrh w10, [sp]");
  COMPARE(ldxr(w11, MemOperand(x12)), "ldxr w11, [x12]");
  COMPARE(ldxr(w13, MemOperand(sp)), "ldxr w13, [sp]");
  COMPARE(ldxr(x14, MemOperand(x15)), "ldxr x14, [x15]");
  COMPARE(ldxr(x16, MemOperand(sp)), "ldxr x16, [sp]");
  COMPARE(stxp(w17, w18, w19, MemOperand(x20)), "stxp w17, w18, w19, [x20]");
  COMPARE(stxp(x21, w22, w23, MemOperand(sp)), "stxp w21, w22, w23, [sp]");
  COMPARE(stxp(w24, x25, x26, MemOperand(x27)), "stxp w24, x25, x26, [x27]");
  COMPARE(stxp(x28, x29, x30, MemOperand(sp)), "stxp w28, x29, x30, [sp]");
  COMPARE(ldxp(w0, w1, MemOperand(x2)), "ldxp w0, w1, [x2]");
  COMPARE(ldxp(w3, w4, MemOperand(sp)), "ldxp w3, w4, [sp]");
  COMPARE(ldxp(x5, x6, MemOperand(x7)), "ldxp x5, x6, [x7]");
  COMPARE(ldxp(x8, x9, MemOperand(sp)), "ldxp x8, x9, [sp]");
  COMPARE(stlxrb(w10, w11, MemOperand(x12)), "stlxrb w10, w11, [x12]");
  COMPARE(stlxrb(x13, w14, MemOperand(sp)), "stlxrb w13, w14, [sp]");
  COMPARE(stlxrb(w15, x16, MemOperand(x17)), "stlxrb w15, w16, [x17]");
  COMPARE(stlxrb(x18, x19, MemOperand(sp)), "stlxrb w18, w19, [sp]");
  COMPARE(stlxrh(w20, w21, MemOperand(x22)), "stlxrh w20, w21, [x22]");
  COMPARE(stlxrh(x23, w24, MemOperand(sp)), "stlxrh w23, w24, [sp]");
  COMPARE(stlxrh(w25, x26, MemOperand(x27)), "stlxrh w25, w26, [x27]");
  COMPARE(stlxrh(x28, x29, MemOperand(sp)), "stlxrh w28, w29, [sp]");
  COMPARE(stlxr(w30, w0, MemOperand(x1)), "stlxr w30, w0, [x1]");
  COMPARE(stlxr(x2, w3, MemOperand(sp)), "stlxr w2, w3, [sp]");
  COMPARE(stlxr(w4, x5, MemOperand(x6)), "stlxr w4, x5, [x6]");
  COMPARE(stlxr(x7, x8, MemOperand(sp)), "stlxr w7, x8, [sp]");
  COMPARE(ldaxrb(w9, MemOperand(x10)), "ldaxrb w9, [x10]");
  COMPARE(ldaxrb(w11, MemOperand(sp)), "ldaxrb w11, [sp]");
  COMPARE(ldaxrb(x12, MemOperand(x13)), "ldaxrb w12, [x13]");
  COMPARE(ldaxrb(x14, MemOperand(sp)), "ldaxrb w14, [sp]");
  COMPARE(ldaxrh(w15, MemOperand(x16)), "ldaxrh w15, [x16]");
  COMPARE(ldaxrh(w17, MemOperand(sp)), "ldaxrh w17, [sp]");
  COMPARE(ldaxrh(x18, MemOperand(x19)), "ldaxrh w18, [x19]");
  COMPARE(ldaxrh(x20, MemOperand(sp)), "ldaxrh w20, [sp]");
  COMPARE(ldaxr(w21, MemOperand(x22)), "ldaxr w21, [x22]");
  COMPARE(ldaxr(w23, MemOperand(sp)), "ldaxr w23, [sp]");
  COMPARE(ldaxr(x24, MemOperand(x25)), "ldaxr x24, [x25]");
  COMPARE(ldaxr(x26, MemOperand(sp)), "ldaxr x26, [sp]");
  COMPARE(stlxp(w27, w28, w29, MemOperand(x30)), "stlxp w27, w28, w29, [x30]");
  COMPARE(stlxp(x0, w1, w2, MemOperand(sp)), "stlxp w0, w1, w2, [sp]");
  COMPARE(stlxp(w3, x4, x5, MemOperand(x6)), "stlxp w3, x4, x5, [x6]");
  COMPARE(stlxp(x7, x8, x9, MemOperand(sp)), "stlxp w7, x8, x9, [sp]");
  COMPARE(ldaxp(w10, w11, MemOperand(x12)), "ldaxp w10, w11, [x12]");
  COMPARE(ldaxp(w13, w14, MemOperand(sp)), "ldaxp w13, w14, [sp]");
  COMPARE(ldaxp(x15, x16, MemOperand(x17)), "ldaxp x15, x16, [x17]");
  COMPARE(ldaxp(x18, x19, MemOperand(sp)), "ldaxp x18, x19, [sp]");
  COMPARE(stlrb(w20, MemOperand(x21)), "stlrb w20, [x21]");
  COMPARE(stlrb(w22, MemOperand(sp)), "stlrb w22, [sp]");
  COMPARE(stlrb(x23, MemOperand(x24)), "stlrb w23, [x24]");
  COMPARE(stlrb(x25, MemOperand(sp)), "stlrb w25, [sp]");
  COMPARE(stlrh(w26, MemOperand(x27)), "stlrh w26, [x27]");
  COMPARE(stlrh(w28, MemOperand(sp)), "stlrh w28, [sp]");
  COMPARE(stlrh(x29, MemOperand(x30)), "stlrh w29, [x30]");
  COMPARE(stlrh(x0, MemOperand(sp)), "stlrh w0, [sp]");
  COMPARE(stlr(w1, MemOperand(x2)), "stlr w1, [x2]");
  COMPARE(stlr(w3, MemOperand(sp)), "stlr w3, [sp]");
  COMPARE(stlr(x4, MemOperand(x5)), "stlr x4, [x5]");
  COMPARE(stlr(x6, MemOperand(sp)), "stlr x6, [sp]");
  COMPARE(stllrb(w7, MemOperand(x8)), "stllrb w7, [x8]");
  COMPARE(stllrb(w9, MemOperand(sp)), "stllrb w9, [sp]");
  COMPARE(stllrb(x10, MemOperand(x11)), "stllrb w10, [x11]");
  COMPARE(stllrb(x12, MemOperand(sp)), "stllrb w12, [sp]");
  COMPARE(stllrh(w13, MemOperand(x14)), "stllrh w13, [x14]");
  COMPARE(stllrh(w15, MemOperand(sp)), "stllrh w15, [sp]");
  COMPARE(stllrh(x16, MemOperand(x17)), "stllrh w16, [x17]");
  COMPARE(stllrh(x18, MemOperand(sp)), "stllrh w18, [sp]");
  COMPARE(stllr(w19, MemOperand(x20)), "stllr w19, [x20]");
  COMPARE(stllr(w21, MemOperand(sp)), "stllr w21, [sp]");
  COMPARE(stllr(x22, MemOperand(x23)), "stllr x22, [x23]");
  COMPARE(stllr(x24, MemOperand(sp)), "stllr x24, [sp]");
  COMPARE(ldarb(w25, MemOperand(x26)), "ldarb w25, [x26]");
  COMPARE(ldarb(w27, MemOperand(sp)), "ldarb w27, [sp]");
  COMPARE(ldarb(x28, MemOperand(x29)), "ldarb w28, [x29]");
  COMPARE(ldarb(x30, MemOperand(sp)), "ldarb w30, [sp]");
  COMPARE(ldarh(w0, MemOperand(x1)), "ldarh w0, [x1]");
  COMPARE(ldarh(w2, MemOperand(sp)), "ldarh w2, [sp]");
  COMPARE(ldarh(x3, MemOperand(x4)), "ldarh w3, [x4]");
  COMPARE(ldarh(x5, MemOperand(sp)), "ldarh w5, [sp]");
  COMPARE(ldar(w6, MemOperand(x7)), "ldar w6, [x7]");
  COMPARE(ldar(w8, MemOperand(sp)), "ldar w8, [sp]");
  COMPARE(ldar(x9, MemOperand(x10)), "ldar x9, [x10]");
  COMPARE(ldar(x11, MemOperand(sp)), "ldar x11, [sp]");
  COMPARE(ldlarb(w12, MemOperand(x13)), "ldlarb w12, [x13]");
  COMPARE(ldlarb(w14, MemOperand(sp)), "ldlarb w14, [sp]");
  COMPARE(ldlarb(x15, MemOperand(x16)), "ldlarb w15, [x16]");
  COMPARE(ldlarb(x17, MemOperand(sp)), "ldlarb w17, [sp]");
  COMPARE(ldlarh(w18, MemOperand(x19)), "ldlarh w18, [x19]");
  COMPARE(ldlarh(w20, MemOperand(sp)), "ldlarh w20, [sp]");
  COMPARE(ldlarh(x21, MemOperand(x22)), "ldlarh w21, [x22]");
  COMPARE(ldlarh(x23, MemOperand(sp)), "ldlarh w23, [sp]");
  COMPARE(ldlar(w24, MemOperand(x25)), "ldlar w24, [x25]");
  COMPARE(ldlar(w26, MemOperand(sp)), "ldlar w26, [sp]");
  COMPARE(ldlar(x27, MemOperand(x28)), "ldlar x27, [x28]");
  COMPARE(ldlar(x29, MemOperand(sp)), "ldlar x29, [sp]");

  COMPARE(cas(w30, w0, MemOperand(x1)), "cas w30, w0, [x1]");
  COMPARE(cas(w2, w3, MemOperand(sp)), "cas w2, w3, [sp]");
  COMPARE(cas(x4, x5, MemOperand(x6)), "cas x4, x5, [x6]");
  COMPARE(cas(x7, x8, MemOperand(sp)), "cas x7, x8, [sp]");
  COMPARE(casa(w9, w10, MemOperand(x11)), "casa w9, w10, [x11]");
  COMPARE(casa(w12, w13, MemOperand(sp)), "casa w12, w13, [sp]");
  COMPARE(casa(x14, x15, MemOperand(x16)), "casa x14, x15, [x16]");
  COMPARE(casa(x17, x18, MemOperand(sp)), "casa x17, x18, [sp]");
  COMPARE(casl(w19, w20, MemOperand(x21)), "casl w19, w20, [x21]");
  COMPARE(casl(w22, w23, MemOperand(sp)), "casl w22, w23, [sp]");
  COMPARE(casl(x24, x25, MemOperand(x26)), "casl x24, x25, [x26]");
  COMPARE(casl(x27, x28, MemOperand(sp)), "casl x27, x28, [sp]");
  COMPARE(casal(w29, w30, MemOperand(x0)), "casal w29, w30, [x0]");
  COMPARE(casal(w1, w2, MemOperand(sp)), "casal w1, w2, [sp]");
  COMPARE(casal(x3, x4, MemOperand(x5)), "casal x3, x4, [x5]");
  COMPARE(casal(x6, x7, MemOperand(sp)), "casal x6, x7, [sp]");
  COMPARE(casb(w8, w9, MemOperand(x10)), "casb w8, w9, [x10]");
  COMPARE(casb(w11, w12, MemOperand(sp)), "casb w11, w12, [sp]");
  COMPARE(casab(w13, w14, MemOperand(x15)), "casab w13, w14, [x15]");
  COMPARE(casab(w16, w17, MemOperand(sp)), "casab w16, w17, [sp]");
  COMPARE(caslb(w18, w19, MemOperand(x20)), "caslb w18, w19, [x20]");
  COMPARE(caslb(w21, w22, MemOperand(sp)), "caslb w21, w22, [sp]");
  COMPARE(casalb(w23, w24, MemOperand(x25)), "casalb w23, w24, [x25]");
  COMPARE(casalb(w26, w27, MemOperand(sp)), "casalb w26, w27, [sp]");
  COMPARE(cash(w28, w29, MemOperand(x30)), "cash w28, w29, [x30]");
  COMPARE(cash(w0, w1, MemOperand(sp)), "cash w0, w1, [sp]");
  COMPARE(casah(w2, w3, MemOperand(x4)), "casah w2, w3, [x4]");
  COMPARE(casah(w5, w6, MemOperand(sp)), "casah w5, w6, [sp]");
  COMPARE(caslh(w7, w8, MemOperand(x9)), "caslh w7, w8, [x9]");
  COMPARE(caslh(w10, w11, MemOperand(sp)), "caslh w10, w11, [sp]");
  COMPARE(casalh(w12, w13, MemOperand(x14)), "casalh w12, w13, [x14]");
  COMPARE(casalh(w15, w16, MemOperand(sp)), "casalh w15, w16, [sp]");
  COMPARE(casp(w18, w19, w20, w21, MemOperand(x22)),
          "casp w18, w19, w20, w21, [x22]");
  COMPARE(casp(w24, w25, w26, w27, MemOperand(sp)),
          "casp w24, w25, w26, w27, [sp]");
  COMPARE(casp(x28, x29, x0, x1, MemOperand(x2)),
          "casp x28, x29, x0, x1, [x2]");
  COMPARE(casp(x4, x5, x6, x7, MemOperand(sp)), "casp x4, x5, x6, x7, [sp]");
  COMPARE(caspa(w8, w9, w10, w11, MemOperand(x12)),
          "caspa w8, w9, w10, w11, [x12]");
  COMPARE(caspa(w14, w15, w16, w17, MemOperand(sp)),
          "caspa w14, w15, w16, w17, [sp]");
  COMPARE(caspa(x18, x19, x20, x21, MemOperand(x22)),
          "caspa x18, x19, x20, x21, [x22]");
  COMPARE(caspa(x24, x25, x26, x27, MemOperand(sp)),
          "caspa x24, x25, x26, x27, [sp]");
  COMPARE(caspl(w28, w29, w0, w1, MemOperand(x2)),
          "caspl w28, w29, w0, w1, [x2]");
  COMPARE(caspl(w4, w5, w6, w7, MemOperand(sp)), "caspl w4, w5, w6, w7, [sp]");
  COMPARE(caspl(x8, x9, x10, x11, MemOperand(x12)),
          "caspl x8, x9, x10, x11, [x12]");
  COMPARE(caspl(x14, x15, x16, x17, MemOperand(sp)),
          "caspl x14, x15, x16, x17, [sp]");
  COMPARE(caspal(w18, w19, w20, w21, MemOperand(x22)),
          "caspal w18, w19, w20, w21, [x22]");
  COMPARE(caspal(w24, w25, w26, w27, MemOperand(sp)),
          "caspal w24, w25, w26, w27, [sp]");
  COMPARE(caspal(x28, x29, x0, x1, MemOperand(x2)),
          "caspal x28, x29, x0, x1, [x2]");
  COMPARE(caspal(x4, x5, x6, x7, MemOperand(sp)),
          "caspal x4, x5, x6, x7, [sp]");


  CLEANUP();
}

// clang-format off
#define ATOMIC_MEMORY_DISASM_LIST(V, DEF) \
  V(DEF, add,  "add")                     \
  V(DEF, clr,  "clr")                     \
  V(DEF, eor,  "eor")                     \
  V(DEF, set,  "set")                     \
  V(DEF, smax, "smax")                    \
  V(DEF, smin, "smin")                    \
  V(DEF, umax, "umax")                    \
  V(DEF, umin, "umin")

#define ATOMIC_MEMORY_DISASM_STORE_X_MODES(V, NAME, STR) \
  V(NAME,    STR)                                        \
  V(NAME##l, STR "l")


#define ATOMIC_MEMORY_DISASM_STORE_W_MODES(V, NAME, STR) \
  ATOMIC_MEMORY_DISASM_STORE_X_MODES(V, NAME, STR)       \
  V(NAME##b,  STR "b")                                   \
  V(NAME##lb, STR "lb")                                  \
  V(NAME##h,  STR "h")                                   \
  V(NAME##lh, STR "lh")

#define ATOMIC_MEMORY_DISASM_LOAD_X_MODES(V, NAME, STR) \
  ATOMIC_MEMORY_DISASM_STORE_X_MODES(V, NAME, STR)      \
  V(NAME##a,  STR "a")                                  \
  V(NAME##al, STR "al")

#define ATOMIC_MEMORY_DISASM_LOAD_W_MODES(V, NAME, STR) \
  ATOMIC_MEMORY_DISASM_LOAD_X_MODES(V, NAME, STR)       \
  V(NAME##ab,  STR "ab")                                \
  V(NAME##alb, STR "alb")                               \
  V(NAME##ah,  STR "ah")                                \
  V(NAME##alh, STR "alh")
// clang-format on

TEST(atomic_memory) {
  SETUP();

// These macros generate tests for all the variations of the atomic memory
// operations, e.g. ldadd, ldadda, ldaddb, staddl, etc.

#define AM_LOAD_X_TESTS(N, MN)                                     \
  COMPARE(ld##N(x0, x1, MemOperand(x2)), "ld" MN " x0, x1, [x2]"); \
  COMPARE(ld##N(x3, x4, MemOperand(sp)), "ld" MN " x3, x4, [sp]");
#define AM_LOAD_W_TESTS(N, MN)                                     \
  COMPARE(ld##N(w0, w1, MemOperand(x2)), "ld" MN " w0, w1, [x2]"); \
  COMPARE(ld##N(w3, w4, MemOperand(sp)), "ld" MN " w3, w4, [sp]");
#define AM_STORE_X_TESTS(N, MN)                            \
  COMPARE(st##N(x0, MemOperand(x1)), "st" MN " x0, [x1]"); \
  COMPARE(st##N(x2, MemOperand(sp)), "st" MN " x2, [sp]");
#define AM_STORE_W_TESTS(N, MN)                            \
  COMPARE(st##N(w0, MemOperand(x1)), "st" MN " w0, [x1]"); \
  COMPARE(st##N(w2, MemOperand(sp)), "st" MN " w2, [sp]");

  ATOMIC_MEMORY_DISASM_LIST(ATOMIC_MEMORY_DISASM_LOAD_X_MODES, AM_LOAD_X_TESTS)
  ATOMIC_MEMORY_DISASM_LIST(ATOMIC_MEMORY_DISASM_LOAD_W_MODES, AM_LOAD_W_TESTS)
  ATOMIC_MEMORY_DISASM_LIST(ATOMIC_MEMORY_DISASM_STORE_X_MODES,
                            AM_STORE_X_TESTS)
  ATOMIC_MEMORY_DISASM_LIST(ATOMIC_MEMORY_DISASM_STORE_W_MODES,
                            AM_STORE_W_TESTS)

#define AM_SWP_X_TESTS(N, MN)                             \
  COMPARE(N(x0, x1, MemOperand(x2)), MN " x0, x1, [x2]"); \
  COMPARE(N(x3, x4, MemOperand(sp)), MN " x3, x4, [sp]");
#define AM_SWP_W_TESTS(N, MN)                             \
  COMPARE(N(w0, w1, MemOperand(x2)), MN " w0, w1, [x2]"); \
  COMPARE(N(w3, w4, MemOperand(sp)), MN " w3, w4, [sp]");


  ATOMIC_MEMORY_DISASM_LOAD_X_MODES(AM_SWP_X_TESTS, swp, "swp")
  ATOMIC_MEMORY_DISASM_LOAD_W_MODES(AM_SWP_W_TESTS, swp, "swp")

#undef AM_LOAD_X_TESTS
#undef AM_LOAD_W_TESTS
#undef AM_STORE_X_TESTS
#undef AM_STORE_W_TESTS
#undef AM_SWP_X_TESTS
#undef AM_SWP_W_TESTS

  COMPARE(ldaprb(w0, MemOperand(x1)), "ldaprb w0, [x1]");
  COMPARE(ldaprb(w2, MemOperand(sp)), "ldaprb w2, [sp]");
  COMPARE(ldaprh(w3, MemOperand(x4)), "ldaprh w3, [x4]");
  COMPARE(ldaprh(w5, MemOperand(sp)), "ldaprh w5, [sp]");
  COMPARE(ldapr(w6, MemOperand(x7)), "ldapr w6, [x7]");
  COMPARE(ldapr(w8, MemOperand(sp)), "ldapr w8, [sp]");
  COMPARE(ldapr(x9, MemOperand(x10)), "ldapr x9, [x10]");
  COMPARE(ldapr(x11, MemOperand(sp)), "ldapr x11, [sp]");

  CLEANUP();
}

TEST(load_store_rcpc_unscaled_offset) {
  SETUP();

  COMPARE(ldapurb(w0, MemOperand(x1)), "ldapurb w0, [x1]");
  COMPARE(ldapurb(w2, MemOperand(x3, 13)), "ldapurb w2, [x3, #13]");
  COMPARE(ldapursb(w4, MemOperand(x5, 129)), "ldapursb w4, [x5, #129]");
  COMPARE(ldapursb(x6, MemOperand(sp, 64)), "ldapursb x6, [sp, #64]");
  COMPARE(ldapurh(w7, MemOperand(x8)), "ldapurh w7, [x8]");
  COMPARE(ldapurh(w9, MemOperand(x10, 13)), "ldapurh w9, [x10, #13]");
  COMPARE(ldapursh(w11, MemOperand(x12, 129)), "ldapursh w11, [x12, #129]");
  COMPARE(ldapursh(x13, MemOperand(sp, 64)), "ldapursh x13, [sp, #64]");
  COMPARE(ldapur(w14, MemOperand(x15)), "ldapur w14, [x15]");
  COMPARE(ldapur(w16, MemOperand(x17, 13)), "ldapur w16, [x17, #13]");
  COMPARE(ldapursw(x18, MemOperand(sp, 64)), "ldapursw x18, [sp, #64]");
  COMPARE(ldapur(x19, MemOperand(x20)), "ldapur x19, [x20]");
  COMPARE(ldapur(x21, MemOperand(sp, 64)), "ldapur x21, [sp, #64]");

  COMPARE(stlurb(w22, MemOperand(x23)), "stlurb w22, [x23]");
  COMPARE(stlurb(w24, MemOperand(sp, 64)), "stlurb w24, [sp, #64]");
  COMPARE(stlurh(w25, MemOperand(x26)), "stlurh w25, [x26]");
  COMPARE(stlurh(w27, MemOperand(sp, 64)), "stlurh w27, [sp, #64]");
  COMPARE(stlur(w28, MemOperand(x29)), "stlur w28, [x29]");
  COMPARE(stlur(w0, MemOperand(sp, 64)), "stlur w0, [sp, #64]");
  COMPARE(stlur(x1, MemOperand(x2)), "stlur x1, [x2]");
  COMPARE(stlur(x3, MemOperand(sp, 64)), "stlur x3, [sp, #64]");


  COMPARE_MACRO(Ldaprb(w0, MemOperand(x1)), "ldaprb w0, [x1]");
  COMPARE_MACRO(Ldaprb(w2, MemOperand(x3, 13)), "ldapurb w2, [x3, #13]");
  COMPARE_MACRO(Ldaprh(w4, MemOperand(x5)), "ldaprh w4, [x5]");
  COMPARE_MACRO(Ldaprh(w6, MemOperand(x7, 13)), "ldapurh w6, [x7, #13]");
  COMPARE_MACRO(Ldapr(w8, MemOperand(x9)), "ldapr w8, [x9]");
  COMPARE_MACRO(Ldapr(w10, MemOperand(x11, 13)), "ldapur w10, [x11, #13]");
  COMPARE_MACRO(Ldapr(x12, MemOperand(x13)), "ldapr x12, [x13]");
  COMPARE_MACRO(Ldapr(x14, MemOperand(sp, 64)), "ldapur x14, [sp, #64]");

  COMPARE_MACRO(Stlrb(w15, MemOperand(x16)), "stlrb w15, [x16]");
  COMPARE_MACRO(Stlrb(w17, MemOperand(sp, 64)), "stlurb w17, [sp, #64]");
  COMPARE_MACRO(Stlrh(w18, MemOperand(x19)), "stlrh w18, [x19]");
  COMPARE_MACRO(Stlrh(w20, MemOperand(sp, 64)), "stlurh w20, [sp, #64]");
  COMPARE_MACRO(Stlr(w21, MemOperand(x22)), "stlr w21, [x22]");
  COMPARE_MACRO(Stlr(w23, MemOperand(sp, 64)), "stlur w23, [sp, #64]");
  COMPARE_MACRO(Stlr(x24, MemOperand(x25)), "stlr x24, [x25]");
  COMPARE_MACRO(Stlr(x26, MemOperand(sp, 64)), "stlur x26, [sp, #64]");

  CLEANUP();
}


TEST(load_store_pair_nontemp) {
  SETUP();

  COMPARE(ldnp(w0, w1, MemOperand(x2)), "ldnp w0, w1, [x2]");
  COMPARE(stnp(w3, w4, MemOperand(x5, 252)), "stnp w3, w4, [x5, #252]");
  COMPARE(ldnp(w6, w7, MemOperand(x8, -256)), "ldnp w6, w7, [x8, #-256]");
  COMPARE(stnp(x9, x10, MemOperand(x11)), "stnp x9, x10, [x11]");
  COMPARE(ldnp(x12, x13, MemOperand(x14, 504)), "ldnp x12, x13, [x14, #504]");
  COMPARE(stnp(x15, x16, MemOperand(x17, -512)), "stnp x15, x16, [x17, #-512]");
  COMPARE(ldnp(s18, s19, MemOperand(x20)), "ldnp s18, s19, [x20]");
  COMPARE(stnp(s21, s22, MemOperand(x23, 252)), "stnp s21, s22, [x23, #252]");
  COMPARE(ldnp(s24, s25, MemOperand(x26, -256)), "ldnp s24, s25, [x26, #-256]");
  COMPARE(stnp(d27, d28, MemOperand(x29)), "stnp d27, d28, [x29]");
  COMPARE(ldnp(d30, d31, MemOperand(x0, 504)), "ldnp d30, d31, [x0, #504]");
  COMPARE(stnp(d1, d2, MemOperand(x3, -512)), "stnp d1, d2, [x3, #-512]");
  COMPARE(ldnp(q4, q5, MemOperand(x6)), "ldnp q4, q5, [x6]");
  COMPARE(stnp(q7, q8, MemOperand(x9, 1008)), "stnp q7, q8, [x9, #1008]");
  COMPARE(ldnp(q10, q11, MemOperand(x12, -1024)),
          "ldnp q10, q11, [x12, #-1024]");

  CLEANUP();
}


TEST(load_literal_macro) {
  SETUP();

  // In each case, the literal will be placed at PC+8:
  //    ldr   x10, pc+8               // Test instruction.
  //    ldr   xzr, pc+12              // Pool marker.
  //    .word64 #0x1234567890abcdef   // Test literal.

  COMPARE_MACRO_PREFIX(Ldr(x10, 0x1234567890abcdef), "ldr x10, pc+8");
  COMPARE_MACRO_PREFIX(Ldr(w20, 0xfedcba09), "ldr w20, pc+8");
  COMPARE_MACRO_PREFIX(Ldr(d11, 1.234), "ldr d11, pc+8");
  COMPARE_MACRO_PREFIX(Ldr(s22, 2.5f), "ldr s22, pc+8");
  COMPARE_MACRO_PREFIX(Ldrsw(x21, 0x80000000), "ldrsw x21, pc+8");

  CLEANUP();
}


TEST(load_literal) {
  SETUP();

  COMPARE_PREFIX(ldr(x20, INT64_C(0)), "ldr x20, pc+0");
  COMPARE_PREFIX(ldr(x20, 1), "ldr x20, pc+4");
  COMPARE_PREFIX(ldr(x20, -1), "ldr x20, pc-4");
  COMPARE_PREFIX(ldr(x20, 0x3ffff), "ldr x20, pc+1048572");
  COMPARE_PREFIX(ldr(x20, -0x40000), "ldr x20, pc-1048576");
  COMPARE_PREFIX(ldr(w21, INT64_C(0)), "ldr w21, pc+0");
  COMPARE_PREFIX(ldr(w21, 1), "ldr w21, pc+4");
  COMPARE_PREFIX(ldr(w21, -1), "ldr w21, pc-4");
  COMPARE_PREFIX(ldr(w21, 0x3ffff), "ldr w21, pc+1048572");
  COMPARE_PREFIX(ldr(w21, -0x40000), "ldr w21, pc-1048576");
  COMPARE_PREFIX(ldr(d22, INT64_C(0)), "ldr d22, pc+0");
  COMPARE_PREFIX(ldr(d22, 1), "ldr d22, pc+4");
  COMPARE_PREFIX(ldr(d22, -1), "ldr d22, pc-4");
  COMPARE_PREFIX(ldr(d22, 0x3ffff), "ldr d22, pc+1048572");
  COMPARE_PREFIX(ldr(d22, -0x40000), "ldr d22, pc-1048576");
  COMPARE_PREFIX(ldr(s23, INT64_C(0)), "ldr s23, pc+0");
  COMPARE_PREFIX(ldr(s23, 1), "ldr s23, pc+4");
  COMPARE_PREFIX(ldr(s23, -1), "ldr s23, pc-4");
  COMPARE_PREFIX(ldr(s23, 0x3ffff), "ldr s23, pc+1048572");
  COMPARE_PREFIX(ldr(s23, -0x40000), "ldr s23, pc-1048576");
  COMPARE_PREFIX(ldrsw(x24, INT64_C(0)), "ldrsw x24, pc+0");
  COMPARE_PREFIX(ldrsw(x24, 1), "ldrsw x24, pc+4");
  COMPARE_PREFIX(ldrsw(x24, -1), "ldrsw x24, pc-4");
  COMPARE_PREFIX(ldrsw(x24, 0x3ffff), "ldrsw x24, pc+1048572");
  COMPARE_PREFIX(ldrsw(x24, -0x40000), "ldrsw x24, pc-1048576");

  CLEANUP();
}


TEST(prfm_operations) {
  SETUP();

  // Test every encodable prefetch operation.
  const char* expected[] = {
      "prfm pldl1keep, ", "prfm pldl1strm, ", "prfm pldl2keep, ",
      "prfm pldl2strm, ", "prfm pldl3keep, ", "prfm pldl3strm, ",
      "prfm #0b00110, ",  "prfm #0b00111, ",  "prfm plil1keep, ",
      "prfm plil1strm, ", "prfm plil2keep, ", "prfm plil2strm, ",
      "prfm plil3keep, ", "prfm plil3strm, ", "prfm #0b01110, ",
      "prfm #0b01111, ",  "prfm pstl1keep, ", "prfm pstl1strm, ",
      "prfm pstl2keep, ", "prfm pstl2strm, ", "prfm pstl3keep, ",
      "prfm pstl3strm, ", "prfm #0b10110, ",  "prfm #0b10111, ",
      "prfm #0b11000, ",  "prfm #0b11001, ",  "prfm #0b11010, ",
      "prfm #0b11011, ",  "prfm #0b11100, ",  "prfm #0b11101, ",
      "prfm #0b11110, ",  "prfm #0b11111, ",
  };
  const int expected_count = sizeof(expected) / sizeof(expected[0]);
  VIXL_STATIC_ASSERT((1 << ImmPrefetchOperation_width) == expected_count);

  for (int op = 0; op < (1 << ImmPrefetchOperation_width); op++) {
    COMPARE_PREFIX(prfm(op, INT64_C(0)), expected[op]);
    COMPARE_PREFIX(prfm(op, MemOperand(x0, 0)), expected[op]);
    COMPARE_PREFIX(prfm(op, MemOperand(x0, x1)), expected[op]);
  }

  CLEANUP();
}


TEST(prfum_operations) {
  SETUP();

  // Test every encodable prefetch operation.
  const char* expected[] = {
      "prfum pldl1keep, ", "prfum pldl1strm, ", "prfum pldl2keep, ",
      "prfum pldl2strm, ", "prfum pldl3keep, ", "prfum pldl3strm, ",
      "prfum #0b00110, ",  "prfum #0b00111, ",  "prfum plil1keep, ",
      "prfum plil1strm, ", "prfum plil2keep, ", "prfum plil2strm, ",
      "prfum plil3keep, ", "prfum plil3strm, ", "prfum #0b01110, ",
      "prfum #0b01111, ",  "prfum pstl1keep, ", "prfum pstl1strm, ",
      "prfum pstl2keep, ", "prfum pstl2strm, ", "prfum pstl3keep, ",
      "prfum pstl3strm, ", "prfum #0b10110, ",  "prfum #0b10111, ",
      "prfum #0b11000, ",  "prfum #0b11001, ",  "prfum #0b11010, ",
      "prfum #0b11011, ",  "prfum #0b11100, ",  "prfum #0b11101, ",
      "prfum #0b11110, ",  "prfum #0b11111, ",
  };
  const int expected_count = sizeof(expected) / sizeof(expected[0]);
  VIXL_STATIC_ASSERT((1 << ImmPrefetchOperation_width) == expected_count);

  for (int op = 0; op < (1 << ImmPrefetchOperation_width); op++) {
    COMPARE_PREFIX(prfum(op, MemOperand(x0, 0)), expected[op]);
  }

  CLEANUP();
}


TEST(prfm_offset) {
  SETUP();

  COMPARE(prfm(PLDL1KEEP, MemOperand(x1)), "prfm pldl1keep, [x1]");
  COMPARE(prfm(PLDL1STRM, MemOperand(x3, 8)), "prfm pldl1strm, [x3, #8]");
  COMPARE(prfm(PLDL2KEEP, MemOperand(x5, 32760)),
          "prfm pldl2keep, [x5, #32760]");

  COMPARE(prfm(PLDL2STRM, MemOperand(sp)), "prfm pldl2strm, [sp]");
  COMPARE(prfm(PLDL3KEEP, MemOperand(sp, 8)), "prfm pldl3keep, [sp, #8]");
  COMPARE(prfm(PLDL3STRM, MemOperand(sp, 32760)),
          "prfm pldl3strm, [sp, #32760]");

  CLEANUP();
}


TEST(prfm_regoffset) {
  SETUP();

  COMPARE(prfm(PLIL1KEEP, MemOperand(x1, x2)), "prfm plil1keep, [x1, x2]");
  COMPARE(prfm(PLIL1STRM, MemOperand(x3, w4, SXTW)),
          "prfm plil1strm, [x3, w4, sxtw]");
  COMPARE(prfm(PLIL2KEEP, MemOperand(x5, x6, LSL, 3)),
          "prfm plil2keep, [x5, x6, lsl #3]");

  COMPARE(prfm(PLIL2STRM, MemOperand(sp, xzr)), "prfm plil2strm, [sp, xzr]");
  COMPARE(prfm(PLIL3KEEP, MemOperand(sp, wzr, SXTW)),
          "prfm plil3keep, [sp, wzr, sxtw]");
  COMPARE(prfm(PLIL3STRM, MemOperand(sp, xzr, LSL, 3)),
          "prfm plil3strm, [sp, xzr, lsl #3]");

  CLEANUP();
}


TEST(prfm_literal) {
  SETUP();

  COMPARE_PREFIX(prfm(PSTL1KEEP, INT64_C(0)), "prfm pstl1keep, pc+0");
  COMPARE_PREFIX(prfm(PSTL1STRM, 1), "prfm pstl1strm, pc+4");
  COMPARE_PREFIX(prfm(PSTL2KEEP, -1), "prfm pstl2keep, pc-4");
  COMPARE_PREFIX(prfm(PSTL2STRM, 0x3ffff), "prfm pstl2strm, pc+1048572");
  COMPARE_PREFIX(prfm(PSTL3KEEP, -0x3ffff), "prfm pstl3keep, pc-1048572");
  COMPARE_PREFIX(prfm(PSTL3STRM, -0x40000), "prfm pstl3strm, pc-1048576");

  CLEANUP();
}


TEST(prfm_unscaled) {
  SETUP();

  // If an unscaled-offset instruction is requested, it is used, even if the
  // offset could be encoded in a scaled-offset instruction.
  COMPARE(prfum(PLDL1KEEP, MemOperand(x1)), "prfum pldl1keep, [x1]");
  COMPARE(prfum(PLDL1STRM, MemOperand(x1, 8)), "prfum pldl1strm, [x1, #8]");
  COMPARE(prfum(PLDL2KEEP, MemOperand(x1, 248)), "prfum pldl2keep, [x1, #248]");

  // Normal offsets are converted to unscaled offsets if necssary.
  COMPARE(prfm(PLDL2STRM, MemOperand(x1, 1)), "prfum pldl2strm, [x1, #1]");
  COMPARE(prfm(PLDL3KEEP, MemOperand(x1, -1)), "prfum pldl3keep, [x1, #-1]");
  COMPARE(prfm(PLDL3STRM, MemOperand(x1, 255)), "prfum pldl3strm, [x1, #255]");
  COMPARE(prfm(PLDL3STRM, MemOperand(x1, -256)),
          "prfum pldl3strm, [x1, #-256]");

  CLEANUP();
}


TEST(prfm_unscaled_option) {
  SETUP();

  // Just like prfm_unscaled, but specify the scaling option explicitly.

  // Require unscaled-offset forms.
  LoadStoreScalingOption option = RequireUnscaledOffset;

  COMPARE(prfum(PLDL1KEEP, MemOperand(x1), option), "prfum pldl1keep, [x1]");
  COMPARE(prfum(PLDL1STRM, MemOperand(x1, 8), option),
          "prfum pldl1strm, [x1, #8]");
  COMPARE(prfum(PLDL2KEEP, MemOperand(x1, 248), option),
          "prfum pldl2keep, [x1, #248]");
  COMPARE(prfum(PLDL2STRM, MemOperand(x1, 1), option),
          "prfum pldl2strm, [x1, #1]");
  COMPARE(prfum(PLDL3KEEP, MemOperand(x1, -1), option),
          "prfum pldl3keep, [x1, #-1]");
  COMPARE(prfum(PLDL3STRM, MemOperand(x1, 255), option),
          "prfum pldl3strm, [x1, #255]");
  COMPARE(prfum(PLIL1KEEP, MemOperand(x1, -256), option),
          "prfum plil1keep, [x1, #-256]");

  // Require scaled-offset forms..
  option = RequireScaledOffset;

  COMPARE(prfm(PLDL1KEEP, MemOperand(x1), option), "prfm pldl1keep, [x1]");
  COMPARE(prfm(PLDL1STRM, MemOperand(x1, 8), option),
          "prfm pldl1strm, [x1, #8]");
  COMPARE(prfm(PLDL2KEEP, MemOperand(x1, 248), option),
          "prfm pldl2keep, [x1, #248]");
  COMPARE(prfm(PLIL2STRM, MemOperand(x1, 256), option),
          "prfm plil2strm, [x1, #256]");
  COMPARE(prfm(PLIL3KEEP, MemOperand(x1, 32760), option),
          "prfm plil3keep, [x1, #32760]");

  // Prefer unscaled-offset forms, but allow scaled-offset forms if necessary.
  option = PreferUnscaledOffset;

  COMPARE(prfum(PLDL1KEEP, MemOperand(x1), option), "prfum pldl1keep, [x1]");
  COMPARE(prfum(PLDL1STRM, MemOperand(x1, 8), option),
          "prfum pldl1strm, [x1, #8]");
  COMPARE(prfum(PLDL2KEEP, MemOperand(x1, 248), option),
          "prfum pldl2keep, [x1, #248]");
  COMPARE(prfum(PLDL2STRM, MemOperand(x1, 1), option),
          "prfum pldl2strm, [x1, #1]");
  COMPARE(prfum(PLDL3KEEP, MemOperand(x1, -1), option),
          "prfum pldl3keep, [x1, #-1]");
  COMPARE(prfum(PLDL3STRM, MemOperand(x1, 255), option),
          "prfum pldl3strm, [x1, #255]");
  COMPARE(prfum(PLIL1KEEP, MemOperand(x1, -256), option),
          "prfum plil1keep, [x1, #-256]");
  COMPARE(prfum(PLIL1STRM, MemOperand(x1, 256), option),
          "prfm plil1strm, [x1, #256]");
  COMPARE(prfum(PLIL2KEEP, MemOperand(x1, 32760), option),
          "prfm plil2keep, [x1, #32760]");

  // Prefer scaled-offset forms, but allow unscaled-offset forms if necessary.
  option = PreferScaledOffset;

  COMPARE(prfm(PLDL1KEEP, MemOperand(x1), option), "prfm pldl1keep, [x1]");
  COMPARE(prfm(PLDL1STRM, MemOperand(x1, 8), option),
          "prfm pldl1strm, [x1, #8]");
  COMPARE(prfm(PLDL2KEEP, MemOperand(x1, 248), option),
          "prfm pldl2keep, [x1, #248]");
  COMPARE(prfm(PLDL2STRM, MemOperand(x1, 1), option),
          "prfum pldl2strm, [x1, #1]");
  COMPARE(prfm(PLDL3KEEP, MemOperand(x1, -1), option),
          "prfum pldl3keep, [x1, #-1]");
  COMPARE(prfm(PLDL3STRM, MemOperand(x1, 255), option),
          "prfum pldl3strm, [x1, #255]");
  COMPARE(prfm(PLIL1KEEP, MemOperand(x1, -256), option),
          "prfum plil1keep, [x1, #-256]");
  COMPARE(prfm(PLIL1STRM, MemOperand(x1, 256), option),
          "prfm plil1strm, [x1, #256]");
  COMPARE(prfm(PLIL2KEEP, MemOperand(x1, 32760), option),
          "prfm plil2keep, [x1, #32760]");

  CLEANUP();
}


TEST(cond_select) {
  SETUP();

  COMPARE(csel(w0, w1, w2, eq), "csel w0, w1, w2, eq");
  COMPARE(csel(x3, x4, x5, ne), "csel x3, x4, x5, ne");
  COMPARE(csinc(w6, w7, w8, hs), "csinc w6, w7, w8, hs");
  COMPARE(csinc(x9, x10, x11, lo), "csinc x9, x10, x11, lo");
  COMPARE(csinv(w12, w13, w14, mi), "csinv w12, w13, w14, mi");
  COMPARE(csinv(x15, x16, x17, pl), "csinv x15, x16, x17, pl");
  COMPARE(csneg(w18, w19, w20, vs), "csneg w18, w19, w20, vs");
  COMPARE(csneg(x21, x22, x23, vc), "csneg x21, x22, x23, vc");
  COMPARE(cset(w24, hi), "cset w24, hi");
  COMPARE(cset(x25, ls), "cset x25, ls");
  COMPARE(csetm(w26, ge), "csetm w26, ge");
  COMPARE(csetm(x27, lt), "csetm x27, lt");
  COMPARE(cinc(w28, w29, gt), "cinc w28, w29, gt");
  COMPARE(cinc(x30, x0, le), "cinc x30, x0, le");
  COMPARE(cinv(w1, w2, eq), "cinv w1, w2, eq");
  COMPARE(cinv(x3, x4, ne), "cinv x3, x4, ne");
  COMPARE(cneg(w5, w6, hs), "cneg w5, w6, hs");
  COMPARE(cneg(x7, x8, lo), "cneg x7, x8, lo");

  COMPARE(csel(x0, x1, x2, al), "csel x0, x1, x2, al");
  COMPARE(csel(x1, x2, x3, nv), "csel x1, x2, x3, nv");
  COMPARE(csinc(x2, x3, x4, al), "csinc x2, x3, x4, al");
  COMPARE(csinc(x3, x4, x5, nv), "csinc x3, x4, x5, nv");
  COMPARE(csinv(x4, x5, x6, al), "csinv x4, x5, x6, al");
  COMPARE(csinv(x5, x6, x7, nv), "csinv x5, x6, x7, nv");
  COMPARE(csneg(x6, x7, x8, al), "csneg x6, x7, x8, al");
  COMPARE(csneg(x7, x8, x9, nv), "csneg x7, x8, x9, nv");

  CLEANUP();
}


TEST(cond_select_macro) {
  SETUP();

  // In the tests below we also test the `GetCselSynthesisInformation()` helper.
  // These tests are here (rather than in test-assembler-aarch64.cc) because the
  // disassembly makes it easy to see whether or not the inputs are synthesised.
  bool synthesises_left = false;
  bool synthesises_right = false;

  COMPARE_MACRO(Csel(w0, w1, -1, eq), "csinv w0, w1, wzr, eq");
  MacroAssembler::GetCselSynthesisInformation(w0,
                                              w1,
                                              -1,
                                              &synthesises_left,
                                              &synthesises_right);
  VIXL_CHECK(!synthesises_left && !synthesises_right);

  COMPARE_MACRO(Csel(w2, w3, 0, ne), "csel w2, w3, wzr, ne");
  MacroAssembler::GetCselSynthesisInformation(w2,
                                              w3,
                                              wzr,
                                              &synthesises_left,
                                              &synthesises_right);
  VIXL_CHECK(!synthesises_left && !synthesises_right);

  COMPARE_MACRO(Csel(w4, w5, 1, hs), "csinc w4, w5, wzr, hs");
  MacroAssembler::GetCselSynthesisInformation(w4,
                                              w5,
                                              1,
                                              &synthesises_left,
                                              &synthesises_right);
  VIXL_CHECK(!synthesises_left && !synthesises_right);

  COMPARE_MACRO(Csel(x6, x7, -1, lo), "csinv x6, x7, xzr, lo");
  MacroAssembler::GetCselSynthesisInformation(x6,
                                              x7,
                                              xzr,
                                              &synthesises_left,
                                              &synthesises_right);
  VIXL_CHECK(!synthesises_left && !synthesises_right);

  COMPARE_MACRO(Csel(x8, x9, 0, mi), "csel x8, x9, xzr, mi");
  MacroAssembler::GetCselSynthesisInformation(x8,
                                              x9,
                                              xzr,
                                              &synthesises_left,
                                              &synthesises_right);
  VIXL_CHECK(!synthesises_left && !synthesises_right);

  COMPARE_MACRO(Csel(x10, x11, 1, pl), "csinc x10, x11, xzr, pl");
  MacroAssembler::GetCselSynthesisInformation(x10,
                                              x11,
                                              xzr,
                                              &synthesises_left,
                                              &synthesises_right);
  VIXL_CHECK(!synthesises_left && !synthesises_right);

  COMPARE_MACRO(Csel(x12, 0, 0, eq), "mov x12, #0x0");
  MacroAssembler::GetCselSynthesisInformation(x12,
                                              0,
                                              0,
                                              &synthesises_left,
                                              &synthesises_right);
  VIXL_CHECK(!synthesises_left && !synthesises_right);

  COMPARE_MACRO(Csel(w13, 0, 1, eq), "cset w13, ne");
  MacroAssembler::GetCselSynthesisInformation(w13,
                                              0,
                                              1,
                                              &synthesises_left,
                                              &synthesises_right);
  VIXL_CHECK(!synthesises_left && !synthesises_right);

  COMPARE_MACRO(Csel(x14, 1, 0, eq), "cset x14, eq");
  MacroAssembler::GetCselSynthesisInformation(x14,
                                              1,
                                              0,
                                              &synthesises_left,
                                              &synthesises_right);
  VIXL_CHECK(!synthesises_left && !synthesises_right);

  COMPARE_MACRO(Csel(w15, 0, -1, eq), "csetm w15, ne");
  MacroAssembler::GetCselSynthesisInformation(w15,
                                              0,
                                              -1,
                                              &synthesises_left,
                                              &synthesises_right);
  VIXL_CHECK(!synthesises_left && !synthesises_right);

  COMPARE_MACRO(Csel(x18, -1, 0, eq), "csetm x18, eq");
  MacroAssembler::GetCselSynthesisInformation(x18,
                                              -1,
                                              0,
                                              &synthesises_left,
                                              &synthesises_right);
  VIXL_CHECK(!synthesises_left && !synthesises_right);

  COMPARE_MACRO(Csel(w19, -1, 1, eq),
                "mov w19, #0x1\n"
                "cneg w19, w19, eq");
  MacroAssembler::GetCselSynthesisInformation(w19,
                                              -1,
                                              1,
                                              &synthesises_left,
                                              &synthesises_right);
  VIXL_CHECK(!synthesises_left && synthesises_right);

  COMPARE_MACRO(Csel(x20, 1, -1, eq),
                "mov x20, #0xffffffffffffffff\n"
                "cneg x20, x20, eq");
  MacroAssembler::GetCselSynthesisInformation(x20,
                                              1,
                                              -1,
                                              &synthesises_left,
                                              &synthesises_right);
  VIXL_CHECK(!synthesises_left && synthesises_right);

  COMPARE_MACRO(Csel(w21, 0xaa, 0xbb, eq),
                "mov w16, #0xaa\n"
                "mov w17, #0xbb\n"
                "csel w21, w16, w17, eq");
  MacroAssembler::GetCselSynthesisInformation(w21,
                                              0xaa,
                                              0xbb,
                                              &synthesises_left,
                                              &synthesises_right);
  VIXL_CHECK(synthesises_left && synthesises_right);

  COMPARE_MACRO(Csel(x22, 0xaa, -0xbb, eq),
                "mov x16, #0xaa\n"
                "mov x17, #0xffffffffffffff45\n"
                "csel x22, x16, x17, eq");
  MacroAssembler::GetCselSynthesisInformation(x22,
                                              0xaa,
                                              -0xbb,
                                              &synthesises_left,
                                              &synthesises_right);
  VIXL_CHECK(synthesises_left && synthesises_right);

  COMPARE_MACRO(Csel(w23, 0, 0xaa, eq),
                "mov w16, #0xaa\n"
                "csel w23, w16, wzr, ne");
  MacroAssembler::GetCselSynthesisInformation(w23,
                                              0,
                                              0xaa,
                                              &synthesises_left,
                                              &synthesises_right);
  VIXL_CHECK(!synthesises_left && synthesises_right);

  COMPARE_MACRO(Csel(x24, -0xaa, 0, eq),
                "mov x16, #0xffffffffffffff56\n"
                "csel x24, x16, xzr, eq");
  MacroAssembler::GetCselSynthesisInformation(x24,
                                              -0xaa,
                                              0,
                                              &synthesises_left,
                                              &synthesises_right);
  VIXL_CHECK(synthesises_left && !synthesises_right);

  COMPARE_MACRO(Csel(w25, 0xcc, -0xcc, eq),
                "mov w25, #0xffffff34\n"
                "cneg w25, w25, eq");
  MacroAssembler::GetCselSynthesisInformation(w25,
                                              0xcc,
                                              -0xcc,
                                              &synthesises_left,
                                              &synthesises_right);
  VIXL_CHECK(!synthesises_left && synthesises_right);

  COMPARE_MACRO(Csel(x26, -0xcc, 0xcc, eq),
                "mov x26, #0xcc\n"
                "cneg x26, x26, eq");
  MacroAssembler::GetCselSynthesisInformation(w25,
                                              -0xcc,
                                              0xcc,
                                              &synthesises_left,
                                              &synthesises_right);
  VIXL_CHECK(!synthesises_left && synthesises_right);

  // Test with `Operand` inputs.
  COMPARE_MACRO(Csel(x0, x1, Operand(x2, LSL, 3), eq),
                "lsl x16, x2, #3\n"
                "csel x0, x1, x16, eq");
  MacroAssembler::GetCselSynthesisInformation(x0,
                                              x1,
                                              Operand(x2, LSL, 3),
                                              &synthesises_left,
                                              &synthesises_right);
  VIXL_CHECK(!synthesises_left && synthesises_right);

  COMPARE_MACRO(Csel(x3, x4, Operand(x5, SXTH), eq),
                "sxth x16, w5\n"
                "csel x3, x4, x16, eq");
  MacroAssembler::GetCselSynthesisInformation(x3,
                                              x4,
                                              Operand(x5, SXTH),
                                              &synthesises_left,
                                              &synthesises_right);
  VIXL_CHECK(!synthesises_left && synthesises_right);

  COMPARE_MACRO(Csel(x6, Operand(x7, LSL, 7), x8, eq),
                "lsl x16, x7, #7\n"
                "csel x6, x16, x8, eq");
  MacroAssembler::GetCselSynthesisInformation(x6,
                                              Operand(x7, LSL, 7),
                                              x8,
                                              &synthesises_left,
                                              &synthesises_right);
  VIXL_CHECK(synthesises_left && !synthesises_right);

  COMPARE_MACRO(Csel(x9, Operand(x10, SXTH), x11, eq),
                "sxth x16, w10\n"
                "csel x9, x16, x11, eq");
  MacroAssembler::GetCselSynthesisInformation(x9,
                                              Operand(x10, SXTH),
                                              x11,
                                              &synthesises_left,
                                              &synthesises_right);
  VIXL_CHECK(synthesises_left && !synthesises_right);

  COMPARE_MACRO(Csel(x12, Operand(x13, LSL, 13), Operand(x14, SXTB), eq),
                "lsl x16, x13, #13\n"
                "sxtb x17, w14\n"
                "csel x12, x16, x17, eq");
  MacroAssembler::GetCselSynthesisInformation(x12,
                                              Operand(x13, LSL, 13),
                                              Operand(x14, SXTB),
                                              &synthesises_left,
                                              &synthesises_right);
  VIXL_CHECK(synthesises_left && synthesises_right);

  COMPARE_MACRO(Csel(x15, 0, Operand(x18, LSR, 18), eq),
                "lsr x16, x18, #18\n"
                "csel x15, x16, xzr, ne");
  MacroAssembler::GetCselSynthesisInformation(x15,
                                              0,
                                              Operand(x18, LSR, 18),
                                              &synthesises_left,
                                              &synthesises_right);
  VIXL_CHECK(!synthesises_left && synthesises_right);

  // Test with the zero register.
  COMPARE_MACRO(Csel(w19, wzr, wzr, eq), "mov w19, #0x0");
  MacroAssembler::GetCselSynthesisInformation(w19,
                                              wzr,
                                              wzr,
                                              &synthesises_left,
                                              &synthesises_right);
  VIXL_CHECK(!synthesises_left && !synthesises_right);

  COMPARE_MACRO(Csel(x20, x21, xzr, eq), "csel x20, x21, xzr, eq");
  MacroAssembler::GetCselSynthesisInformation(x20,
                                              x21,
                                              xzr,
                                              &synthesises_left,
                                              &synthesises_right);
  VIXL_CHECK(!synthesises_left && !synthesises_right);

  COMPARE_MACRO(Csel(w22, wzr, w23, eq), "csel w22, w23, wzr, ne");
  MacroAssembler::GetCselSynthesisInformation(w22,
                                              wzr,
                                              w23,
                                              &synthesises_left,
                                              &synthesises_right);
  VIXL_CHECK(!synthesises_left && !synthesises_right);

  COMPARE_MACRO(Csel(x24, xzr, 0, eq), "mov x24, #0x0");
  MacroAssembler::GetCselSynthesisInformation(x24,
                                              xzr,
                                              0,
                                              &synthesises_left,
                                              &synthesises_right);
  VIXL_CHECK(!synthesises_left && !synthesises_right);

  COMPARE_MACRO(Csel(w25, wzr, 1, eq), "cset w25, ne");
  MacroAssembler::GetCselSynthesisInformation(w25,
                                              wzr,
                                              1,
                                              &synthesises_left,
                                              &synthesises_right);
  VIXL_CHECK(!synthesises_left && !synthesises_right);


  CLEANUP();
}


TEST(cond_cmp) {
  SETUP();

  COMPARE(ccmn(w0, w1, NZCVFlag, eq), "ccmn w0, w1, #NZCV, eq");
  COMPARE(ccmn(x2, x3, NZCFlag, ne), "ccmn x2, x3, #NZCv, ne");
  COMPARE(ccmp(w4, w5, NZVFlag, hs), "ccmp w4, w5, #NZcV, hs");
  COMPARE(ccmp(x6, x7, NZFlag, lo), "ccmp x6, x7, #NZcv, lo");
  COMPARE(ccmn(w8, 31, NFlag, mi), "ccmn w8, #31, #Nzcv, mi");
  COMPARE(ccmn(x9, 30, NCFlag, pl), "ccmn x9, #30, #NzCv, pl");
  COMPARE(ccmp(w10, 29, NVFlag, vs), "ccmp w10, #29, #NzcV, vs");
  COMPARE(ccmp(x11, 28, NFlag, vc), "ccmp x11, #28, #Nzcv, vc");
  COMPARE(ccmn(w12, w13, NoFlag, al), "ccmn w12, w13, #nzcv, al");
  COMPARE(ccmp(x14, 27, ZVFlag, nv), "ccmp x14, #27, #nZcV, nv");

  CLEANUP();
}


TEST(cond_cmp_macro) {
  SETUP();

  COMPARE_MACRO(Ccmp(w0, -1, VFlag, hi), "ccmn w0, #1, #nzcV, hi");
  COMPARE_MACRO(Ccmp(x1, -31, CFlag, ge), "ccmn x1, #31, #nzCv, ge");
  COMPARE_MACRO(Ccmn(w2, -1, CVFlag, gt), "ccmp w2, #1, #nzCV, gt");
  COMPARE_MACRO(Ccmn(x3, -31, ZCVFlag, ls), "ccmp x3, #31, #nZCV, ls");

  CLEANUP();
}


TEST(system_clrex) {
  SETUP();

  COMPARE(clrex(0), "clrex #0x0");
  COMPARE(clrex(14), "clrex #0xe");
  COMPARE(clrex(15), "clrex");
  COMPARE(clrex(), "clrex");

  CLEANUP();
}


TEST(system_mrs) {
  SETUP();

  COMPARE(mrs(x0, NZCV), "mrs x0, nzcv");
  COMPARE(mrs(x30, NZCV), "mrs x30, nzcv");
  COMPARE(mrs(x15, FPCR), "mrs x15, fpcr");
  COMPARE(mrs(x20, RNDR), "mrs x20, rndr");
  COMPARE(mrs(x5, RNDRRS), "mrs x5, rndrrs");

  // Test mrs that use system registers we haven't named.
  COMPARE(dci(MRS | (0x5555 << 5)), "mrs x0, S3_2_c10_c10_5");
  COMPARE(dci(0xd53e1000), "mrs x0, S3_6_c1_c0_0");

  CLEANUP();
}


TEST(system_msr) {
  SETUP();

  COMPARE(msr(NZCV, x0), "msr nzcv, x0");
  COMPARE(msr(NZCV, x30), "msr nzcv, x30");
  COMPARE(msr(FPCR, x15), "msr fpcr, x15");

  // Test msr that use system registers we haven't named.
  COMPARE(dci(MSR | (0x1234 << 5)), "msr S2_2_c4_c6_4, x0");
  COMPARE(dci(0xd51e1000), "msr S3_6_c1_c0_0, x0");

  CLEANUP();
}


TEST(system_pstate) {
  SETUP();

  COMPARE(cfinv(), "cfinv");
  COMPARE(axflag(), "axflag");
  COMPARE(xaflag(), "xaflag");

  CLEANUP();
}


TEST(system_sys) {
  SETUP();

  COMPARE(sys(0x3, 0x7, 0x5, 0x1, x1), "ic ivau, x1");
  COMPARE(sys(0x3, 0x7, 0xa, 0x1, x2), "dc cvac, x2");
  COMPARE(sys(0x3, 0x7, 0xb, 0x1, x3), "dc cvau, x3");
  COMPARE(sys(0x3, 0x7, 0xe, 0x1, x4), "dc civac, x4");
  COMPARE(sys(0x3, 0x7, 0xc, 0x1, x5), "dc cvap, x5");
  COMPARE(sys(0x3, 0x7, 0xd, 0x1, x6), "dc cvadp, x6");
  COMPARE(sys(0x3, 0x7, 0x4, 0x1, x0), "dc zva, x0");
  COMPARE(sys(0x0, 0x0, 0x0, 0x0, x0), "sys #0, C0, C0, #0, x0");
  COMPARE(sys(0x1, 0x2, 0x5, 0x2, x5), "sys #1, C2, C5, #2, x5");
  COMPARE(sys(0x2, 0x8, 0xa, 0x3, x6), "sys #2, C8, C10, #3, x6");
  COMPARE(sys(0x2, 0xf, 0xf, 0x1, xzr), "sys #2, C15, C15, #1");
  COMPARE(sys(0x2, 0xf, 0xf, 0x1), "sys #2, C15, C15, #1");

  CLEANUP();
}


TEST(system_ic) {
  SETUP();

  COMPARE(ic(IVAU, x0), "ic ivau, x0");
  COMPARE(ic(IVAU, x1), "ic ivau, x1");
  COMPARE(ic(IVAU, xzr), "ic ivau, xzr");

  CLEANUP();
}


TEST(system_dc) {
  SETUP();

  COMPARE(dc(CVAC, x2), "dc cvac, x2");
  COMPARE(dc(CVAU, x3), "dc cvau, x3");
  COMPARE(dc(CVAP, x4), "dc cvap, x4");
  COMPARE(dc(CIVAC, x5), "dc civac, x5");
  COMPARE(dc(CVADP, x6), "dc cvadp, x6");
  COMPARE(dc(ZVA, x0), "dc zva, x0");
  COMPARE(dc(ZVA, xzr), "dc zva, xzr");

  CLEANUP();
}


TEST(system_nop) {
  SETUP();

  COMPARE(nop(), "nop");
  COMPARE_MACRO(Nop(), "nop");

  CLEANUP();
}


TEST(system_pauth) {
  SETUP();

  COMPARE(pacia1716(), "pacia1716");
  COMPARE(pacib1716(), "pacib1716");
  COMPARE(paciaz(), "paciaz");
  COMPARE(pacibz(), "pacibz");
  COMPARE(paciasp(), "paciasp");
  COMPARE(pacibsp(), "pacibsp");
  COMPARE(autia1716(), "autia1716");
  COMPARE(autib1716(), "autib1716");
  COMPARE(autiaz(), "autiaz");
  COMPARE(autibz(), "autibz");
  COMPARE(autiasp(), "autiasp");
  COMPARE(autibsp(), "autibsp");
  COMPARE(xpaclri(), "xpaclri");

  CLEANUP();
}


TEST(unreachable) {
  SETUP();

  VIXL_ASSERT(kUnreachableOpcode == 0xdeb0);
#ifdef VIXL_INCLUDE_SIMULATOR_AARCH64
  COMPARE_MACRO(Unreachable(), "hlt #0xdeb0");
#else
  COMPARE_MACRO(Unreachable(), "udf #0xdeb0");
#endif

  CLEANUP();
}


#ifdef VIXL_INCLUDE_SIMULATOR_AARCH64
TEST(trace) {
  SETUP();

  VIXL_ASSERT(kTraceOpcode == 0xdeb2);

  // All Trace calls should produce the same instruction.
  COMPARE_MACRO_PREFIX(Trace(LOG_ALL, TRACE_ENABLE), "hlt #0xdeb2");
  COMPARE_MACRO_PREFIX(Trace(LOG_REGS, TRACE_DISABLE), "hlt #0xdeb2");

  CLEANUP();
}
#endif


#ifdef VIXL_INCLUDE_SIMULATOR_AARCH64
TEST(log) {
  SETUP();

  VIXL_ASSERT(kLogOpcode == 0xdeb3);

  // All Log calls should produce the same instruction.
  COMPARE_MACRO_PREFIX(Log(LOG_ALL), "hlt #0xdeb3");
  COMPARE_MACRO_PREFIX(Log(LOG_SYSREGS), "hlt #0xdeb3");

  CLEANUP();
}
#endif


TEST(hlt) {
  SETUP();

  COMPARE(hlt(0), "hlt #0x0");
  COMPARE(hlt(1), "hlt #0x1");
  COMPARE(hlt(65535), "hlt #0xffff");

  CLEANUP();
}


TEST(brk) {
  SETUP();

  COMPARE(brk(0), "brk #0x0");
  COMPARE(brk(1), "brk #0x1");
  COMPARE(brk(65535), "brk #0xffff");

  CLEANUP();
}


TEST(svc) {
  SETUP();

  COMPARE(svc(0), "svc #0x0");
  COMPARE(svc(1), "svc #0x1");
  COMPARE(svc(65535), "svc #0xffff");

  CLEANUP();
}


TEST(add_sub_negative) {
  SETUP();

  COMPARE_MACRO(Add(x10, x0, -42), "sub x10, x0, #0x2a (42)");
  COMPARE_MACRO(Add(x11, x1, -687), "sub x11, x1, #0x2af (687)");
  COMPARE_MACRO(Add(x12, x2, -0x88), "sub x12, x2, #0x88 (136)");

  COMPARE_MACRO(Sub(x13, x0, -600), "add x13, x0, #0x258 (600)");
  COMPARE_MACRO(Sub(x14, x1, -313), "add x14, x1, #0x139 (313)");
  COMPARE_MACRO(Sub(x15, x2, -0x555), "add x15, x2, #0x555 (1365)");

  COMPARE_MACRO(Add(w19, w3, -0x344), "sub w19, w3, #0x344 (836)");
  COMPARE_MACRO(Add(w20, w4, -2000), "sub w20, w4, #0x7d0 (2000)");

  COMPARE_MACRO(Add(w0, w1, 5, LeaveFlags), "add w0, w1, #0x5 (5)");
  COMPARE_MACRO(Add(w1, w2, 15, SetFlags), "adds w1, w2, #0xf (15)");

  COMPARE_MACRO(Sub(w0, w1, 5, LeaveFlags), "sub w0, w1, #0x5 (5)");
  COMPARE_MACRO(Sub(w1, w2, 15, SetFlags), "subs w1, w2, #0xf (15)");

  COMPARE_MACRO(Sub(w21, w3, -0xbc), "add w21, w3, #0xbc (188)");
  COMPARE_MACRO(Sub(w22, w4, -2000), "add w22, w4, #0x7d0 (2000)");

  COMPARE_MACRO(Cmp(w0, -1), "cmn w0, #0x1 (1)");
  COMPARE_MACRO(Cmp(x1, -1), "cmn x1, #0x1 (1)");
  COMPARE_MACRO(Cmp(w2, -4095), "cmn w2, #0xfff (4095)");
  COMPARE_MACRO(Cmp(x3, -4095), "cmn x3, #0xfff (4095)");

  COMPARE_MACRO(Cmn(w0, -1), "cmp w0, #0x1 (1)");
  COMPARE_MACRO(Cmn(x1, -1), "cmp x1, #0x1 (1)");
  COMPARE_MACRO(Cmn(w2, -4095), "cmp w2, #0xfff (4095)");
  COMPARE_MACRO(Cmn(x3, -4095), "cmp x3, #0xfff (4095)");

  CLEANUP();
}

TEST(add_sub_macro) {
  SETUP();

  // Add and Sub use their destination register as a scratch if they can.
  COMPARE_MACRO(Add(x0, x1, 0x4242),
                "mov x0, #0x4242\n"
                "add x0, x1, x0");
  COMPARE_MACRO(Add(x0, x0, 0x4242),
                "mov x16, #0x4242\n"
                "add x0, x0, x16");
  COMPARE_MACRO(Adds(x0, xzr, Operand(w1, SXTW)),
                "sxtw x0, w1\n"
                "adds x0, xzr, x0");
  COMPARE_MACRO(Sub(x0, x1, 0x4242),
                "mov x0, #0x4242\n"
                "sub x0, x1, x0");
  COMPARE_MACRO(Sub(x0, x0, 0x4242),
                "mov x16, #0x4242\n"
                "sub x0, x0, x16");
  COMPARE_MACRO(Subs(x0, xzr, Operand(w1, SXTW)),
                "sxtw x0, w1\n"
                "negs x0, x0");
}

TEST(adc_sbc_macro) {
  SETUP();

  // Adc and Sbc use their destination register as a scratch if they can.
  COMPARE_MACRO(Adc(x0, x1, 0x4242),
                "mov x0, #0x4242\n"
                "adc x0, x1, x0");
  COMPARE_MACRO(Adc(x0, x0, 0x4242),
                "mov x16, #0x4242\n"
                "adc x0, x0, x16");
  COMPARE_MACRO(Adcs(x0, xzr, Operand(w1, SXTW)),
                "sxtw x0, w1\n"
                "adcs x0, xzr, x0");
  COMPARE_MACRO(Sbc(x0, x1, 0x4242),
                "mov x0, #0x4242\n"
                "sbc x0, x1, x0");
  COMPARE_MACRO(Sbc(x0, x0, 0x4242),
                "mov x16, #0x4242\n"
                "sbc x0, x0, x16");
  COMPARE_MACRO(Sbcs(x0, xzr, Operand(w1, SXTW)),
                "sxtw x0, w1\n"
                "ngcs x0, x0");
}

TEST(logical_immediate_move) {
  SETUP();

  COMPARE_MACRO(And(w0, w1, 0), "mov w0, #0x0");
  COMPARE_MACRO(And(x0, x1, 0), "mov x0, #0x0");
  COMPARE_MACRO(Orr(w2, w3, 0), "mov w2, w3");
  COMPARE_MACRO(Orr(x2, x3, 0), "mov x2, x3");
  COMPARE_MACRO(Eor(w4, w5, 0), "mov w4, w5");
  COMPARE_MACRO(Eor(x4, x5, 0), "mov x4, x5");
  COMPARE_MACRO(Bic(w6, w7, 0), "mov w6, w7");
  COMPARE_MACRO(Bic(x6, x7, 0), "mov x6, x7");
  COMPARE_MACRO(Orn(w8, w9, 0), "mov w8, #0xffffffff");
  COMPARE_MACRO(Orn(x8, x9, 0), "mov x8, #0xffffffffffffffff");
  COMPARE_MACRO(Eon(w10, w11, 0), "mvn w10, w11");
  COMPARE_MACRO(Eon(x10, x11, 0), "mvn x10, x11");

  COMPARE_MACRO(And(w12, w13, 0xffffffff), "mov w12, w13");
  COMPARE_MACRO(And(x12, x13, 0xffffffff), "and x12, x13, #0xffffffff");
  COMPARE_MACRO(And(x12, x13, 0xffffffffffffffff), "mov x12, x13");
  COMPARE_MACRO(Orr(w14, w15, 0xffffffff), "mov w14, #0xffffffff");
  COMPARE_MACRO(Orr(x14, x15, 0xffffffff), "orr x14, x15, #0xffffffff");
  COMPARE_MACRO(Orr(x14, x15, 0xffffffffffffffff),
                "mov x14, #0xffffffffffffffff");
  COMPARE_MACRO(Eor(w16, w17, 0xffffffff), "mvn w16, w17");
  COMPARE_MACRO(Eor(x16, x17, 0xffffffff), "eor x16, x17, #0xffffffff");
  COMPARE_MACRO(Eor(x16, x17, 0xffffffffffffffff), "mvn x16, x17");
  COMPARE_MACRO(Bic(w18, w19, 0xffffffff), "mov w18, #0x0");
  COMPARE_MACRO(Bic(x18, x19, 0xffffffff), "and x18, x19, #0xffffffff00000000");
  COMPARE_MACRO(Bic(x18, x19, 0xffffffffffffffff), "mov x18, #0x0");
  COMPARE_MACRO(Orn(w20, w21, 0xffffffff), "mov w20, w21");
  COMPARE_MACRO(Orn(x20, x21, 0xffffffff), "orr x20, x21, #0xffffffff00000000");
  COMPARE_MACRO(Orn(x20, x21, 0xffffffffffffffff), "mov x20, x21");
  COMPARE_MACRO(Eon(w22, w23, 0xffffffff), "mov w22, w23");
  COMPARE_MACRO(Eon(x22, x23, 0xffffffff), "eor x22, x23, #0xffffffff00000000");
  COMPARE_MACRO(Eon(x22, x23, 0xffffffffffffffff), "mov x22, x23");

  // Test stack pointer with non encodable immediate.
  COMPARE_MACRO(Orr(wsp, w5, 0x1234),
                "mov w16, #0x1234\n"
                "orr w16, w5, w16\n"
                "mov wsp, w16");
  COMPARE_MACRO(Orr(sp, x15, 0x123),
                "mov x16, #0x123\n"
                "orr x16, x15, x16\n"
                "mov sp, x16");

  CLEANUP();
}

TEST(logical_macro) {
  SETUP();

  // LogicalMacro uses the destination register as a scratch if it can.
  COMPARE_MACRO(And(x0, x1, 0x4242),
                "mov x0, #0x4242\n"
                "and x0, x1, x0");
  COMPARE_MACRO(Bic(x0, x0, 0x4242),
                "mov x16, #0xffffffffffffbdbd\n"
                "and x0, x0, x16");
  COMPARE_MACRO(Orn(x0, xzr, Operand(w1, SXTW)),
                "sxtw x0, w1\n"
                "mvn x0, x0");
  COMPARE_MACRO(Orr(x0, x1, 0x4242),
                "mov x0, #0x4242\n"
                "orr x0, x1, x0");
  COMPARE_MACRO(Ands(x0, x0, 0x4242),
                "mov x16, #0x4242\n"
                "ands x0, x0, x16");
  COMPARE_MACRO(Tst(xzr, Operand(w1, SXTW)),
                "sxtw x16, w1\n"
                "tst xzr, x16");
}

TEST(barriers) {
  SETUP();

  // DMB
  COMPARE_MACRO(Dmb(FullSystem, BarrierAll), "dmb sy");
  COMPARE_MACRO(Dmb(FullSystem, BarrierReads), "dmb ld");
  COMPARE_MACRO(Dmb(FullSystem, BarrierWrites), "dmb st");

  COMPARE_MACRO(Dmb(InnerShareable, BarrierAll), "dmb ish");
  COMPARE_MACRO(Dmb(InnerShareable, BarrierReads), "dmb ishld");
  COMPARE_MACRO(Dmb(InnerShareable, BarrierWrites), "dmb ishst");

  COMPARE_MACRO(Dmb(NonShareable, BarrierAll), "dmb nsh");
  COMPARE_MACRO(Dmb(NonShareable, BarrierReads), "dmb nshld");
  COMPARE_MACRO(Dmb(NonShareable, BarrierWrites), "dmb nshst");

  COMPARE_MACRO(Dmb(OuterShareable, BarrierAll), "dmb osh");
  COMPARE_MACRO(Dmb(OuterShareable, BarrierReads), "dmb oshld");
  COMPARE_MACRO(Dmb(OuterShareable, BarrierWrites), "dmb oshst");

  COMPARE_MACRO(Dmb(FullSystem, BarrierOther), "dmb sy (0b1100)");
  COMPARE_MACRO(Dmb(InnerShareable, BarrierOther), "dmb sy (0b1000)");
  COMPARE_MACRO(Dmb(NonShareable, BarrierOther), "dmb sy (0b0100)");
  COMPARE_MACRO(Dmb(OuterShareable, BarrierOther), "dmb sy (0b0000)");

  // DSB
  COMPARE_MACRO(Dsb(FullSystem, BarrierAll), "dsb sy");
  COMPARE_MACRO(Dsb(FullSystem, BarrierReads), "dsb ld");
  COMPARE_MACRO(Dsb(FullSystem, BarrierWrites), "dsb st");

  COMPARE_MACRO(Dsb(InnerShareable, BarrierAll), "dsb ish");
  COMPARE_MACRO(Dsb(InnerShareable, BarrierReads), "dsb ishld");
  COMPARE_MACRO(Dsb(InnerShareable, BarrierWrites), "dsb ishst");

  COMPARE_MACRO(Dsb(NonShareable, BarrierAll), "dsb nsh");
  COMPARE_MACRO(Dsb(NonShareable, BarrierReads), "dsb nshld");
  COMPARE_MACRO(Dsb(NonShareable, BarrierWrites), "dsb nshst");

  COMPARE_MACRO(Dsb(OuterShareable, BarrierAll), "dsb osh");
  COMPARE_MACRO(Dsb(OuterShareable, BarrierReads), "dsb oshld");
  COMPARE_MACRO(Dsb(OuterShareable, BarrierWrites), "dsb oshst");

  COMPARE_MACRO(Dsb(FullSystem, BarrierOther), "dsb sy (0b1100)");
  COMPARE_MACRO(Dsb(InnerShareable, BarrierOther), "dsb sy (0b1000)");
  COMPARE_MACRO(Dsb(NonShareable, BarrierOther), "dsb sy (0b0100)");
  COMPARE_MACRO(Dsb(OuterShareable, BarrierOther), "dsb sy (0b0000)");

  // ISB
  COMPARE_MACRO(Isb(), "isb");

  // ESB
  COMPARE_MACRO(Esb(), "esb");

  // CSDB
  COMPARE_MACRO(Csdb(), "csdb");

  CLEANUP();
}

TEST(address_map) {
  // Check that we can disassemble from a fake base address.
  SETUP();

  disasm.MapCodeAddress(0, masm.GetBuffer()->GetStartAddress<Instruction*>());
  COMPARE(ldr(x0, INT64_C(0)), "ldr x0, pc+0 (addr 0x0)");
  COMPARE(ldr(x0, -1), "ldr x0, pc-4 (addr -0x4)");
  COMPARE(ldr(x0, 1), "ldr x0, pc+4 (addr 0x4)");
  COMPARE(prfm(PLIL1KEEP, INT64_C(0)), "prfm plil1keep, pc+0 (addr 0x0)");
  COMPARE(prfm(PLIL1KEEP, -1), "prfm plil1keep, pc-4 (addr -0x4)");
  COMPARE(prfm(PLIL1KEEP, 1), "prfm plil1keep, pc+4 (addr 0x4)");
  COMPARE(adr(x0, INT64_C(0)), "adr x0, #+0x0 (addr 0x0)");
  COMPARE(adr(x0, -1), "adr x0, #-0x1 (addr -0x1)");
  COMPARE(adr(x0, 1), "adr x0, #+0x1 (addr 0x1)");
  COMPARE(adrp(x0, INT64_C(0)), "adrp x0, #+0x0 (addr 0x0)");
  COMPARE(adrp(x0, -1), "adrp x0, #-0x1000 (addr -0x1000)");
  COMPARE(adrp(x0, 1), "adrp x0, #+0x1000 (addr 0x1000)");
  COMPARE(b(INT64_C(0)), "b #+0x0 (addr 0x0)");
  COMPARE(b(-1), "b #-0x4 (addr -0x4)");
  COMPARE(b(1), "b #+0x4 (addr 0x4)");

  disasm.MapCodeAddress(0x1234,
                        masm.GetBuffer()->GetStartAddress<Instruction*>());
  COMPARE(ldr(x0, INT64_C(0)), "ldr x0, pc+0 (addr 0x1234)");
  COMPARE(ldr(x0, -1), "ldr x0, pc-4 (addr 0x1230)");
  COMPARE(ldr(x0, 1), "ldr x0, pc+4 (addr 0x1238)");
  COMPARE(prfm(PLIL1KEEP, INT64_C(0)), "prfm plil1keep, pc+0 (addr 0x1234)");
  COMPARE(prfm(PLIL1KEEP, -1), "prfm plil1keep, pc-4 (addr 0x1230)");
  COMPARE(prfm(PLIL1KEEP, 1), "prfm plil1keep, pc+4 (addr 0x1238)");
  COMPARE(adr(x0, INT64_C(0)), "adr x0, #+0x0 (addr 0x1234)");
  COMPARE(adr(x0, -1), "adr x0, #-0x1 (addr 0x1233)");
  COMPARE(adr(x0, 1), "adr x0, #+0x1 (addr 0x1235)");
  COMPARE(adrp(x0, INT64_C(0)), "adrp x0, #+0x0 (addr 0x1000)");
  COMPARE(adrp(x0, -1), "adrp x0, #-0x1000 (addr 0x0)");
  COMPARE(adrp(x0, 1), "adrp x0, #+0x1000 (addr 0x2000)");
  COMPARE(b(INT64_C(0)), "b #+0x0 (addr 0x1234)");
  COMPARE(b(-1), "b #-0x4 (addr 0x1230)");
  COMPARE(b(1), "b #+0x4 (addr 0x1238)");

  // Check that 64-bit addresses work.
  disasm.MapCodeAddress(UINT64_C(0x100000000),
                        masm.GetBuffer()->GetStartAddress<Instruction*>());
  COMPARE(ldr(x0, INT64_C(0)), "ldr x0, pc+0 (addr 0x100000000)");
  COMPARE(ldr(x0, -1), "ldr x0, pc-4 (addr 0xfffffffc)");
  COMPARE(ldr(x0, 1), "ldr x0, pc+4 (addr 0x100000004)");
  COMPARE(prfm(PLIL1KEEP, INT64_C(0)),
          "prfm plil1keep, pc+0 (addr 0x100000000)");
  COMPARE(prfm(PLIL1KEEP, -1), "prfm plil1keep, pc-4 (addr 0xfffffffc)");
  COMPARE(prfm(PLIL1KEEP, 1), "prfm plil1keep, pc+4 (addr 0x100000004)");
  COMPARE(adr(x0, INT64_C(0)), "adr x0, #+0x0 (addr 0x100000000)");
  COMPARE(adr(x0, -1), "adr x0, #-0x1 (addr 0xffffffff)");
  COMPARE(adr(x0, 1), "adr x0, #+0x1 (addr 0x100000001)");
  COMPARE(adrp(x0, INT64_C(0)), "adrp x0, #+0x0 (addr 0x100000000)");
  COMPARE(adrp(x0, -1), "adrp x0, #-0x1000 (addr 0xfffff000)");
  COMPARE(adrp(x0, 1), "adrp x0, #+0x1000 (addr 0x100001000)");
  COMPARE(b(INT64_C(0)), "b #+0x0 (addr 0x100000000)");
  COMPARE(b(-1), "b #-0x4 (addr 0xfffffffc)");
  COMPARE(b(1), "b #+0x4 (addr 0x100000004)");

  disasm.MapCodeAddress(0xfffffffc,
                        masm.GetBuffer()->GetStartAddress<Instruction*>());
  COMPARE(ldr(x0, 1), "ldr x0, pc+4 (addr 0x100000000)");
  COMPARE(prfm(PLIL1KEEP, 1), "prfm plil1keep, pc+4 (addr 0x100000000)");
  COMPARE(b(1), "b #+0x4 (addr 0x100000000)");
  COMPARE(adr(x0, 4), "adr x0, #+0x4 (addr 0x100000000)");
  COMPARE(adrp(x0, 1), "adrp x0, #+0x1000 (addr 0x100000000)");

  // Check that very large offsets are handled properly. This detects misuse of
  // the host's ptrdiff_t type when run on a 32-bit host. Only adrp is capable
  // of encoding such offsets.
  disasm.MapCodeAddress(0, masm.GetBuffer()->GetStartAddress<Instruction*>());
  COMPARE(adrp(x0, 0x000fffff), "adrp x0, #+0xfffff000 (addr 0xfffff000)");
  COMPARE(adrp(x0, -0x00100000), "adrp x0, #-0x100000000 (addr -0x100000000)");

  CLEANUP();
}

TEST(hint) {
  SETUP();

  // Test that we properly disassemble named and unnamed hints.
  COMPARE(hint(NOP), "nop");
  COMPARE(hint(YIELD), "yield");
  COMPARE(hint(WFE), "wfe");
  COMPARE(hint(WFI), "wfi");
  COMPARE(hint(SEV), "sev");
  COMPARE(hint(SEVL), "sevl");
  COMPARE(hint(6), "hint #6");
  COMPARE(hint(ESB), "esb");
  COMPARE(hint(CSDB), "csdb");
  COMPARE(hint(42), "hint #42");
  COMPARE(hint(127), "hint #127");

  // The MacroAssembler should simply pass through to the Assembler.
  COMPARE_MACRO(Hint(NOP), "nop");
  COMPARE_MACRO(Hint(CSDB), "csdb");
  COMPARE_MACRO(Hint(42), "hint #42");
  COMPARE_MACRO(Hint(127), "hint #127");

  CLEANUP();
}

TEST(bti) {
  SETUP();

  COMPARE(bti(EmitBTI), "bti");
  COMPARE(bti(EmitBTI_c), "bti c");
  COMPARE(bti(EmitBTI_j), "bti j");
  COMPARE(bti(EmitBTI_jc), "bti jc");
  COMPARE(hint(BTI), "bti");
  COMPARE(hint(BTI_c), "bti c");
  COMPARE(hint(BTI_j), "bti j");
  COMPARE(hint(BTI_jc), "bti jc");

  Label placeholder1, placeholder2, placeholder3, placeholder4;
  COMPARE_MACRO(Bind(&placeholder1, EmitBTI), "bti");
  COMPARE_MACRO(Bind(&placeholder2, EmitBTI_c), "bti c");
  COMPARE_MACRO(Bind(&placeholder3, EmitBTI_j), "bti j");
  COMPARE_MACRO(Bind(&placeholder4, EmitBTI_jc), "bti jc");

  CLEANUP();
}

TEST(udf) {
  SETUP();

  COMPARE(udf(0), "udf #0x0");
  COMPARE(udf(0x1234), "udf #0x1234");
  COMPARE(udf(0xffff), "udf #0xffff");

  // UDF gives the useful property that zero-initialised memory is guaranteed to
  // generate undefined instruction exceptions.
  COMPARE(dc(0), "udf #0x0");

  // Check related unallocated bit patterns from the reserved block.
  COMPARE(dc(0x00010000), "unallocated (Unallocated)");
  COMPARE(dc(0x01000000), "unallocated (Unallocated)");
  COMPARE(dc(0x20000000), "unallocated (Unallocated)");
  COMPARE(dc(0x80000000), "unallocated (Unallocated)");

  CLEANUP();
}

}  // namespace aarch64
}  // namespace vixl
