% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/groupedSimes.R
\name{groupedSimes}
\alias{groupedSimes}
\title{Combine grouped p-values with Simes' method}
\usage{
groupedSimes(p.values, grouping, weights = NULL, log.p = FALSE)
}
\arguments{
\item{p.values}{A numeric vector containing p-values for individual tests.}

\item{grouping}{A vector or factor of length equal to \code{p.values}, specifying the group to which each test is assigned.

Alternatively, an \link{rle} object where each run corresponds to a group and specifies the entries of \code{p.values} belonging to that group.
This assumes that \code{p.values} is ordered such that all entries in the same group are adjacent to each other.}

\item{weights}{A numeric vector of length equal to \code{p.values}, containing a positive weight for each test.
Alternatively \code{NULL}, in which case equal weights are assigned to all tests.}

\item{log.p}{Logical scalar indicating whether the p-values in \code{p.values} are log-transformed.}
}
\value{
A list containing:
\itemize{
\item \code{p.value}, a named numeric vector of length equal to the number of unique levels in \code{grouping}.
This contains the combined Simes p-value for each group, log-transformed if \code{log.p=TRUE}.
Each entry is named according to the group.
\item \code{representative}, a named integer scalar specifying the index of representative test for each group.
Each index refers to an entry of \code{p.values} and is named according to its group.
\item \code{influential}, a logical vector of length equal to \code{p.values}.
Entries are \code{TRUE} for any p-value that is deemed \dQuote{influential} to the final combined p-value for its group.
}
}
\description{
Combine p-values from grouped tests with Simes' method.
Groups are defined according to unique levels of a grouping factor.
}
\details{
The joint null hypothesis for each group is that all of the individual null hypotheses are true.
Simes' method will reject the joint null if any of the individual nulls are rejected, providing weak control of the family-wise error rate.

In theory, the method is only applicable to independent tests, but experience suggests that it is quite robust to dependencies.
The calculation itself is very closely related to the Benjamini-Hochberg method for controlling the false discovery rate.
One can actually obtain Simes' combined p-value by taking the smallest BH-adjusted p-value across a group.

If non-equal weights are provided, they are treated as relative frequency weights.
That is, if one p-value is given a weight of 10 and another p-value is given a weight of 1, 
the former is considered to occur 10 times more frequently than the latter.

The representative test for each group is defined as the test with the p-value that is ultimately used as the combined p-value.
Briefly, one can identify this test as that with the smallest BH-adjusted p-value if the monotonicity adjustment were omitted.
The influential tests for each group are defined as those with p-values no greater than the representative test's p-value.
This is based on the fact that increasing them (e.g., by setting them to unity) would result in a larger combined p-value.
}
\examples{
p1 <- rbeta(100, 0.8, 1)
g <- sample(10, length(p1), replace=TRUE)

# Standard application:
out <- groupedSimes(p1, g)
str(out)

# With weights:
out <- groupedSimes(p1, g, weights=rexp(length(p1)))
str(out)

# With log p-values. 
out <- groupedSimes(log(p1), g, log.p=TRUE)
str(out)

}
\references{
Simes RJ (1986).
An improved Bonferroni procedure for multiple tests of significance.
\emph{Biometrika} 73:751-754.

Sarkar SK and Chung CK (1997).
The Simes method for multiple hypothesis testing with positively dependent test statistics.
\emph{J. Am. Stat. Assoc.} 92, 1601-1608.

Benjamini Y and Hochberg Y (1997).
Multiple hypotheses testing with weights.
\emph{Scand. J. Stat.} 24, 407-418.
}
\seealso{
\code{\link{parallelSimes}}, for a version that operates on parallel vectors of p-values.
}
\author{
Aaron Lun
}
