//===----------------------------------------------------------------------===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//

// <string>

// int compare(size_type pos1, size_type n1, const basic_string& str,
//             size_type pos2, size_type n2=npos) const;
//  the "=npos" was added in C++14

// When back-deploying to macosx10.7, the RTTI for exception classes
// incorrectly provided by libc++.dylib is mixed with the one in
// libc++abi.dylib and exceptions are not caught properly.
// XFAIL: with_system_cxx_lib=macosx10.7

#include <string>
#include <stdexcept>
#include <cassert>

#include "min_allocator.h"

#include "test_macros.h"

int sign(int x)
{
    if (x == 0)
        return 0;
    if (x < 0)
        return -1;
    return 1;
}

template <class S>
void
test(const S& s,   typename S::size_type pos1, typename S::size_type n1,
     const S& str, typename S::size_type pos2, typename S::size_type n2, int x)
{
    if (pos1 <= s.size() && pos2 <= str.size())
        assert(sign(s.compare(pos1, n1, str, pos2, n2)) == sign(x));
#ifndef TEST_HAS_NO_EXCEPTIONS
    else
    {
        try
        {
            TEST_IGNORE_NODISCARD s.compare(pos1, n1, str, pos2, n2);
            assert(false);
        }
        catch (const std::out_of_range&)
        {
            assert(pos1 > s.size() || pos2 > str.size());
        }
    }
#endif
}

template <class S>
void
test_npos(const S& s,   typename S::size_type pos1, typename S::size_type n1,
          const S& str, typename S::size_type pos2, int x)
{
    if (pos1 <= s.size() && pos2 <= str.size())
        assert(sign(s.compare(pos1, n1, str, pos2)) == sign(x));
#ifndef TEST_HAS_NO_EXCEPTIONS
    else
    {
        try
        {
            TEST_IGNORE_NODISCARD s.compare(pos1, n1, str, pos2);
            assert(false);
        }
        catch (const std::out_of_range&)
        {
            assert(pos1 > s.size() || pos2 > str.size());
        }
    }
#endif
}

template <class S>
void test0()
{
    test(S(""), 0, 0, S(""), 0, 0, 0);
    test(S(""), 0, 0, S(""), 0, 1, 0);
    test(S(""), 0, 0, S(""), 1, 0, 0);
    test(S(""), 0, 0, S("abcde"), 0, 0, 0);
    test(S(""), 0, 0, S("abcde"), 0, 1, -1);
    test(S(""), 0, 0, S("abcde"), 0, 2, -2);
    test(S(""), 0, 0, S("abcde"), 0, 4, -4);
    test(S(""), 0, 0, S("abcde"), 0, 5, -5);
    test(S(""), 0, 0, S("abcde"), 0, 6, -5);
    test(S(""), 0, 0, S("abcde"), 1, 0, 0);
    test(S(""), 0, 0, S("abcde"), 1, 1, -1);
    test(S(""), 0, 0, S("abcde"), 1, 2, -2);
    test(S(""), 0, 0, S("abcde"), 1, 3, -3);
    test(S(""), 0, 0, S("abcde"), 1, 4, -4);
    test(S(""), 0, 0, S("abcde"), 1, 5, -4);
    test(S(""), 0, 0, S("abcde"), 2, 0, 0);
    test(S(""), 0, 0, S("abcde"), 2, 1, -1);
    test(S(""), 0, 0, S("abcde"), 2, 2, -2);
    test(S(""), 0, 0, S("abcde"), 2, 3, -3);
    test(S(""), 0, 0, S("abcde"), 2, 4, -3);
    test(S(""), 0, 0, S("abcde"), 4, 0, 0);
    test(S(""), 0, 0, S("abcde"), 4, 1, -1);
    test(S(""), 0, 0, S("abcde"), 4, 2, -1);
    test(S(""), 0, 0, S("abcde"), 5, 0, 0);
    test(S(""), 0, 0, S("abcde"), 5, 1, 0);
    test(S(""), 0, 0, S("abcde"), 6, 0, 0);
    test(S(""), 0, 0, S("abcdefghij"), 0, 0, 0);
    test(S(""), 0, 0, S("abcdefghij"), 0, 1, -1);
    test(S(""), 0, 0, S("abcdefghij"), 0, 5, -5);
    test(S(""), 0, 0, S("abcdefghij"), 0, 9, -9);
    test(S(""), 0, 0, S("abcdefghij"), 0, 10, -10);
    test(S(""), 0, 0, S("abcdefghij"), 0, 11, -10);
    test(S(""), 0, 0, S("abcdefghij"), 1, 0, 0);
    test(S(""), 0, 0, S("abcdefghij"), 1, 1, -1);
    test(S(""), 0, 0, S("abcdefghij"), 1, 4, -4);
    test(S(""), 0, 0, S("abcdefghij"), 1, 8, -8);
    test(S(""), 0, 0, S("abcdefghij"), 1, 9, -9);
    test(S(""), 0, 0, S("abcdefghij"), 1, 10, -9);
    test(S(""), 0, 0, S("abcdefghij"), 5, 0, 0);
    test(S(""), 0, 0, S("abcdefghij"), 5, 1, -1);
    test(S(""), 0, 0, S("abcdefghij"), 5, 2, -2);
    test(S(""), 0, 0, S("abcdefghij"), 5, 4, -4);
    test(S(""), 0, 0, S("abcdefghij"), 5, 5, -5);
    test(S(""), 0, 0, S("abcdefghij"), 5, 6, -5);
    test(S(""), 0, 0, S("abcdefghij"), 9, 0, 0);
    test(S(""), 0, 0, S("abcdefghij"), 9, 1, -1);
    test(S(""), 0, 0, S("abcdefghij"), 9, 2, -1);
    test(S(""), 0, 0, S("abcdefghij"), 10, 0, 0);
    test(S(""), 0, 0, S("abcdefghij"), 10, 1, 0);
    test(S(""), 0, 0, S("abcdefghij"), 11, 0, 0);
    test(S(""), 0, 0, S("abcdefghijklmnopqrst"), 0, 0, 0);
    test(S(""), 0, 0, S("abcdefghijklmnopqrst"), 0, 1, -1);
    test(S(""), 0, 0, S("abcdefghijklmnopqrst"), 0, 10, -10);
    test(S(""), 0, 0, S("abcdefghijklmnopqrst"), 0, 19, -19);
    test(S(""), 0, 0, S("abcdefghijklmnopqrst"), 0, 20, -20);
    test(S(""), 0, 0, S("abcdefghijklmnopqrst"), 0, 21, -20);
    test(S(""), 0, 0, S("abcdefghijklmnopqrst"), 1, 0, 0);
    test(S(""), 0, 0, S("abcdefghijklmnopqrst"), 1, 1, -1);
    test(S(""), 0, 0, S("abcdefghijklmnopqrst"), 1, 9, -9);
    test(S(""), 0, 0, S("abcdefghijklmnopqrst"), 1, 18, -18);
    test(S(""), 0, 0, S("abcdefghijklmnopqrst"), 1, 19, -19);
    test(S(""), 0, 0, S("abcdefghijklmnopqrst"), 1, 20, -19);
    test(S(""), 0, 0, S("abcdefghijklmnopqrst"), 10, 0, 0);
    test(S(""), 0, 0, S("abcdefghijklmnopqrst"), 10, 1, -1);
    test(S(""), 0, 0, S("abcdefghijklmnopqrst"), 10, 5, -5);
    test(S(""), 0, 0, S("abcdefghijklmnopqrst"), 10, 9, -9);
    test(S(""), 0, 0, S("abcdefghijklmnopqrst"), 10, 10, -10);
    test(S(""), 0, 0, S("abcdefghijklmnopqrst"), 10, 11, -10);
    test(S(""), 0, 0, S("abcdefghijklmnopqrst"), 19, 0, 0);
    test(S(""), 0, 0, S("abcdefghijklmnopqrst"), 19, 1, -1);
    test(S(""), 0, 0, S("abcdefghijklmnopqrst"), 19, 2, -1);
    test(S(""), 0, 0, S("abcdefghijklmnopqrst"), 20, 0, 0);
    test(S(""), 0, 0, S("abcdefghijklmnopqrst"), 20, 1, 0);
    test(S(""), 0, 0, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S(""), 0, 1, S(""), 0, 0, 0);
    test(S(""), 0, 1, S(""), 0, 1, 0);
    test(S(""), 0, 1, S(""), 1, 0, 0);
    test(S(""), 0, 1, S("abcde"), 0, 0, 0);
    test(S(""), 0, 1, S("abcde"), 0, 1, -1);
    test(S(""), 0, 1, S("abcde"), 0, 2, -2);
    test(S(""), 0, 1, S("abcde"), 0, 4, -4);
    test(S(""), 0, 1, S("abcde"), 0, 5, -5);
    test(S(""), 0, 1, S("abcde"), 0, 6, -5);
    test(S(""), 0, 1, S("abcde"), 1, 0, 0);
    test(S(""), 0, 1, S("abcde"), 1, 1, -1);
    test(S(""), 0, 1, S("abcde"), 1, 2, -2);
    test(S(""), 0, 1, S("abcde"), 1, 3, -3);
    test(S(""), 0, 1, S("abcde"), 1, 4, -4);
    test(S(""), 0, 1, S("abcde"), 1, 5, -4);
    test(S(""), 0, 1, S("abcde"), 2, 0, 0);
    test(S(""), 0, 1, S("abcde"), 2, 1, -1);
    test(S(""), 0, 1, S("abcde"), 2, 2, -2);
    test(S(""), 0, 1, S("abcde"), 2, 3, -3);
    test(S(""), 0, 1, S("abcde"), 2, 4, -3);
    test(S(""), 0, 1, S("abcde"), 4, 0, 0);
    test(S(""), 0, 1, S("abcde"), 4, 1, -1);
    test(S(""), 0, 1, S("abcde"), 4, 2, -1);
    test(S(""), 0, 1, S("abcde"), 5, 0, 0);
    test(S(""), 0, 1, S("abcde"), 5, 1, 0);
    test(S(""), 0, 1, S("abcde"), 6, 0, 0);
}

template <class S>
void test1()
{
    test(S(""), 0, 1, S("abcdefghij"), 0, 0, 0);
    test(S(""), 0, 1, S("abcdefghij"), 0, 1, -1);
    test(S(""), 0, 1, S("abcdefghij"), 0, 5, -5);
    test(S(""), 0, 1, S("abcdefghij"), 0, 9, -9);
    test(S(""), 0, 1, S("abcdefghij"), 0, 10, -10);
    test(S(""), 0, 1, S("abcdefghij"), 0, 11, -10);
    test(S(""), 0, 1, S("abcdefghij"), 1, 0, 0);
    test(S(""), 0, 1, S("abcdefghij"), 1, 1, -1);
    test(S(""), 0, 1, S("abcdefghij"), 1, 4, -4);
    test(S(""), 0, 1, S("abcdefghij"), 1, 8, -8);
    test(S(""), 0, 1, S("abcdefghij"), 1, 9, -9);
    test(S(""), 0, 1, S("abcdefghij"), 1, 10, -9);
    test(S(""), 0, 1, S("abcdefghij"), 5, 0, 0);
    test(S(""), 0, 1, S("abcdefghij"), 5, 1, -1);
    test(S(""), 0, 1, S("abcdefghij"), 5, 2, -2);
    test(S(""), 0, 1, S("abcdefghij"), 5, 4, -4);
    test(S(""), 0, 1, S("abcdefghij"), 5, 5, -5);
    test(S(""), 0, 1, S("abcdefghij"), 5, 6, -5);
    test(S(""), 0, 1, S("abcdefghij"), 9, 0, 0);
    test(S(""), 0, 1, S("abcdefghij"), 9, 1, -1);
    test(S(""), 0, 1, S("abcdefghij"), 9, 2, -1);
    test(S(""), 0, 1, S("abcdefghij"), 10, 0, 0);
    test(S(""), 0, 1, S("abcdefghij"), 10, 1, 0);
    test(S(""), 0, 1, S("abcdefghij"), 11, 0, 0);
    test(S(""), 0, 1, S("abcdefghijklmnopqrst"), 0, 0, 0);
    test(S(""), 0, 1, S("abcdefghijklmnopqrst"), 0, 1, -1);
    test(S(""), 0, 1, S("abcdefghijklmnopqrst"), 0, 10, -10);
    test(S(""), 0, 1, S("abcdefghijklmnopqrst"), 0, 19, -19);
    test(S(""), 0, 1, S("abcdefghijklmnopqrst"), 0, 20, -20);
    test(S(""), 0, 1, S("abcdefghijklmnopqrst"), 0, 21, -20);
    test(S(""), 0, 1, S("abcdefghijklmnopqrst"), 1, 0, 0);
    test(S(""), 0, 1, S("abcdefghijklmnopqrst"), 1, 1, -1);
    test(S(""), 0, 1, S("abcdefghijklmnopqrst"), 1, 9, -9);
    test(S(""), 0, 1, S("abcdefghijklmnopqrst"), 1, 18, -18);
    test(S(""), 0, 1, S("abcdefghijklmnopqrst"), 1, 19, -19);
    test(S(""), 0, 1, S("abcdefghijklmnopqrst"), 1, 20, -19);
    test(S(""), 0, 1, S("abcdefghijklmnopqrst"), 10, 0, 0);
    test(S(""), 0, 1, S("abcdefghijklmnopqrst"), 10, 1, -1);
    test(S(""), 0, 1, S("abcdefghijklmnopqrst"), 10, 5, -5);
    test(S(""), 0, 1, S("abcdefghijklmnopqrst"), 10, 9, -9);
    test(S(""), 0, 1, S("abcdefghijklmnopqrst"), 10, 10, -10);
    test(S(""), 0, 1, S("abcdefghijklmnopqrst"), 10, 11, -10);
    test(S(""), 0, 1, S("abcdefghijklmnopqrst"), 19, 0, 0);
    test(S(""), 0, 1, S("abcdefghijklmnopqrst"), 19, 1, -1);
    test(S(""), 0, 1, S("abcdefghijklmnopqrst"), 19, 2, -1);
    test(S(""), 0, 1, S("abcdefghijklmnopqrst"), 20, 0, 0);
    test(S(""), 0, 1, S("abcdefghijklmnopqrst"), 20, 1, 0);
    test(S(""), 0, 1, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S(""), 1, 0, S(""), 0, 0, 0);
    test(S(""), 1, 0, S(""), 0, 1, 0);
    test(S(""), 1, 0, S(""), 1, 0, 0);
    test(S(""), 1, 0, S("abcde"), 0, 0, 0);
    test(S(""), 1, 0, S("abcde"), 0, 1, 0);
    test(S(""), 1, 0, S("abcde"), 0, 2, 0);
    test(S(""), 1, 0, S("abcde"), 0, 4, 0);
    test(S(""), 1, 0, S("abcde"), 0, 5, 0);
    test(S(""), 1, 0, S("abcde"), 0, 6, 0);
    test(S(""), 1, 0, S("abcde"), 1, 0, 0);
    test(S(""), 1, 0, S("abcde"), 1, 1, 0);
    test(S(""), 1, 0, S("abcde"), 1, 2, 0);
    test(S(""), 1, 0, S("abcde"), 1, 3, 0);
    test(S(""), 1, 0, S("abcde"), 1, 4, 0);
    test(S(""), 1, 0, S("abcde"), 1, 5, 0);
    test(S(""), 1, 0, S("abcde"), 2, 0, 0);
    test(S(""), 1, 0, S("abcde"), 2, 1, 0);
    test(S(""), 1, 0, S("abcde"), 2, 2, 0);
    test(S(""), 1, 0, S("abcde"), 2, 3, 0);
    test(S(""), 1, 0, S("abcde"), 2, 4, 0);
    test(S(""), 1, 0, S("abcde"), 4, 0, 0);
    test(S(""), 1, 0, S("abcde"), 4, 1, 0);
    test(S(""), 1, 0, S("abcde"), 4, 2, 0);
    test(S(""), 1, 0, S("abcde"), 5, 0, 0);
    test(S(""), 1, 0, S("abcde"), 5, 1, 0);
    test(S(""), 1, 0, S("abcde"), 6, 0, 0);
    test(S(""), 1, 0, S("abcdefghij"), 0, 0, 0);
    test(S(""), 1, 0, S("abcdefghij"), 0, 1, 0);
    test(S(""), 1, 0, S("abcdefghij"), 0, 5, 0);
    test(S(""), 1, 0, S("abcdefghij"), 0, 9, 0);
    test(S(""), 1, 0, S("abcdefghij"), 0, 10, 0);
    test(S(""), 1, 0, S("abcdefghij"), 0, 11, 0);
    test(S(""), 1, 0, S("abcdefghij"), 1, 0, 0);
    test(S(""), 1, 0, S("abcdefghij"), 1, 1, 0);
    test(S(""), 1, 0, S("abcdefghij"), 1, 4, 0);
    test(S(""), 1, 0, S("abcdefghij"), 1, 8, 0);
    test(S(""), 1, 0, S("abcdefghij"), 1, 9, 0);
    test(S(""), 1, 0, S("abcdefghij"), 1, 10, 0);
    test(S(""), 1, 0, S("abcdefghij"), 5, 0, 0);
    test(S(""), 1, 0, S("abcdefghij"), 5, 1, 0);
    test(S(""), 1, 0, S("abcdefghij"), 5, 2, 0);
    test(S(""), 1, 0, S("abcdefghij"), 5, 4, 0);
    test(S(""), 1, 0, S("abcdefghij"), 5, 5, 0);
    test(S(""), 1, 0, S("abcdefghij"), 5, 6, 0);
    test(S(""), 1, 0, S("abcdefghij"), 9, 0, 0);
    test(S(""), 1, 0, S("abcdefghij"), 9, 1, 0);
    test(S(""), 1, 0, S("abcdefghij"), 9, 2, 0);
    test(S(""), 1, 0, S("abcdefghij"), 10, 0, 0);
    test(S(""), 1, 0, S("abcdefghij"), 10, 1, 0);
    test(S(""), 1, 0, S("abcdefghij"), 11, 0, 0);
    test(S(""), 1, 0, S("abcdefghijklmnopqrst"), 0, 0, 0);
    test(S(""), 1, 0, S("abcdefghijklmnopqrst"), 0, 1, 0);
}

template <class S>
void test2()
{
    test(S(""), 1, 0, S("abcdefghijklmnopqrst"), 0, 10, 0);
    test(S(""), 1, 0, S("abcdefghijklmnopqrst"), 0, 19, 0);
    test(S(""), 1, 0, S("abcdefghijklmnopqrst"), 0, 20, 0);
    test(S(""), 1, 0, S("abcdefghijklmnopqrst"), 0, 21, 0);
    test(S(""), 1, 0, S("abcdefghijklmnopqrst"), 1, 0, 0);
    test(S(""), 1, 0, S("abcdefghijklmnopqrst"), 1, 1, 0);
    test(S(""), 1, 0, S("abcdefghijklmnopqrst"), 1, 9, 0);
    test(S(""), 1, 0, S("abcdefghijklmnopqrst"), 1, 18, 0);
    test(S(""), 1, 0, S("abcdefghijklmnopqrst"), 1, 19, 0);
    test(S(""), 1, 0, S("abcdefghijklmnopqrst"), 1, 20, 0);
    test(S(""), 1, 0, S("abcdefghijklmnopqrst"), 10, 0, 0);
    test(S(""), 1, 0, S("abcdefghijklmnopqrst"), 10, 1, 0);
    test(S(""), 1, 0, S("abcdefghijklmnopqrst"), 10, 5, 0);
    test(S(""), 1, 0, S("abcdefghijklmnopqrst"), 10, 9, 0);
    test(S(""), 1, 0, S("abcdefghijklmnopqrst"), 10, 10, 0);
    test(S(""), 1, 0, S("abcdefghijklmnopqrst"), 10, 11, 0);
    test(S(""), 1, 0, S("abcdefghijklmnopqrst"), 19, 0, 0);
    test(S(""), 1, 0, S("abcdefghijklmnopqrst"), 19, 1, 0);
    test(S(""), 1, 0, S("abcdefghijklmnopqrst"), 19, 2, 0);
    test(S(""), 1, 0, S("abcdefghijklmnopqrst"), 20, 0, 0);
    test(S(""), 1, 0, S("abcdefghijklmnopqrst"), 20, 1, 0);
    test(S(""), 1, 0, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcde"), 0, 0, S(""), 0, 0, 0);
    test(S("abcde"), 0, 0, S(""), 0, 1, 0);
    test(S("abcde"), 0, 0, S(""), 1, 0, 0);
    test(S("abcde"), 0, 0, S("abcde"), 0, 0, 0);
    test(S("abcde"), 0, 0, S("abcde"), 0, 1, -1);
    test(S("abcde"), 0, 0, S("abcde"), 0, 2, -2);
    test(S("abcde"), 0, 0, S("abcde"), 0, 4, -4);
    test(S("abcde"), 0, 0, S("abcde"), 0, 5, -5);
    test(S("abcde"), 0, 0, S("abcde"), 0, 6, -5);
    test(S("abcde"), 0, 0, S("abcde"), 1, 0, 0);
    test(S("abcde"), 0, 0, S("abcde"), 1, 1, -1);
    test(S("abcde"), 0, 0, S("abcde"), 1, 2, -2);
    test(S("abcde"), 0, 0, S("abcde"), 1, 3, -3);
    test(S("abcde"), 0, 0, S("abcde"), 1, 4, -4);
    test(S("abcde"), 0, 0, S("abcde"), 1, 5, -4);
    test(S("abcde"), 0, 0, S("abcde"), 2, 0, 0);
    test(S("abcde"), 0, 0, S("abcde"), 2, 1, -1);
    test(S("abcde"), 0, 0, S("abcde"), 2, 2, -2);
    test(S("abcde"), 0, 0, S("abcde"), 2, 3, -3);
    test(S("abcde"), 0, 0, S("abcde"), 2, 4, -3);
    test(S("abcde"), 0, 0, S("abcde"), 4, 0, 0);
    test(S("abcde"), 0, 0, S("abcde"), 4, 1, -1);
    test(S("abcde"), 0, 0, S("abcde"), 4, 2, -1);
    test(S("abcde"), 0, 0, S("abcde"), 5, 0, 0);
    test(S("abcde"), 0, 0, S("abcde"), 5, 1, 0);
    test(S("abcde"), 0, 0, S("abcde"), 6, 0, 0);
    test(S("abcde"), 0, 0, S("abcdefghij"), 0, 0, 0);
    test(S("abcde"), 0, 0, S("abcdefghij"), 0, 1, -1);
    test(S("abcde"), 0, 0, S("abcdefghij"), 0, 5, -5);
    test(S("abcde"), 0, 0, S("abcdefghij"), 0, 9, -9);
    test(S("abcde"), 0, 0, S("abcdefghij"), 0, 10, -10);
    test(S("abcde"), 0, 0, S("abcdefghij"), 0, 11, -10);
    test(S("abcde"), 0, 0, S("abcdefghij"), 1, 0, 0);
    test(S("abcde"), 0, 0, S("abcdefghij"), 1, 1, -1);
    test(S("abcde"), 0, 0, S("abcdefghij"), 1, 4, -4);
    test(S("abcde"), 0, 0, S("abcdefghij"), 1, 8, -8);
    test(S("abcde"), 0, 0, S("abcdefghij"), 1, 9, -9);
    test(S("abcde"), 0, 0, S("abcdefghij"), 1, 10, -9);
    test(S("abcde"), 0, 0, S("abcdefghij"), 5, 0, 0);
    test(S("abcde"), 0, 0, S("abcdefghij"), 5, 1, -1);
    test(S("abcde"), 0, 0, S("abcdefghij"), 5, 2, -2);
    test(S("abcde"), 0, 0, S("abcdefghij"), 5, 4, -4);
    test(S("abcde"), 0, 0, S("abcdefghij"), 5, 5, -5);
    test(S("abcde"), 0, 0, S("abcdefghij"), 5, 6, -5);
    test(S("abcde"), 0, 0, S("abcdefghij"), 9, 0, 0);
    test(S("abcde"), 0, 0, S("abcdefghij"), 9, 1, -1);
    test(S("abcde"), 0, 0, S("abcdefghij"), 9, 2, -1);
    test(S("abcde"), 0, 0, S("abcdefghij"), 10, 0, 0);
    test(S("abcde"), 0, 0, S("abcdefghij"), 10, 1, 0);
    test(S("abcde"), 0, 0, S("abcdefghij"), 11, 0, 0);
    test(S("abcde"), 0, 0, S("abcdefghijklmnopqrst"), 0, 0, 0);
    test(S("abcde"), 0, 0, S("abcdefghijklmnopqrst"), 0, 1, -1);
    test(S("abcde"), 0, 0, S("abcdefghijklmnopqrst"), 0, 10, -10);
    test(S("abcde"), 0, 0, S("abcdefghijklmnopqrst"), 0, 19, -19);
    test(S("abcde"), 0, 0, S("abcdefghijklmnopqrst"), 0, 20, -20);
    test(S("abcde"), 0, 0, S("abcdefghijklmnopqrst"), 0, 21, -20);
    test(S("abcde"), 0, 0, S("abcdefghijklmnopqrst"), 1, 0, 0);
    test(S("abcde"), 0, 0, S("abcdefghijklmnopqrst"), 1, 1, -1);
    test(S("abcde"), 0, 0, S("abcdefghijklmnopqrst"), 1, 9, -9);
    test(S("abcde"), 0, 0, S("abcdefghijklmnopqrst"), 1, 18, -18);
    test(S("abcde"), 0, 0, S("abcdefghijklmnopqrst"), 1, 19, -19);
    test(S("abcde"), 0, 0, S("abcdefghijklmnopqrst"), 1, 20, -19);
    test(S("abcde"), 0, 0, S("abcdefghijklmnopqrst"), 10, 0, 0);
    test(S("abcde"), 0, 0, S("abcdefghijklmnopqrst"), 10, 1, -1);
    test(S("abcde"), 0, 0, S("abcdefghijklmnopqrst"), 10, 5, -5);
    test(S("abcde"), 0, 0, S("abcdefghijklmnopqrst"), 10, 9, -9);
    test(S("abcde"), 0, 0, S("abcdefghijklmnopqrst"), 10, 10, -10);
    test(S("abcde"), 0, 0, S("abcdefghijklmnopqrst"), 10, 11, -10);
    test(S("abcde"), 0, 0, S("abcdefghijklmnopqrst"), 19, 0, 0);
    test(S("abcde"), 0, 0, S("abcdefghijklmnopqrst"), 19, 1, -1);
    test(S("abcde"), 0, 0, S("abcdefghijklmnopqrst"), 19, 2, -1);
    test(S("abcde"), 0, 0, S("abcdefghijklmnopqrst"), 20, 0, 0);
    test(S("abcde"), 0, 0, S("abcdefghijklmnopqrst"), 20, 1, 0);
    test(S("abcde"), 0, 0, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcde"), 0, 1, S(""), 0, 0, 1);
    test(S("abcde"), 0, 1, S(""), 0, 1, 1);
    test(S("abcde"), 0, 1, S(""), 1, 0, 0);
    test(S("abcde"), 0, 1, S("abcde"), 0, 0, 1);
}

template <class S>
void test3()
{
    test(S("abcde"), 0, 1, S("abcde"), 0, 1, 0);
    test(S("abcde"), 0, 1, S("abcde"), 0, 2, -1);
    test(S("abcde"), 0, 1, S("abcde"), 0, 4, -3);
    test(S("abcde"), 0, 1, S("abcde"), 0, 5, -4);
    test(S("abcde"), 0, 1, S("abcde"), 0, 6, -4);
    test(S("abcde"), 0, 1, S("abcde"), 1, 0, 1);
    test(S("abcde"), 0, 1, S("abcde"), 1, 1, -1);
    test(S("abcde"), 0, 1, S("abcde"), 1, 2, -1);
    test(S("abcde"), 0, 1, S("abcde"), 1, 3, -1);
    test(S("abcde"), 0, 1, S("abcde"), 1, 4, -1);
    test(S("abcde"), 0, 1, S("abcde"), 1, 5, -1);
    test(S("abcde"), 0, 1, S("abcde"), 2, 0, 1);
    test(S("abcde"), 0, 1, S("abcde"), 2, 1, -2);
    test(S("abcde"), 0, 1, S("abcde"), 2, 2, -2);
    test(S("abcde"), 0, 1, S("abcde"), 2, 3, -2);
    test(S("abcde"), 0, 1, S("abcde"), 2, 4, -2);
    test(S("abcde"), 0, 1, S("abcde"), 4, 0, 1);
    test(S("abcde"), 0, 1, S("abcde"), 4, 1, -4);
    test(S("abcde"), 0, 1, S("abcde"), 4, 2, -4);
    test(S("abcde"), 0, 1, S("abcde"), 5, 0, 1);
    test(S("abcde"), 0, 1, S("abcde"), 5, 1, 1);
    test(S("abcde"), 0, 1, S("abcde"), 6, 0, 0);
    test(S("abcde"), 0, 1, S("abcdefghij"), 0, 0, 1);
    test(S("abcde"), 0, 1, S("abcdefghij"), 0, 1, 0);
    test(S("abcde"), 0, 1, S("abcdefghij"), 0, 5, -4);
    test(S("abcde"), 0, 1, S("abcdefghij"), 0, 9, -8);
    test(S("abcde"), 0, 1, S("abcdefghij"), 0, 10, -9);
    test(S("abcde"), 0, 1, S("abcdefghij"), 0, 11, -9);
    test(S("abcde"), 0, 1, S("abcdefghij"), 1, 0, 1);
    test(S("abcde"), 0, 1, S("abcdefghij"), 1, 1, -1);
    test(S("abcde"), 0, 1, S("abcdefghij"), 1, 4, -1);
    test(S("abcde"), 0, 1, S("abcdefghij"), 1, 8, -1);
    test(S("abcde"), 0, 1, S("abcdefghij"), 1, 9, -1);
    test(S("abcde"), 0, 1, S("abcdefghij"), 1, 10, -1);
    test(S("abcde"), 0, 1, S("abcdefghij"), 5, 0, 1);
    test(S("abcde"), 0, 1, S("abcdefghij"), 5, 1, -5);
    test(S("abcde"), 0, 1, S("abcdefghij"), 5, 2, -5);
    test(S("abcde"), 0, 1, S("abcdefghij"), 5, 4, -5);
    test(S("abcde"), 0, 1, S("abcdefghij"), 5, 5, -5);
    test(S("abcde"), 0, 1, S("abcdefghij"), 5, 6, -5);
    test(S("abcde"), 0, 1, S("abcdefghij"), 9, 0, 1);
    test(S("abcde"), 0, 1, S("abcdefghij"), 9, 1, -9);
    test(S("abcde"), 0, 1, S("abcdefghij"), 9, 2, -9);
    test(S("abcde"), 0, 1, S("abcdefghij"), 10, 0, 1);
    test(S("abcde"), 0, 1, S("abcdefghij"), 10, 1, 1);
    test(S("abcde"), 0, 1, S("abcdefghij"), 11, 0, 0);
    test(S("abcde"), 0, 1, S("abcdefghijklmnopqrst"), 0, 0, 1);
    test(S("abcde"), 0, 1, S("abcdefghijklmnopqrst"), 0, 1, 0);
    test(S("abcde"), 0, 1, S("abcdefghijklmnopqrst"), 0, 10, -9);
    test(S("abcde"), 0, 1, S("abcdefghijklmnopqrst"), 0, 19, -18);
    test(S("abcde"), 0, 1, S("abcdefghijklmnopqrst"), 0, 20, -19);
    test(S("abcde"), 0, 1, S("abcdefghijklmnopqrst"), 0, 21, -19);
    test(S("abcde"), 0, 1, S("abcdefghijklmnopqrst"), 1, 0, 1);
    test(S("abcde"), 0, 1, S("abcdefghijklmnopqrst"), 1, 1, -1);
    test(S("abcde"), 0, 1, S("abcdefghijklmnopqrst"), 1, 9, -1);
    test(S("abcde"), 0, 1, S("abcdefghijklmnopqrst"), 1, 18, -1);
    test(S("abcde"), 0, 1, S("abcdefghijklmnopqrst"), 1, 19, -1);
    test(S("abcde"), 0, 1, S("abcdefghijklmnopqrst"), 1, 20, -1);
    test(S("abcde"), 0, 1, S("abcdefghijklmnopqrst"), 10, 0, 1);
    test(S("abcde"), 0, 1, S("abcdefghijklmnopqrst"), 10, 1, -10);
    test(S("abcde"), 0, 1, S("abcdefghijklmnopqrst"), 10, 5, -10);
    test(S("abcde"), 0, 1, S("abcdefghijklmnopqrst"), 10, 9, -10);
    test(S("abcde"), 0, 1, S("abcdefghijklmnopqrst"), 10, 10, -10);
    test(S("abcde"), 0, 1, S("abcdefghijklmnopqrst"), 10, 11, -10);
    test(S("abcde"), 0, 1, S("abcdefghijklmnopqrst"), 19, 0, 1);
    test(S("abcde"), 0, 1, S("abcdefghijklmnopqrst"), 19, 1, -19);
    test(S("abcde"), 0, 1, S("abcdefghijklmnopqrst"), 19, 2, -19);
    test(S("abcde"), 0, 1, S("abcdefghijklmnopqrst"), 20, 0, 1);
    test(S("abcde"), 0, 1, S("abcdefghijklmnopqrst"), 20, 1, 1);
    test(S("abcde"), 0, 1, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcde"), 0, 2, S(""), 0, 0, 2);
    test(S("abcde"), 0, 2, S(""), 0, 1, 2);
    test(S("abcde"), 0, 2, S(""), 1, 0, 0);
    test(S("abcde"), 0, 2, S("abcde"), 0, 0, 2);
    test(S("abcde"), 0, 2, S("abcde"), 0, 1, 1);
    test(S("abcde"), 0, 2, S("abcde"), 0, 2, 0);
    test(S("abcde"), 0, 2, S("abcde"), 0, 4, -2);
    test(S("abcde"), 0, 2, S("abcde"), 0, 5, -3);
    test(S("abcde"), 0, 2, S("abcde"), 0, 6, -3);
    test(S("abcde"), 0, 2, S("abcde"), 1, 0, 2);
    test(S("abcde"), 0, 2, S("abcde"), 1, 1, -1);
    test(S("abcde"), 0, 2, S("abcde"), 1, 2, -1);
    test(S("abcde"), 0, 2, S("abcde"), 1, 3, -1);
    test(S("abcde"), 0, 2, S("abcde"), 1, 4, -1);
    test(S("abcde"), 0, 2, S("abcde"), 1, 5, -1);
    test(S("abcde"), 0, 2, S("abcde"), 2, 0, 2);
    test(S("abcde"), 0, 2, S("abcde"), 2, 1, -2);
    test(S("abcde"), 0, 2, S("abcde"), 2, 2, -2);
    test(S("abcde"), 0, 2, S("abcde"), 2, 3, -2);
    test(S("abcde"), 0, 2, S("abcde"), 2, 4, -2);
    test(S("abcde"), 0, 2, S("abcde"), 4, 0, 2);
    test(S("abcde"), 0, 2, S("abcde"), 4, 1, -4);
    test(S("abcde"), 0, 2, S("abcde"), 4, 2, -4);
    test(S("abcde"), 0, 2, S("abcde"), 5, 0, 2);
    test(S("abcde"), 0, 2, S("abcde"), 5, 1, 2);
    test(S("abcde"), 0, 2, S("abcde"), 6, 0, 0);
    test(S("abcde"), 0, 2, S("abcdefghij"), 0, 0, 2);
    test(S("abcde"), 0, 2, S("abcdefghij"), 0, 1, 1);
    test(S("abcde"), 0, 2, S("abcdefghij"), 0, 5, -3);
    test(S("abcde"), 0, 2, S("abcdefghij"), 0, 9, -7);
}

template <class S>
void test4()
{
    test(S("abcde"), 0, 2, S("abcdefghij"), 0, 10, -8);
    test(S("abcde"), 0, 2, S("abcdefghij"), 0, 11, -8);
    test(S("abcde"), 0, 2, S("abcdefghij"), 1, 0, 2);
    test(S("abcde"), 0, 2, S("abcdefghij"), 1, 1, -1);
    test(S("abcde"), 0, 2, S("abcdefghij"), 1, 4, -1);
    test(S("abcde"), 0, 2, S("abcdefghij"), 1, 8, -1);
    test(S("abcde"), 0, 2, S("abcdefghij"), 1, 9, -1);
    test(S("abcde"), 0, 2, S("abcdefghij"), 1, 10, -1);
    test(S("abcde"), 0, 2, S("abcdefghij"), 5, 0, 2);
    test(S("abcde"), 0, 2, S("abcdefghij"), 5, 1, -5);
    test(S("abcde"), 0, 2, S("abcdefghij"), 5, 2, -5);
    test(S("abcde"), 0, 2, S("abcdefghij"), 5, 4, -5);
    test(S("abcde"), 0, 2, S("abcdefghij"), 5, 5, -5);
    test(S("abcde"), 0, 2, S("abcdefghij"), 5, 6, -5);
    test(S("abcde"), 0, 2, S("abcdefghij"), 9, 0, 2);
    test(S("abcde"), 0, 2, S("abcdefghij"), 9, 1, -9);
    test(S("abcde"), 0, 2, S("abcdefghij"), 9, 2, -9);
    test(S("abcde"), 0, 2, S("abcdefghij"), 10, 0, 2);
    test(S("abcde"), 0, 2, S("abcdefghij"), 10, 1, 2);
    test(S("abcde"), 0, 2, S("abcdefghij"), 11, 0, 0);
    test(S("abcde"), 0, 2, S("abcdefghijklmnopqrst"), 0, 0, 2);
    test(S("abcde"), 0, 2, S("abcdefghijklmnopqrst"), 0, 1, 1);
    test(S("abcde"), 0, 2, S("abcdefghijklmnopqrst"), 0, 10, -8);
    test(S("abcde"), 0, 2, S("abcdefghijklmnopqrst"), 0, 19, -17);
    test(S("abcde"), 0, 2, S("abcdefghijklmnopqrst"), 0, 20, -18);
    test(S("abcde"), 0, 2, S("abcdefghijklmnopqrst"), 0, 21, -18);
    test(S("abcde"), 0, 2, S("abcdefghijklmnopqrst"), 1, 0, 2);
    test(S("abcde"), 0, 2, S("abcdefghijklmnopqrst"), 1, 1, -1);
    test(S("abcde"), 0, 2, S("abcdefghijklmnopqrst"), 1, 9, -1);
    test(S("abcde"), 0, 2, S("abcdefghijklmnopqrst"), 1, 18, -1);
    test(S("abcde"), 0, 2, S("abcdefghijklmnopqrst"), 1, 19, -1);
    test(S("abcde"), 0, 2, S("abcdefghijklmnopqrst"), 1, 20, -1);
    test(S("abcde"), 0, 2, S("abcdefghijklmnopqrst"), 10, 0, 2);
    test(S("abcde"), 0, 2, S("abcdefghijklmnopqrst"), 10, 1, -10);
    test(S("abcde"), 0, 2, S("abcdefghijklmnopqrst"), 10, 5, -10);
    test(S("abcde"), 0, 2, S("abcdefghijklmnopqrst"), 10, 9, -10);
    test(S("abcde"), 0, 2, S("abcdefghijklmnopqrst"), 10, 10, -10);
    test(S("abcde"), 0, 2, S("abcdefghijklmnopqrst"), 10, 11, -10);
    test(S("abcde"), 0, 2, S("abcdefghijklmnopqrst"), 19, 0, 2);
    test(S("abcde"), 0, 2, S("abcdefghijklmnopqrst"), 19, 1, -19);
    test(S("abcde"), 0, 2, S("abcdefghijklmnopqrst"), 19, 2, -19);
    test(S("abcde"), 0, 2, S("abcdefghijklmnopqrst"), 20, 0, 2);
    test(S("abcde"), 0, 2, S("abcdefghijklmnopqrst"), 20, 1, 2);
    test(S("abcde"), 0, 2, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcde"), 0, 4, S(""), 0, 0, 4);
    test(S("abcde"), 0, 4, S(""), 0, 1, 4);
    test(S("abcde"), 0, 4, S(""), 1, 0, 0);
    test(S("abcde"), 0, 4, S("abcde"), 0, 0, 4);
    test(S("abcde"), 0, 4, S("abcde"), 0, 1, 3);
    test(S("abcde"), 0, 4, S("abcde"), 0, 2, 2);
    test(S("abcde"), 0, 4, S("abcde"), 0, 4, 0);
    test(S("abcde"), 0, 4, S("abcde"), 0, 5, -1);
    test(S("abcde"), 0, 4, S("abcde"), 0, 6, -1);
    test(S("abcde"), 0, 4, S("abcde"), 1, 0, 4);
    test(S("abcde"), 0, 4, S("abcde"), 1, 1, -1);
    test(S("abcde"), 0, 4, S("abcde"), 1, 2, -1);
    test(S("abcde"), 0, 4, S("abcde"), 1, 3, -1);
    test(S("abcde"), 0, 4, S("abcde"), 1, 4, -1);
    test(S("abcde"), 0, 4, S("abcde"), 1, 5, -1);
    test(S("abcde"), 0, 4, S("abcde"), 2, 0, 4);
    test(S("abcde"), 0, 4, S("abcde"), 2, 1, -2);
    test(S("abcde"), 0, 4, S("abcde"), 2, 2, -2);
    test(S("abcde"), 0, 4, S("abcde"), 2, 3, -2);
    test(S("abcde"), 0, 4, S("abcde"), 2, 4, -2);
    test(S("abcde"), 0, 4, S("abcde"), 4, 0, 4);
    test(S("abcde"), 0, 4, S("abcde"), 4, 1, -4);
    test(S("abcde"), 0, 4, S("abcde"), 4, 2, -4);
    test(S("abcde"), 0, 4, S("abcde"), 5, 0, 4);
    test(S("abcde"), 0, 4, S("abcde"), 5, 1, 4);
    test(S("abcde"), 0, 4, S("abcde"), 6, 0, 0);
    test(S("abcde"), 0, 4, S("abcdefghij"), 0, 0, 4);
    test(S("abcde"), 0, 4, S("abcdefghij"), 0, 1, 3);
    test(S("abcde"), 0, 4, S("abcdefghij"), 0, 5, -1);
    test(S("abcde"), 0, 4, S("abcdefghij"), 0, 9, -5);
    test(S("abcde"), 0, 4, S("abcdefghij"), 0, 10, -6);
    test(S("abcde"), 0, 4, S("abcdefghij"), 0, 11, -6);
    test(S("abcde"), 0, 4, S("abcdefghij"), 1, 0, 4);
    test(S("abcde"), 0, 4, S("abcdefghij"), 1, 1, -1);
    test(S("abcde"), 0, 4, S("abcdefghij"), 1, 4, -1);
    test(S("abcde"), 0, 4, S("abcdefghij"), 1, 8, -1);
    test(S("abcde"), 0, 4, S("abcdefghij"), 1, 9, -1);
    test(S("abcde"), 0, 4, S("abcdefghij"), 1, 10, -1);
    test(S("abcde"), 0, 4, S("abcdefghij"), 5, 0, 4);
    test(S("abcde"), 0, 4, S("abcdefghij"), 5, 1, -5);
    test(S("abcde"), 0, 4, S("abcdefghij"), 5, 2, -5);
    test(S("abcde"), 0, 4, S("abcdefghij"), 5, 4, -5);
    test(S("abcde"), 0, 4, S("abcdefghij"), 5, 5, -5);
    test(S("abcde"), 0, 4, S("abcdefghij"), 5, 6, -5);
    test(S("abcde"), 0, 4, S("abcdefghij"), 9, 0, 4);
    test(S("abcde"), 0, 4, S("abcdefghij"), 9, 1, -9);
    test(S("abcde"), 0, 4, S("abcdefghij"), 9, 2, -9);
    test(S("abcde"), 0, 4, S("abcdefghij"), 10, 0, 4);
    test(S("abcde"), 0, 4, S("abcdefghij"), 10, 1, 4);
    test(S("abcde"), 0, 4, S("abcdefghij"), 11, 0, 0);
    test(S("abcde"), 0, 4, S("abcdefghijklmnopqrst"), 0, 0, 4);
    test(S("abcde"), 0, 4, S("abcdefghijklmnopqrst"), 0, 1, 3);
    test(S("abcde"), 0, 4, S("abcdefghijklmnopqrst"), 0, 10, -6);
    test(S("abcde"), 0, 4, S("abcdefghijklmnopqrst"), 0, 19, -15);
    test(S("abcde"), 0, 4, S("abcdefghijklmnopqrst"), 0, 20, -16);
    test(S("abcde"), 0, 4, S("abcdefghijklmnopqrst"), 0, 21, -16);
}

template <class S>
void test5()
{
    test(S("abcde"), 0, 4, S("abcdefghijklmnopqrst"), 1, 0, 4);
    test(S("abcde"), 0, 4, S("abcdefghijklmnopqrst"), 1, 1, -1);
    test(S("abcde"), 0, 4, S("abcdefghijklmnopqrst"), 1, 9, -1);
    test(S("abcde"), 0, 4, S("abcdefghijklmnopqrst"), 1, 18, -1);
    test(S("abcde"), 0, 4, S("abcdefghijklmnopqrst"), 1, 19, -1);
    test(S("abcde"), 0, 4, S("abcdefghijklmnopqrst"), 1, 20, -1);
    test(S("abcde"), 0, 4, S("abcdefghijklmnopqrst"), 10, 0, 4);
    test(S("abcde"), 0, 4, S("abcdefghijklmnopqrst"), 10, 1, -10);
    test(S("abcde"), 0, 4, S("abcdefghijklmnopqrst"), 10, 5, -10);
    test(S("abcde"), 0, 4, S("abcdefghijklmnopqrst"), 10, 9, -10);
    test(S("abcde"), 0, 4, S("abcdefghijklmnopqrst"), 10, 10, -10);
    test(S("abcde"), 0, 4, S("abcdefghijklmnopqrst"), 10, 11, -10);
    test(S("abcde"), 0, 4, S("abcdefghijklmnopqrst"), 19, 0, 4);
    test(S("abcde"), 0, 4, S("abcdefghijklmnopqrst"), 19, 1, -19);
    test(S("abcde"), 0, 4, S("abcdefghijklmnopqrst"), 19, 2, -19);
    test(S("abcde"), 0, 4, S("abcdefghijklmnopqrst"), 20, 0, 4);
    test(S("abcde"), 0, 4, S("abcdefghijklmnopqrst"), 20, 1, 4);
    test(S("abcde"), 0, 4, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcde"), 0, 5, S(""), 0, 0, 5);
    test(S("abcde"), 0, 5, S(""), 0, 1, 5);
    test(S("abcde"), 0, 5, S(""), 1, 0, 0);
    test(S("abcde"), 0, 5, S("abcde"), 0, 0, 5);
    test(S("abcde"), 0, 5, S("abcde"), 0, 1, 4);
    test(S("abcde"), 0, 5, S("abcde"), 0, 2, 3);
    test(S("abcde"), 0, 5, S("abcde"), 0, 4, 1);
    test(S("abcde"), 0, 5, S("abcde"), 0, 5, 0);
    test(S("abcde"), 0, 5, S("abcde"), 0, 6, 0);
    test(S("abcde"), 0, 5, S("abcde"), 1, 0, 5);
    test(S("abcde"), 0, 5, S("abcde"), 1, 1, -1);
    test(S("abcde"), 0, 5, S("abcde"), 1, 2, -1);
    test(S("abcde"), 0, 5, S("abcde"), 1, 3, -1);
    test(S("abcde"), 0, 5, S("abcde"), 1, 4, -1);
    test(S("abcde"), 0, 5, S("abcde"), 1, 5, -1);
    test(S("abcde"), 0, 5, S("abcde"), 2, 0, 5);
    test(S("abcde"), 0, 5, S("abcde"), 2, 1, -2);
    test(S("abcde"), 0, 5, S("abcde"), 2, 2, -2);
    test(S("abcde"), 0, 5, S("abcde"), 2, 3, -2);
    test(S("abcde"), 0, 5, S("abcde"), 2, 4, -2);
    test(S("abcde"), 0, 5, S("abcde"), 4, 0, 5);
    test(S("abcde"), 0, 5, S("abcde"), 4, 1, -4);
    test(S("abcde"), 0, 5, S("abcde"), 4, 2, -4);
    test(S("abcde"), 0, 5, S("abcde"), 5, 0, 5);
    test(S("abcde"), 0, 5, S("abcde"), 5, 1, 5);
    test(S("abcde"), 0, 5, S("abcde"), 6, 0, 0);
    test(S("abcde"), 0, 5, S("abcdefghij"), 0, 0, 5);
    test(S("abcde"), 0, 5, S("abcdefghij"), 0, 1, 4);
    test(S("abcde"), 0, 5, S("abcdefghij"), 0, 5, 0);
    test(S("abcde"), 0, 5, S("abcdefghij"), 0, 9, -4);
    test(S("abcde"), 0, 5, S("abcdefghij"), 0, 10, -5);
    test(S("abcde"), 0, 5, S("abcdefghij"), 0, 11, -5);
    test(S("abcde"), 0, 5, S("abcdefghij"), 1, 0, 5);
    test(S("abcde"), 0, 5, S("abcdefghij"), 1, 1, -1);
    test(S("abcde"), 0, 5, S("abcdefghij"), 1, 4, -1);
    test(S("abcde"), 0, 5, S("abcdefghij"), 1, 8, -1);
    test(S("abcde"), 0, 5, S("abcdefghij"), 1, 9, -1);
    test(S("abcde"), 0, 5, S("abcdefghij"), 1, 10, -1);
    test(S("abcde"), 0, 5, S("abcdefghij"), 5, 0, 5);
    test(S("abcde"), 0, 5, S("abcdefghij"), 5, 1, -5);
    test(S("abcde"), 0, 5, S("abcdefghij"), 5, 2, -5);
    test(S("abcde"), 0, 5, S("abcdefghij"), 5, 4, -5);
    test(S("abcde"), 0, 5, S("abcdefghij"), 5, 5, -5);
    test(S("abcde"), 0, 5, S("abcdefghij"), 5, 6, -5);
    test(S("abcde"), 0, 5, S("abcdefghij"), 9, 0, 5);
    test(S("abcde"), 0, 5, S("abcdefghij"), 9, 1, -9);
    test(S("abcde"), 0, 5, S("abcdefghij"), 9, 2, -9);
    test(S("abcde"), 0, 5, S("abcdefghij"), 10, 0, 5);
    test(S("abcde"), 0, 5, S("abcdefghij"), 10, 1, 5);
    test(S("abcde"), 0, 5, S("abcdefghij"), 11, 0, 0);
    test(S("abcde"), 0, 5, S("abcdefghijklmnopqrst"), 0, 0, 5);
    test(S("abcde"), 0, 5, S("abcdefghijklmnopqrst"), 0, 1, 4);
    test(S("abcde"), 0, 5, S("abcdefghijklmnopqrst"), 0, 10, -5);
    test(S("abcde"), 0, 5, S("abcdefghijklmnopqrst"), 0, 19, -14);
    test(S("abcde"), 0, 5, S("abcdefghijklmnopqrst"), 0, 20, -15);
    test(S("abcde"), 0, 5, S("abcdefghijklmnopqrst"), 0, 21, -15);
    test(S("abcde"), 0, 5, S("abcdefghijklmnopqrst"), 1, 0, 5);
    test(S("abcde"), 0, 5, S("abcdefghijklmnopqrst"), 1, 1, -1);
    test(S("abcde"), 0, 5, S("abcdefghijklmnopqrst"), 1, 9, -1);
    test(S("abcde"), 0, 5, S("abcdefghijklmnopqrst"), 1, 18, -1);
    test(S("abcde"), 0, 5, S("abcdefghijklmnopqrst"), 1, 19, -1);
    test(S("abcde"), 0, 5, S("abcdefghijklmnopqrst"), 1, 20, -1);
    test(S("abcde"), 0, 5, S("abcdefghijklmnopqrst"), 10, 0, 5);
    test(S("abcde"), 0, 5, S("abcdefghijklmnopqrst"), 10, 1, -10);
    test(S("abcde"), 0, 5, S("abcdefghijklmnopqrst"), 10, 5, -10);
    test(S("abcde"), 0, 5, S("abcdefghijklmnopqrst"), 10, 9, -10);
    test(S("abcde"), 0, 5, S("abcdefghijklmnopqrst"), 10, 10, -10);
    test(S("abcde"), 0, 5, S("abcdefghijklmnopqrst"), 10, 11, -10);
    test(S("abcde"), 0, 5, S("abcdefghijklmnopqrst"), 19, 0, 5);
    test(S("abcde"), 0, 5, S("abcdefghijklmnopqrst"), 19, 1, -19);
    test(S("abcde"), 0, 5, S("abcdefghijklmnopqrst"), 19, 2, -19);
    test(S("abcde"), 0, 5, S("abcdefghijklmnopqrst"), 20, 0, 5);
    test(S("abcde"), 0, 5, S("abcdefghijklmnopqrst"), 20, 1, 5);
    test(S("abcde"), 0, 5, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcde"), 0, 6, S(""), 0, 0, 5);
    test(S("abcde"), 0, 6, S(""), 0, 1, 5);
    test(S("abcde"), 0, 6, S(""), 1, 0, 0);
    test(S("abcde"), 0, 6, S("abcde"), 0, 0, 5);
    test(S("abcde"), 0, 6, S("abcde"), 0, 1, 4);
    test(S("abcde"), 0, 6, S("abcde"), 0, 2, 3);
    test(S("abcde"), 0, 6, S("abcde"), 0, 4, 1);
    test(S("abcde"), 0, 6, S("abcde"), 0, 5, 0);
}

template <class S>
void test6()
{
    test(S("abcde"), 0, 6, S("abcde"), 0, 6, 0);
    test(S("abcde"), 0, 6, S("abcde"), 1, 0, 5);
    test(S("abcde"), 0, 6, S("abcde"), 1, 1, -1);
    test(S("abcde"), 0, 6, S("abcde"), 1, 2, -1);
    test(S("abcde"), 0, 6, S("abcde"), 1, 3, -1);
    test(S("abcde"), 0, 6, S("abcde"), 1, 4, -1);
    test(S("abcde"), 0, 6, S("abcde"), 1, 5, -1);
    test(S("abcde"), 0, 6, S("abcde"), 2, 0, 5);
    test(S("abcde"), 0, 6, S("abcde"), 2, 1, -2);
    test(S("abcde"), 0, 6, S("abcde"), 2, 2, -2);
    test(S("abcde"), 0, 6, S("abcde"), 2, 3, -2);
    test(S("abcde"), 0, 6, S("abcde"), 2, 4, -2);
    test(S("abcde"), 0, 6, S("abcde"), 4, 0, 5);
    test(S("abcde"), 0, 6, S("abcde"), 4, 1, -4);
    test(S("abcde"), 0, 6, S("abcde"), 4, 2, -4);
    test(S("abcde"), 0, 6, S("abcde"), 5, 0, 5);
    test(S("abcde"), 0, 6, S("abcde"), 5, 1, 5);
    test(S("abcde"), 0, 6, S("abcde"), 6, 0, 0);
    test(S("abcde"), 0, 6, S("abcdefghij"), 0, 0, 5);
    test(S("abcde"), 0, 6, S("abcdefghij"), 0, 1, 4);
    test(S("abcde"), 0, 6, S("abcdefghij"), 0, 5, 0);
    test(S("abcde"), 0, 6, S("abcdefghij"), 0, 9, -4);
    test(S("abcde"), 0, 6, S("abcdefghij"), 0, 10, -5);
    test(S("abcde"), 0, 6, S("abcdefghij"), 0, 11, -5);
    test(S("abcde"), 0, 6, S("abcdefghij"), 1, 0, 5);
    test(S("abcde"), 0, 6, S("abcdefghij"), 1, 1, -1);
    test(S("abcde"), 0, 6, S("abcdefghij"), 1, 4, -1);
    test(S("abcde"), 0, 6, S("abcdefghij"), 1, 8, -1);
    test(S("abcde"), 0, 6, S("abcdefghij"), 1, 9, -1);
    test(S("abcde"), 0, 6, S("abcdefghij"), 1, 10, -1);
    test(S("abcde"), 0, 6, S("abcdefghij"), 5, 0, 5);
    test(S("abcde"), 0, 6, S("abcdefghij"), 5, 1, -5);
    test(S("abcde"), 0, 6, S("abcdefghij"), 5, 2, -5);
    test(S("abcde"), 0, 6, S("abcdefghij"), 5, 4, -5);
    test(S("abcde"), 0, 6, S("abcdefghij"), 5, 5, -5);
    test(S("abcde"), 0, 6, S("abcdefghij"), 5, 6, -5);
    test(S("abcde"), 0, 6, S("abcdefghij"), 9, 0, 5);
    test(S("abcde"), 0, 6, S("abcdefghij"), 9, 1, -9);
    test(S("abcde"), 0, 6, S("abcdefghij"), 9, 2, -9);
    test(S("abcde"), 0, 6, S("abcdefghij"), 10, 0, 5);
    test(S("abcde"), 0, 6, S("abcdefghij"), 10, 1, 5);
    test(S("abcde"), 0, 6, S("abcdefghij"), 11, 0, 0);
    test(S("abcde"), 0, 6, S("abcdefghijklmnopqrst"), 0, 0, 5);
    test(S("abcde"), 0, 6, S("abcdefghijklmnopqrst"), 0, 1, 4);
    test(S("abcde"), 0, 6, S("abcdefghijklmnopqrst"), 0, 10, -5);
    test(S("abcde"), 0, 6, S("abcdefghijklmnopqrst"), 0, 19, -14);
    test(S("abcde"), 0, 6, S("abcdefghijklmnopqrst"), 0, 20, -15);
    test(S("abcde"), 0, 6, S("abcdefghijklmnopqrst"), 0, 21, -15);
    test(S("abcde"), 0, 6, S("abcdefghijklmnopqrst"), 1, 0, 5);
    test(S("abcde"), 0, 6, S("abcdefghijklmnopqrst"), 1, 1, -1);
    test(S("abcde"), 0, 6, S("abcdefghijklmnopqrst"), 1, 9, -1);
    test(S("abcde"), 0, 6, S("abcdefghijklmnopqrst"), 1, 18, -1);
    test(S("abcde"), 0, 6, S("abcdefghijklmnopqrst"), 1, 19, -1);
    test(S("abcde"), 0, 6, S("abcdefghijklmnopqrst"), 1, 20, -1);
    test(S("abcde"), 0, 6, S("abcdefghijklmnopqrst"), 10, 0, 5);
    test(S("abcde"), 0, 6, S("abcdefghijklmnopqrst"), 10, 1, -10);
    test(S("abcde"), 0, 6, S("abcdefghijklmnopqrst"), 10, 5, -10);
    test(S("abcde"), 0, 6, S("abcdefghijklmnopqrst"), 10, 9, -10);
    test(S("abcde"), 0, 6, S("abcdefghijklmnopqrst"), 10, 10, -10);
    test(S("abcde"), 0, 6, S("abcdefghijklmnopqrst"), 10, 11, -10);
    test(S("abcde"), 0, 6, S("abcdefghijklmnopqrst"), 19, 0, 5);
    test(S("abcde"), 0, 6, S("abcdefghijklmnopqrst"), 19, 1, -19);
    test(S("abcde"), 0, 6, S("abcdefghijklmnopqrst"), 19, 2, -19);
    test(S("abcde"), 0, 6, S("abcdefghijklmnopqrst"), 20, 0, 5);
    test(S("abcde"), 0, 6, S("abcdefghijklmnopqrst"), 20, 1, 5);
    test(S("abcde"), 0, 6, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcde"), 1, 0, S(""), 0, 0, 0);
    test(S("abcde"), 1, 0, S(""), 0, 1, 0);
    test(S("abcde"), 1, 0, S(""), 1, 0, 0);
    test(S("abcde"), 1, 0, S("abcde"), 0, 0, 0);
    test(S("abcde"), 1, 0, S("abcde"), 0, 1, -1);
    test(S("abcde"), 1, 0, S("abcde"), 0, 2, -2);
    test(S("abcde"), 1, 0, S("abcde"), 0, 4, -4);
    test(S("abcde"), 1, 0, S("abcde"), 0, 5, -5);
    test(S("abcde"), 1, 0, S("abcde"), 0, 6, -5);
    test(S("abcde"), 1, 0, S("abcde"), 1, 0, 0);
    test(S("abcde"), 1, 0, S("abcde"), 1, 1, -1);
    test(S("abcde"), 1, 0, S("abcde"), 1, 2, -2);
    test(S("abcde"), 1, 0, S("abcde"), 1, 3, -3);
    test(S("abcde"), 1, 0, S("abcde"), 1, 4, -4);
    test(S("abcde"), 1, 0, S("abcde"), 1, 5, -4);
    test(S("abcde"), 1, 0, S("abcde"), 2, 0, 0);
    test(S("abcde"), 1, 0, S("abcde"), 2, 1, -1);
    test(S("abcde"), 1, 0, S("abcde"), 2, 2, -2);
    test(S("abcde"), 1, 0, S("abcde"), 2, 3, -3);
    test(S("abcde"), 1, 0, S("abcde"), 2, 4, -3);
    test(S("abcde"), 1, 0, S("abcde"), 4, 0, 0);
    test(S("abcde"), 1, 0, S("abcde"), 4, 1, -1);
    test(S("abcde"), 1, 0, S("abcde"), 4, 2, -1);
    test(S("abcde"), 1, 0, S("abcde"), 5, 0, 0);
    test(S("abcde"), 1, 0, S("abcde"), 5, 1, 0);
    test(S("abcde"), 1, 0, S("abcde"), 6, 0, 0);
    test(S("abcde"), 1, 0, S("abcdefghij"), 0, 0, 0);
    test(S("abcde"), 1, 0, S("abcdefghij"), 0, 1, -1);
    test(S("abcde"), 1, 0, S("abcdefghij"), 0, 5, -5);
    test(S("abcde"), 1, 0, S("abcdefghij"), 0, 9, -9);
    test(S("abcde"), 1, 0, S("abcdefghij"), 0, 10, -10);
    test(S("abcde"), 1, 0, S("abcdefghij"), 0, 11, -10);
    test(S("abcde"), 1, 0, S("abcdefghij"), 1, 0, 0);
    test(S("abcde"), 1, 0, S("abcdefghij"), 1, 1, -1);
}

template <class S>
void test7()
{
    test(S("abcde"), 1, 0, S("abcdefghij"), 1, 4, -4);
    test(S("abcde"), 1, 0, S("abcdefghij"), 1, 8, -8);
    test(S("abcde"), 1, 0, S("abcdefghij"), 1, 9, -9);
    test(S("abcde"), 1, 0, S("abcdefghij"), 1, 10, -9);
    test(S("abcde"), 1, 0, S("abcdefghij"), 5, 0, 0);
    test(S("abcde"), 1, 0, S("abcdefghij"), 5, 1, -1);
    test(S("abcde"), 1, 0, S("abcdefghij"), 5, 2, -2);
    test(S("abcde"), 1, 0, S("abcdefghij"), 5, 4, -4);
    test(S("abcde"), 1, 0, S("abcdefghij"), 5, 5, -5);
    test(S("abcde"), 1, 0, S("abcdefghij"), 5, 6, -5);
    test(S("abcde"), 1, 0, S("abcdefghij"), 9, 0, 0);
    test(S("abcde"), 1, 0, S("abcdefghij"), 9, 1, -1);
    test(S("abcde"), 1, 0, S("abcdefghij"), 9, 2, -1);
    test(S("abcde"), 1, 0, S("abcdefghij"), 10, 0, 0);
    test(S("abcde"), 1, 0, S("abcdefghij"), 10, 1, 0);
    test(S("abcde"), 1, 0, S("abcdefghij"), 11, 0, 0);
    test(S("abcde"), 1, 0, S("abcdefghijklmnopqrst"), 0, 0, 0);
    test(S("abcde"), 1, 0, S("abcdefghijklmnopqrst"), 0, 1, -1);
    test(S("abcde"), 1, 0, S("abcdefghijklmnopqrst"), 0, 10, -10);
    test(S("abcde"), 1, 0, S("abcdefghijklmnopqrst"), 0, 19, -19);
    test(S("abcde"), 1, 0, S("abcdefghijklmnopqrst"), 0, 20, -20);
    test(S("abcde"), 1, 0, S("abcdefghijklmnopqrst"), 0, 21, -20);
    test(S("abcde"), 1, 0, S("abcdefghijklmnopqrst"), 1, 0, 0);
    test(S("abcde"), 1, 0, S("abcdefghijklmnopqrst"), 1, 1, -1);
    test(S("abcde"), 1, 0, S("abcdefghijklmnopqrst"), 1, 9, -9);
    test(S("abcde"), 1, 0, S("abcdefghijklmnopqrst"), 1, 18, -18);
    test(S("abcde"), 1, 0, S("abcdefghijklmnopqrst"), 1, 19, -19);
    test(S("abcde"), 1, 0, S("abcdefghijklmnopqrst"), 1, 20, -19);
    test(S("abcde"), 1, 0, S("abcdefghijklmnopqrst"), 10, 0, 0);
    test(S("abcde"), 1, 0, S("abcdefghijklmnopqrst"), 10, 1, -1);
    test(S("abcde"), 1, 0, S("abcdefghijklmnopqrst"), 10, 5, -5);
    test(S("abcde"), 1, 0, S("abcdefghijklmnopqrst"), 10, 9, -9);
    test(S("abcde"), 1, 0, S("abcdefghijklmnopqrst"), 10, 10, -10);
    test(S("abcde"), 1, 0, S("abcdefghijklmnopqrst"), 10, 11, -10);
    test(S("abcde"), 1, 0, S("abcdefghijklmnopqrst"), 19, 0, 0);
    test(S("abcde"), 1, 0, S("abcdefghijklmnopqrst"), 19, 1, -1);
    test(S("abcde"), 1, 0, S("abcdefghijklmnopqrst"), 19, 2, -1);
    test(S("abcde"), 1, 0, S("abcdefghijklmnopqrst"), 20, 0, 0);
    test(S("abcde"), 1, 0, S("abcdefghijklmnopqrst"), 20, 1, 0);
    test(S("abcde"), 1, 0, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcde"), 1, 1, S(""), 0, 0, 1);
    test(S("abcde"), 1, 1, S(""), 0, 1, 1);
    test(S("abcde"), 1, 1, S(""), 1, 0, 0);
    test(S("abcde"), 1, 1, S("abcde"), 0, 0, 1);
    test(S("abcde"), 1, 1, S("abcde"), 0, 1, 1);
    test(S("abcde"), 1, 1, S("abcde"), 0, 2, 1);
    test(S("abcde"), 1, 1, S("abcde"), 0, 4, 1);
    test(S("abcde"), 1, 1, S("abcde"), 0, 5, 1);
    test(S("abcde"), 1, 1, S("abcde"), 0, 6, 1);
    test(S("abcde"), 1, 1, S("abcde"), 1, 0, 1);
    test(S("abcde"), 1, 1, S("abcde"), 1, 1, 0);
    test(S("abcde"), 1, 1, S("abcde"), 1, 2, -1);
    test(S("abcde"), 1, 1, S("abcde"), 1, 3, -2);
    test(S("abcde"), 1, 1, S("abcde"), 1, 4, -3);
    test(S("abcde"), 1, 1, S("abcde"), 1, 5, -3);
    test(S("abcde"), 1, 1, S("abcde"), 2, 0, 1);
    test(S("abcde"), 1, 1, S("abcde"), 2, 1, -1);
    test(S("abcde"), 1, 1, S("abcde"), 2, 2, -1);
    test(S("abcde"), 1, 1, S("abcde"), 2, 3, -1);
    test(S("abcde"), 1, 1, S("abcde"), 2, 4, -1);
    test(S("abcde"), 1, 1, S("abcde"), 4, 0, 1);
    test(S("abcde"), 1, 1, S("abcde"), 4, 1, -3);
    test(S("abcde"), 1, 1, S("abcde"), 4, 2, -3);
    test(S("abcde"), 1, 1, S("abcde"), 5, 0, 1);
    test(S("abcde"), 1, 1, S("abcde"), 5, 1, 1);
    test(S("abcde"), 1, 1, S("abcde"), 6, 0, 0);
    test(S("abcde"), 1, 1, S("abcdefghij"), 0, 0, 1);
    test(S("abcde"), 1, 1, S("abcdefghij"), 0, 1, 1);
    test(S("abcde"), 1, 1, S("abcdefghij"), 0, 5, 1);
    test(S("abcde"), 1, 1, S("abcdefghij"), 0, 9, 1);
    test(S("abcde"), 1, 1, S("abcdefghij"), 0, 10, 1);
    test(S("abcde"), 1, 1, S("abcdefghij"), 0, 11, 1);
    test(S("abcde"), 1, 1, S("abcdefghij"), 1, 0, 1);
    test(S("abcde"), 1, 1, S("abcdefghij"), 1, 1, 0);
    test(S("abcde"), 1, 1, S("abcdefghij"), 1, 4, -3);
    test(S("abcde"), 1, 1, S("abcdefghij"), 1, 8, -7);
    test(S("abcde"), 1, 1, S("abcdefghij"), 1, 9, -8);
    test(S("abcde"), 1, 1, S("abcdefghij"), 1, 10, -8);
    test(S("abcde"), 1, 1, S("abcdefghij"), 5, 0, 1);
    test(S("abcde"), 1, 1, S("abcdefghij"), 5, 1, -4);
    test(S("abcde"), 1, 1, S("abcdefghij"), 5, 2, -4);
    test(S("abcde"), 1, 1, S("abcdefghij"), 5, 4, -4);
    test(S("abcde"), 1, 1, S("abcdefghij"), 5, 5, -4);
    test(S("abcde"), 1, 1, S("abcdefghij"), 5, 6, -4);
    test(S("abcde"), 1, 1, S("abcdefghij"), 9, 0, 1);
    test(S("abcde"), 1, 1, S("abcdefghij"), 9, 1, -8);
    test(S("abcde"), 1, 1, S("abcdefghij"), 9, 2, -8);
    test(S("abcde"), 1, 1, S("abcdefghij"), 10, 0, 1);
    test(S("abcde"), 1, 1, S("abcdefghij"), 10, 1, 1);
    test(S("abcde"), 1, 1, S("abcdefghij"), 11, 0, 0);
    test(S("abcde"), 1, 1, S("abcdefghijklmnopqrst"), 0, 0, 1);
    test(S("abcde"), 1, 1, S("abcdefghijklmnopqrst"), 0, 1, 1);
    test(S("abcde"), 1, 1, S("abcdefghijklmnopqrst"), 0, 10, 1);
    test(S("abcde"), 1, 1, S("abcdefghijklmnopqrst"), 0, 19, 1);
    test(S("abcde"), 1, 1, S("abcdefghijklmnopqrst"), 0, 20, 1);
    test(S("abcde"), 1, 1, S("abcdefghijklmnopqrst"), 0, 21, 1);
    test(S("abcde"), 1, 1, S("abcdefghijklmnopqrst"), 1, 0, 1);
    test(S("abcde"), 1, 1, S("abcdefghijklmnopqrst"), 1, 1, 0);
    test(S("abcde"), 1, 1, S("abcdefghijklmnopqrst"), 1, 9, -8);
    test(S("abcde"), 1, 1, S("abcdefghijklmnopqrst"), 1, 18, -17);
}

template <class S>
void test8()
{
    test(S("abcde"), 1, 1, S("abcdefghijklmnopqrst"), 1, 19, -18);
    test(S("abcde"), 1, 1, S("abcdefghijklmnopqrst"), 1, 20, -18);
    test(S("abcde"), 1, 1, S("abcdefghijklmnopqrst"), 10, 0, 1);
    test(S("abcde"), 1, 1, S("abcdefghijklmnopqrst"), 10, 1, -9);
    test(S("abcde"), 1, 1, S("abcdefghijklmnopqrst"), 10, 5, -9);
    test(S("abcde"), 1, 1, S("abcdefghijklmnopqrst"), 10, 9, -9);
    test(S("abcde"), 1, 1, S("abcdefghijklmnopqrst"), 10, 10, -9);
    test(S("abcde"), 1, 1, S("abcdefghijklmnopqrst"), 10, 11, -9);
    test(S("abcde"), 1, 1, S("abcdefghijklmnopqrst"), 19, 0, 1);
    test(S("abcde"), 1, 1, S("abcdefghijklmnopqrst"), 19, 1, -18);
    test(S("abcde"), 1, 1, S("abcdefghijklmnopqrst"), 19, 2, -18);
    test(S("abcde"), 1, 1, S("abcdefghijklmnopqrst"), 20, 0, 1);
    test(S("abcde"), 1, 1, S("abcdefghijklmnopqrst"), 20, 1, 1);
    test(S("abcde"), 1, 1, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcde"), 1, 2, S(""), 0, 0, 2);
    test(S("abcde"), 1, 2, S(""), 0, 1, 2);
    test(S("abcde"), 1, 2, S(""), 1, 0, 0);
    test(S("abcde"), 1, 2, S("abcde"), 0, 0, 2);
    test(S("abcde"), 1, 2, S("abcde"), 0, 1, 1);
    test(S("abcde"), 1, 2, S("abcde"), 0, 2, 1);
    test(S("abcde"), 1, 2, S("abcde"), 0, 4, 1);
    test(S("abcde"), 1, 2, S("abcde"), 0, 5, 1);
    test(S("abcde"), 1, 2, S("abcde"), 0, 6, 1);
    test(S("abcde"), 1, 2, S("abcde"), 1, 0, 2);
    test(S("abcde"), 1, 2, S("abcde"), 1, 1, 1);
    test(S("abcde"), 1, 2, S("abcde"), 1, 2, 0);
    test(S("abcde"), 1, 2, S("abcde"), 1, 3, -1);
    test(S("abcde"), 1, 2, S("abcde"), 1, 4, -2);
    test(S("abcde"), 1, 2, S("abcde"), 1, 5, -2);
    test(S("abcde"), 1, 2, S("abcde"), 2, 0, 2);
    test(S("abcde"), 1, 2, S("abcde"), 2, 1, -1);
    test(S("abcde"), 1, 2, S("abcde"), 2, 2, -1);
    test(S("abcde"), 1, 2, S("abcde"), 2, 3, -1);
    test(S("abcde"), 1, 2, S("abcde"), 2, 4, -1);
    test(S("abcde"), 1, 2, S("abcde"), 4, 0, 2);
    test(S("abcde"), 1, 2, S("abcde"), 4, 1, -3);
    test(S("abcde"), 1, 2, S("abcde"), 4, 2, -3);
    test(S("abcde"), 1, 2, S("abcde"), 5, 0, 2);
    test(S("abcde"), 1, 2, S("abcde"), 5, 1, 2);
    test(S("abcde"), 1, 2, S("abcde"), 6, 0, 0);
    test(S("abcde"), 1, 2, S("abcdefghij"), 0, 0, 2);
    test(S("abcde"), 1, 2, S("abcdefghij"), 0, 1, 1);
    test(S("abcde"), 1, 2, S("abcdefghij"), 0, 5, 1);
    test(S("abcde"), 1, 2, S("abcdefghij"), 0, 9, 1);
    test(S("abcde"), 1, 2, S("abcdefghij"), 0, 10, 1);
    test(S("abcde"), 1, 2, S("abcdefghij"), 0, 11, 1);
    test(S("abcde"), 1, 2, S("abcdefghij"), 1, 0, 2);
    test(S("abcde"), 1, 2, S("abcdefghij"), 1, 1, 1);
    test(S("abcde"), 1, 2, S("abcdefghij"), 1, 4, -2);
    test(S("abcde"), 1, 2, S("abcdefghij"), 1, 8, -6);
    test(S("abcde"), 1, 2, S("abcdefghij"), 1, 9, -7);
    test(S("abcde"), 1, 2, S("abcdefghij"), 1, 10, -7);
    test(S("abcde"), 1, 2, S("abcdefghij"), 5, 0, 2);
    test(S("abcde"), 1, 2, S("abcdefghij"), 5, 1, -4);
    test(S("abcde"), 1, 2, S("abcdefghij"), 5, 2, -4);
    test(S("abcde"), 1, 2, S("abcdefghij"), 5, 4, -4);
    test(S("abcde"), 1, 2, S("abcdefghij"), 5, 5, -4);
    test(S("abcde"), 1, 2, S("abcdefghij"), 5, 6, -4);
    test(S("abcde"), 1, 2, S("abcdefghij"), 9, 0, 2);
    test(S("abcde"), 1, 2, S("abcdefghij"), 9, 1, -8);
    test(S("abcde"), 1, 2, S("abcdefghij"), 9, 2, -8);
    test(S("abcde"), 1, 2, S("abcdefghij"), 10, 0, 2);
    test(S("abcde"), 1, 2, S("abcdefghij"), 10, 1, 2);
    test(S("abcde"), 1, 2, S("abcdefghij"), 11, 0, 0);
    test(S("abcde"), 1, 2, S("abcdefghijklmnopqrst"), 0, 0, 2);
    test(S("abcde"), 1, 2, S("abcdefghijklmnopqrst"), 0, 1, 1);
    test(S("abcde"), 1, 2, S("abcdefghijklmnopqrst"), 0, 10, 1);
    test(S("abcde"), 1, 2, S("abcdefghijklmnopqrst"), 0, 19, 1);
    test(S("abcde"), 1, 2, S("abcdefghijklmnopqrst"), 0, 20, 1);
    test(S("abcde"), 1, 2, S("abcdefghijklmnopqrst"), 0, 21, 1);
    test(S("abcde"), 1, 2, S("abcdefghijklmnopqrst"), 1, 0, 2);
    test(S("abcde"), 1, 2, S("abcdefghijklmnopqrst"), 1, 1, 1);
    test(S("abcde"), 1, 2, S("abcdefghijklmnopqrst"), 1, 9, -7);
    test(S("abcde"), 1, 2, S("abcdefghijklmnopqrst"), 1, 18, -16);
    test(S("abcde"), 1, 2, S("abcdefghijklmnopqrst"), 1, 19, -17);
    test(S("abcde"), 1, 2, S("abcdefghijklmnopqrst"), 1, 20, -17);
    test(S("abcde"), 1, 2, S("abcdefghijklmnopqrst"), 10, 0, 2);
    test(S("abcde"), 1, 2, S("abcdefghijklmnopqrst"), 10, 1, -9);
    test(S("abcde"), 1, 2, S("abcdefghijklmnopqrst"), 10, 5, -9);
    test(S("abcde"), 1, 2, S("abcdefghijklmnopqrst"), 10, 9, -9);
    test(S("abcde"), 1, 2, S("abcdefghijklmnopqrst"), 10, 10, -9);
    test(S("abcde"), 1, 2, S("abcdefghijklmnopqrst"), 10, 11, -9);
    test(S("abcde"), 1, 2, S("abcdefghijklmnopqrst"), 19, 0, 2);
    test(S("abcde"), 1, 2, S("abcdefghijklmnopqrst"), 19, 1, -18);
    test(S("abcde"), 1, 2, S("abcdefghijklmnopqrst"), 19, 2, -18);
    test(S("abcde"), 1, 2, S("abcdefghijklmnopqrst"), 20, 0, 2);
    test(S("abcde"), 1, 2, S("abcdefghijklmnopqrst"), 20, 1, 2);
    test(S("abcde"), 1, 2, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcde"), 1, 3, S(""), 0, 0, 3);
    test(S("abcde"), 1, 3, S(""), 0, 1, 3);
    test(S("abcde"), 1, 3, S(""), 1, 0, 0);
    test(S("abcde"), 1, 3, S("abcde"), 0, 0, 3);
    test(S("abcde"), 1, 3, S("abcde"), 0, 1, 1);
    test(S("abcde"), 1, 3, S("abcde"), 0, 2, 1);
    test(S("abcde"), 1, 3, S("abcde"), 0, 4, 1);
    test(S("abcde"), 1, 3, S("abcde"), 0, 5, 1);
    test(S("abcde"), 1, 3, S("abcde"), 0, 6, 1);
    test(S("abcde"), 1, 3, S("abcde"), 1, 0, 3);
    test(S("abcde"), 1, 3, S("abcde"), 1, 1, 2);
    test(S("abcde"), 1, 3, S("abcde"), 1, 2, 1);
}

template <class S>
void test9()
{
    test(S("abcde"), 1, 3, S("abcde"), 1, 3, 0);
    test(S("abcde"), 1, 3, S("abcde"), 1, 4, -1);
    test(S("abcde"), 1, 3, S("abcde"), 1, 5, -1);
    test(S("abcde"), 1, 3, S("abcde"), 2, 0, 3);
    test(S("abcde"), 1, 3, S("abcde"), 2, 1, -1);
    test(S("abcde"), 1, 3, S("abcde"), 2, 2, -1);
    test(S("abcde"), 1, 3, S("abcde"), 2, 3, -1);
    test(S("abcde"), 1, 3, S("abcde"), 2, 4, -1);
    test(S("abcde"), 1, 3, S("abcde"), 4, 0, 3);
    test(S("abcde"), 1, 3, S("abcde"), 4, 1, -3);
    test(S("abcde"), 1, 3, S("abcde"), 4, 2, -3);
    test(S("abcde"), 1, 3, S("abcde"), 5, 0, 3);
    test(S("abcde"), 1, 3, S("abcde"), 5, 1, 3);
    test(S("abcde"), 1, 3, S("abcde"), 6, 0, 0);
    test(S("abcde"), 1, 3, S("abcdefghij"), 0, 0, 3);
    test(S("abcde"), 1, 3, S("abcdefghij"), 0, 1, 1);
    test(S("abcde"), 1, 3, S("abcdefghij"), 0, 5, 1);
    test(S("abcde"), 1, 3, S("abcdefghij"), 0, 9, 1);
    test(S("abcde"), 1, 3, S("abcdefghij"), 0, 10, 1);
    test(S("abcde"), 1, 3, S("abcdefghij"), 0, 11, 1);
    test(S("abcde"), 1, 3, S("abcdefghij"), 1, 0, 3);
    test(S("abcde"), 1, 3, S("abcdefghij"), 1, 1, 2);
    test(S("abcde"), 1, 3, S("abcdefghij"), 1, 4, -1);
    test(S("abcde"), 1, 3, S("abcdefghij"), 1, 8, -5);
    test(S("abcde"), 1, 3, S("abcdefghij"), 1, 9, -6);
    test(S("abcde"), 1, 3, S("abcdefghij"), 1, 10, -6);
    test(S("abcde"), 1, 3, S("abcdefghij"), 5, 0, 3);
    test(S("abcde"), 1, 3, S("abcdefghij"), 5, 1, -4);
    test(S("abcde"), 1, 3, S("abcdefghij"), 5, 2, -4);
    test(S("abcde"), 1, 3, S("abcdefghij"), 5, 4, -4);
    test(S("abcde"), 1, 3, S("abcdefghij"), 5, 5, -4);
    test(S("abcde"), 1, 3, S("abcdefghij"), 5, 6, -4);
    test(S("abcde"), 1, 3, S("abcdefghij"), 9, 0, 3);
    test(S("abcde"), 1, 3, S("abcdefghij"), 9, 1, -8);
    test(S("abcde"), 1, 3, S("abcdefghij"), 9, 2, -8);
    test(S("abcde"), 1, 3, S("abcdefghij"), 10, 0, 3);
    test(S("abcde"), 1, 3, S("abcdefghij"), 10, 1, 3);
    test(S("abcde"), 1, 3, S("abcdefghij"), 11, 0, 0);
    test(S("abcde"), 1, 3, S("abcdefghijklmnopqrst"), 0, 0, 3);
    test(S("abcde"), 1, 3, S("abcdefghijklmnopqrst"), 0, 1, 1);
    test(S("abcde"), 1, 3, S("abcdefghijklmnopqrst"), 0, 10, 1);
    test(S("abcde"), 1, 3, S("abcdefghijklmnopqrst"), 0, 19, 1);
    test(S("abcde"), 1, 3, S("abcdefghijklmnopqrst"), 0, 20, 1);
    test(S("abcde"), 1, 3, S("abcdefghijklmnopqrst"), 0, 21, 1);
    test(S("abcde"), 1, 3, S("abcdefghijklmnopqrst"), 1, 0, 3);
    test(S("abcde"), 1, 3, S("abcdefghijklmnopqrst"), 1, 1, 2);
    test(S("abcde"), 1, 3, S("abcdefghijklmnopqrst"), 1, 9, -6);
    test(S("abcde"), 1, 3, S("abcdefghijklmnopqrst"), 1, 18, -15);
    test(S("abcde"), 1, 3, S("abcdefghijklmnopqrst"), 1, 19, -16);
    test(S("abcde"), 1, 3, S("abcdefghijklmnopqrst"), 1, 20, -16);
    test(S("abcde"), 1, 3, S("abcdefghijklmnopqrst"), 10, 0, 3);
    test(S("abcde"), 1, 3, S("abcdefghijklmnopqrst"), 10, 1, -9);
    test(S("abcde"), 1, 3, S("abcdefghijklmnopqrst"), 10, 5, -9);
    test(S("abcde"), 1, 3, S("abcdefghijklmnopqrst"), 10, 9, -9);
    test(S("abcde"), 1, 3, S("abcdefghijklmnopqrst"), 10, 10, -9);
    test(S("abcde"), 1, 3, S("abcdefghijklmnopqrst"), 10, 11, -9);
    test(S("abcde"), 1, 3, S("abcdefghijklmnopqrst"), 19, 0, 3);
    test(S("abcde"), 1, 3, S("abcdefghijklmnopqrst"), 19, 1, -18);
    test(S("abcde"), 1, 3, S("abcdefghijklmnopqrst"), 19, 2, -18);
    test(S("abcde"), 1, 3, S("abcdefghijklmnopqrst"), 20, 0, 3);
    test(S("abcde"), 1, 3, S("abcdefghijklmnopqrst"), 20, 1, 3);
    test(S("abcde"), 1, 3, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcde"), 1, 4, S(""), 0, 0, 4);
    test(S("abcde"), 1, 4, S(""), 0, 1, 4);
    test(S("abcde"), 1, 4, S(""), 1, 0, 0);
    test(S("abcde"), 1, 4, S("abcde"), 0, 0, 4);
    test(S("abcde"), 1, 4, S("abcde"), 0, 1, 1);
    test(S("abcde"), 1, 4, S("abcde"), 0, 2, 1);
    test(S("abcde"), 1, 4, S("abcde"), 0, 4, 1);
    test(S("abcde"), 1, 4, S("abcde"), 0, 5, 1);
    test(S("abcde"), 1, 4, S("abcde"), 0, 6, 1);
    test(S("abcde"), 1, 4, S("abcde"), 1, 0, 4);
    test(S("abcde"), 1, 4, S("abcde"), 1, 1, 3);
    test(S("abcde"), 1, 4, S("abcde"), 1, 2, 2);
    test(S("abcde"), 1, 4, S("abcde"), 1, 3, 1);
    test(S("abcde"), 1, 4, S("abcde"), 1, 4, 0);
    test(S("abcde"), 1, 4, S("abcde"), 1, 5, 0);
    test(S("abcde"), 1, 4, S("abcde"), 2, 0, 4);
    test(S("abcde"), 1, 4, S("abcde"), 2, 1, -1);
    test(S("abcde"), 1, 4, S("abcde"), 2, 2, -1);
    test(S("abcde"), 1, 4, S("abcde"), 2, 3, -1);
    test(S("abcde"), 1, 4, S("abcde"), 2, 4, -1);
    test(S("abcde"), 1, 4, S("abcde"), 4, 0, 4);
    test(S("abcde"), 1, 4, S("abcde"), 4, 1, -3);
    test(S("abcde"), 1, 4, S("abcde"), 4, 2, -3);
    test(S("abcde"), 1, 4, S("abcde"), 5, 0, 4);
    test(S("abcde"), 1, 4, S("abcde"), 5, 1, 4);
    test(S("abcde"), 1, 4, S("abcde"), 6, 0, 0);
    test(S("abcde"), 1, 4, S("abcdefghij"), 0, 0, 4);
    test(S("abcde"), 1, 4, S("abcdefghij"), 0, 1, 1);
    test(S("abcde"), 1, 4, S("abcdefghij"), 0, 5, 1);
    test(S("abcde"), 1, 4, S("abcdefghij"), 0, 9, 1);
    test(S("abcde"), 1, 4, S("abcdefghij"), 0, 10, 1);
    test(S("abcde"), 1, 4, S("abcdefghij"), 0, 11, 1);
    test(S("abcde"), 1, 4, S("abcdefghij"), 1, 0, 4);
    test(S("abcde"), 1, 4, S("abcdefghij"), 1, 1, 3);
    test(S("abcde"), 1, 4, S("abcdefghij"), 1, 4, 0);
    test(S("abcde"), 1, 4, S("abcdefghij"), 1, 8, -4);
    test(S("abcde"), 1, 4, S("abcdefghij"), 1, 9, -5);
    test(S("abcde"), 1, 4, S("abcdefghij"), 1, 10, -5);
}

template <class S>
void test10()
{
    test(S("abcde"), 1, 4, S("abcdefghij"), 5, 0, 4);
    test(S("abcde"), 1, 4, S("abcdefghij"), 5, 1, -4);
    test(S("abcde"), 1, 4, S("abcdefghij"), 5, 2, -4);
    test(S("abcde"), 1, 4, S("abcdefghij"), 5, 4, -4);
    test(S("abcde"), 1, 4, S("abcdefghij"), 5, 5, -4);
    test(S("abcde"), 1, 4, S("abcdefghij"), 5, 6, -4);
    test(S("abcde"), 1, 4, S("abcdefghij"), 9, 0, 4);
    test(S("abcde"), 1, 4, S("abcdefghij"), 9, 1, -8);
    test(S("abcde"), 1, 4, S("abcdefghij"), 9, 2, -8);
    test(S("abcde"), 1, 4, S("abcdefghij"), 10, 0, 4);
    test(S("abcde"), 1, 4, S("abcdefghij"), 10, 1, 4);
    test(S("abcde"), 1, 4, S("abcdefghij"), 11, 0, 0);
    test(S("abcde"), 1, 4, S("abcdefghijklmnopqrst"), 0, 0, 4);
    test(S("abcde"), 1, 4, S("abcdefghijklmnopqrst"), 0, 1, 1);
    test(S("abcde"), 1, 4, S("abcdefghijklmnopqrst"), 0, 10, 1);
    test(S("abcde"), 1, 4, S("abcdefghijklmnopqrst"), 0, 19, 1);
    test(S("abcde"), 1, 4, S("abcdefghijklmnopqrst"), 0, 20, 1);
    test(S("abcde"), 1, 4, S("abcdefghijklmnopqrst"), 0, 21, 1);
    test(S("abcde"), 1, 4, S("abcdefghijklmnopqrst"), 1, 0, 4);
    test(S("abcde"), 1, 4, S("abcdefghijklmnopqrst"), 1, 1, 3);
    test(S("abcde"), 1, 4, S("abcdefghijklmnopqrst"), 1, 9, -5);
    test(S("abcde"), 1, 4, S("abcdefghijklmnopqrst"), 1, 18, -14);
    test(S("abcde"), 1, 4, S("abcdefghijklmnopqrst"), 1, 19, -15);
    test(S("abcde"), 1, 4, S("abcdefghijklmnopqrst"), 1, 20, -15);
    test(S("abcde"), 1, 4, S("abcdefghijklmnopqrst"), 10, 0, 4);
    test(S("abcde"), 1, 4, S("abcdefghijklmnopqrst"), 10, 1, -9);
    test(S("abcde"), 1, 4, S("abcdefghijklmnopqrst"), 10, 5, -9);
    test(S("abcde"), 1, 4, S("abcdefghijklmnopqrst"), 10, 9, -9);
    test(S("abcde"), 1, 4, S("abcdefghijklmnopqrst"), 10, 10, -9);
    test(S("abcde"), 1, 4, S("abcdefghijklmnopqrst"), 10, 11, -9);
    test(S("abcde"), 1, 4, S("abcdefghijklmnopqrst"), 19, 0, 4);
    test(S("abcde"), 1, 4, S("abcdefghijklmnopqrst"), 19, 1, -18);
    test(S("abcde"), 1, 4, S("abcdefghijklmnopqrst"), 19, 2, -18);
    test(S("abcde"), 1, 4, S("abcdefghijklmnopqrst"), 20, 0, 4);
    test(S("abcde"), 1, 4, S("abcdefghijklmnopqrst"), 20, 1, 4);
    test(S("abcde"), 1, 4, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcde"), 1, 5, S(""), 0, 0, 4);
    test(S("abcde"), 1, 5, S(""), 0, 1, 4);
    test(S("abcde"), 1, 5, S(""), 1, 0, 0);
    test(S("abcde"), 1, 5, S("abcde"), 0, 0, 4);
    test(S("abcde"), 1, 5, S("abcde"), 0, 1, 1);
    test(S("abcde"), 1, 5, S("abcde"), 0, 2, 1);
    test(S("abcde"), 1, 5, S("abcde"), 0, 4, 1);
    test(S("abcde"), 1, 5, S("abcde"), 0, 5, 1);
    test(S("abcde"), 1, 5, S("abcde"), 0, 6, 1);
    test(S("abcde"), 1, 5, S("abcde"), 1, 0, 4);
    test(S("abcde"), 1, 5, S("abcde"), 1, 1, 3);
    test(S("abcde"), 1, 5, S("abcde"), 1, 2, 2);
    test(S("abcde"), 1, 5, S("abcde"), 1, 3, 1);
    test(S("abcde"), 1, 5, S("abcde"), 1, 4, 0);
    test(S("abcde"), 1, 5, S("abcde"), 1, 5, 0);
    test(S("abcde"), 1, 5, S("abcde"), 2, 0, 4);
    test(S("abcde"), 1, 5, S("abcde"), 2, 1, -1);
    test(S("abcde"), 1, 5, S("abcde"), 2, 2, -1);
    test(S("abcde"), 1, 5, S("abcde"), 2, 3, -1);
    test(S("abcde"), 1, 5, S("abcde"), 2, 4, -1);
    test(S("abcde"), 1, 5, S("abcde"), 4, 0, 4);
    test(S("abcde"), 1, 5, S("abcde"), 4, 1, -3);
    test(S("abcde"), 1, 5, S("abcde"), 4, 2, -3);
    test(S("abcde"), 1, 5, S("abcde"), 5, 0, 4);
    test(S("abcde"), 1, 5, S("abcde"), 5, 1, 4);
    test(S("abcde"), 1, 5, S("abcde"), 6, 0, 0);
    test(S("abcde"), 1, 5, S("abcdefghij"), 0, 0, 4);
    test(S("abcde"), 1, 5, S("abcdefghij"), 0, 1, 1);
    test(S("abcde"), 1, 5, S("abcdefghij"), 0, 5, 1);
    test(S("abcde"), 1, 5, S("abcdefghij"), 0, 9, 1);
    test(S("abcde"), 1, 5, S("abcdefghij"), 0, 10, 1);
    test(S("abcde"), 1, 5, S("abcdefghij"), 0, 11, 1);
    test(S("abcde"), 1, 5, S("abcdefghij"), 1, 0, 4);
    test(S("abcde"), 1, 5, S("abcdefghij"), 1, 1, 3);
    test(S("abcde"), 1, 5, S("abcdefghij"), 1, 4, 0);
    test(S("abcde"), 1, 5, S("abcdefghij"), 1, 8, -4);
    test(S("abcde"), 1, 5, S("abcdefghij"), 1, 9, -5);
    test(S("abcde"), 1, 5, S("abcdefghij"), 1, 10, -5);
    test(S("abcde"), 1, 5, S("abcdefghij"), 5, 0, 4);
    test(S("abcde"), 1, 5, S("abcdefghij"), 5, 1, -4);
    test(S("abcde"), 1, 5, S("abcdefghij"), 5, 2, -4);
    test(S("abcde"), 1, 5, S("abcdefghij"), 5, 4, -4);
    test(S("abcde"), 1, 5, S("abcdefghij"), 5, 5, -4);
    test(S("abcde"), 1, 5, S("abcdefghij"), 5, 6, -4);
    test(S("abcde"), 1, 5, S("abcdefghij"), 9, 0, 4);
    test(S("abcde"), 1, 5, S("abcdefghij"), 9, 1, -8);
    test(S("abcde"), 1, 5, S("abcdefghij"), 9, 2, -8);
    test(S("abcde"), 1, 5, S("abcdefghij"), 10, 0, 4);
    test(S("abcde"), 1, 5, S("abcdefghij"), 10, 1, 4);
    test(S("abcde"), 1, 5, S("abcdefghij"), 11, 0, 0);
    test(S("abcde"), 1, 5, S("abcdefghijklmnopqrst"), 0, 0, 4);
    test(S("abcde"), 1, 5, S("abcdefghijklmnopqrst"), 0, 1, 1);
    test(S("abcde"), 1, 5, S("abcdefghijklmnopqrst"), 0, 10, 1);
    test(S("abcde"), 1, 5, S("abcdefghijklmnopqrst"), 0, 19, 1);
    test(S("abcde"), 1, 5, S("abcdefghijklmnopqrst"), 0, 20, 1);
    test(S("abcde"), 1, 5, S("abcdefghijklmnopqrst"), 0, 21, 1);
    test(S("abcde"), 1, 5, S("abcdefghijklmnopqrst"), 1, 0, 4);
    test(S("abcde"), 1, 5, S("abcdefghijklmnopqrst"), 1, 1, 3);
    test(S("abcde"), 1, 5, S("abcdefghijklmnopqrst"), 1, 9, -5);
    test(S("abcde"), 1, 5, S("abcdefghijklmnopqrst"), 1, 18, -14);
    test(S("abcde"), 1, 5, S("abcdefghijklmnopqrst"), 1, 19, -15);
    test(S("abcde"), 1, 5, S("abcdefghijklmnopqrst"), 1, 20, -15);
    test(S("abcde"), 1, 5, S("abcdefghijklmnopqrst"), 10, 0, 4);
    test(S("abcde"), 1, 5, S("abcdefghijklmnopqrst"), 10, 1, -9);
}

template <class S>
void test11()
{
    test(S("abcde"), 1, 5, S("abcdefghijklmnopqrst"), 10, 5, -9);
    test(S("abcde"), 1, 5, S("abcdefghijklmnopqrst"), 10, 9, -9);
    test(S("abcde"), 1, 5, S("abcdefghijklmnopqrst"), 10, 10, -9);
    test(S("abcde"), 1, 5, S("abcdefghijklmnopqrst"), 10, 11, -9);
    test(S("abcde"), 1, 5, S("abcdefghijklmnopqrst"), 19, 0, 4);
    test(S("abcde"), 1, 5, S("abcdefghijklmnopqrst"), 19, 1, -18);
    test(S("abcde"), 1, 5, S("abcdefghijklmnopqrst"), 19, 2, -18);
    test(S("abcde"), 1, 5, S("abcdefghijklmnopqrst"), 20, 0, 4);
    test(S("abcde"), 1, 5, S("abcdefghijklmnopqrst"), 20, 1, 4);
    test(S("abcde"), 1, 5, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcde"), 2, 0, S(""), 0, 0, 0);
    test(S("abcde"), 2, 0, S(""), 0, 1, 0);
    test(S("abcde"), 2, 0, S(""), 1, 0, 0);
    test(S("abcde"), 2, 0, S("abcde"), 0, 0, 0);
    test(S("abcde"), 2, 0, S("abcde"), 0, 1, -1);
    test(S("abcde"), 2, 0, S("abcde"), 0, 2, -2);
    test(S("abcde"), 2, 0, S("abcde"), 0, 4, -4);
    test(S("abcde"), 2, 0, S("abcde"), 0, 5, -5);
    test(S("abcde"), 2, 0, S("abcde"), 0, 6, -5);
    test(S("abcde"), 2, 0, S("abcde"), 1, 0, 0);
    test(S("abcde"), 2, 0, S("abcde"), 1, 1, -1);
    test(S("abcde"), 2, 0, S("abcde"), 1, 2, -2);
    test(S("abcde"), 2, 0, S("abcde"), 1, 3, -3);
    test(S("abcde"), 2, 0, S("abcde"), 1, 4, -4);
    test(S("abcde"), 2, 0, S("abcde"), 1, 5, -4);
    test(S("abcde"), 2, 0, S("abcde"), 2, 0, 0);
    test(S("abcde"), 2, 0, S("abcde"), 2, 1, -1);
    test(S("abcde"), 2, 0, S("abcde"), 2, 2, -2);
    test(S("abcde"), 2, 0, S("abcde"), 2, 3, -3);
    test(S("abcde"), 2, 0, S("abcde"), 2, 4, -3);
    test(S("abcde"), 2, 0, S("abcde"), 4, 0, 0);
    test(S("abcde"), 2, 0, S("abcde"), 4, 1, -1);
    test(S("abcde"), 2, 0, S("abcde"), 4, 2, -1);
    test(S("abcde"), 2, 0, S("abcde"), 5, 0, 0);
    test(S("abcde"), 2, 0, S("abcde"), 5, 1, 0);
    test(S("abcde"), 2, 0, S("abcde"), 6, 0, 0);
    test(S("abcde"), 2, 0, S("abcdefghij"), 0, 0, 0);
    test(S("abcde"), 2, 0, S("abcdefghij"), 0, 1, -1);
    test(S("abcde"), 2, 0, S("abcdefghij"), 0, 5, -5);
    test(S("abcde"), 2, 0, S("abcdefghij"), 0, 9, -9);
    test(S("abcde"), 2, 0, S("abcdefghij"), 0, 10, -10);
    test(S("abcde"), 2, 0, S("abcdefghij"), 0, 11, -10);
    test(S("abcde"), 2, 0, S("abcdefghij"), 1, 0, 0);
    test(S("abcde"), 2, 0, S("abcdefghij"), 1, 1, -1);
    test(S("abcde"), 2, 0, S("abcdefghij"), 1, 4, -4);
    test(S("abcde"), 2, 0, S("abcdefghij"), 1, 8, -8);
    test(S("abcde"), 2, 0, S("abcdefghij"), 1, 9, -9);
    test(S("abcde"), 2, 0, S("abcdefghij"), 1, 10, -9);
    test(S("abcde"), 2, 0, S("abcdefghij"), 5, 0, 0);
    test(S("abcde"), 2, 0, S("abcdefghij"), 5, 1, -1);
    test(S("abcde"), 2, 0, S("abcdefghij"), 5, 2, -2);
    test(S("abcde"), 2, 0, S("abcdefghij"), 5, 4, -4);
    test(S("abcde"), 2, 0, S("abcdefghij"), 5, 5, -5);
    test(S("abcde"), 2, 0, S("abcdefghij"), 5, 6, -5);
    test(S("abcde"), 2, 0, S("abcdefghij"), 9, 0, 0);
    test(S("abcde"), 2, 0, S("abcdefghij"), 9, 1, -1);
    test(S("abcde"), 2, 0, S("abcdefghij"), 9, 2, -1);
    test(S("abcde"), 2, 0, S("abcdefghij"), 10, 0, 0);
    test(S("abcde"), 2, 0, S("abcdefghij"), 10, 1, 0);
    test(S("abcde"), 2, 0, S("abcdefghij"), 11, 0, 0);
    test(S("abcde"), 2, 0, S("abcdefghijklmnopqrst"), 0, 0, 0);
    test(S("abcde"), 2, 0, S("abcdefghijklmnopqrst"), 0, 1, -1);
    test(S("abcde"), 2, 0, S("abcdefghijklmnopqrst"), 0, 10, -10);
    test(S("abcde"), 2, 0, S("abcdefghijklmnopqrst"), 0, 19, -19);
    test(S("abcde"), 2, 0, S("abcdefghijklmnopqrst"), 0, 20, -20);
    test(S("abcde"), 2, 0, S("abcdefghijklmnopqrst"), 0, 21, -20);
    test(S("abcde"), 2, 0, S("abcdefghijklmnopqrst"), 1, 0, 0);
    test(S("abcde"), 2, 0, S("abcdefghijklmnopqrst"), 1, 1, -1);
    test(S("abcde"), 2, 0, S("abcdefghijklmnopqrst"), 1, 9, -9);
    test(S("abcde"), 2, 0, S("abcdefghijklmnopqrst"), 1, 18, -18);
    test(S("abcde"), 2, 0, S("abcdefghijklmnopqrst"), 1, 19, -19);
    test(S("abcde"), 2, 0, S("abcdefghijklmnopqrst"), 1, 20, -19);
    test(S("abcde"), 2, 0, S("abcdefghijklmnopqrst"), 10, 0, 0);
    test(S("abcde"), 2, 0, S("abcdefghijklmnopqrst"), 10, 1, -1);
    test(S("abcde"), 2, 0, S("abcdefghijklmnopqrst"), 10, 5, -5);
    test(S("abcde"), 2, 0, S("abcdefghijklmnopqrst"), 10, 9, -9);
    test(S("abcde"), 2, 0, S("abcdefghijklmnopqrst"), 10, 10, -10);
    test(S("abcde"), 2, 0, S("abcdefghijklmnopqrst"), 10, 11, -10);
    test(S("abcde"), 2, 0, S("abcdefghijklmnopqrst"), 19, 0, 0);
    test(S("abcde"), 2, 0, S("abcdefghijklmnopqrst"), 19, 1, -1);
    test(S("abcde"), 2, 0, S("abcdefghijklmnopqrst"), 19, 2, -1);
    test(S("abcde"), 2, 0, S("abcdefghijklmnopqrst"), 20, 0, 0);
    test(S("abcde"), 2, 0, S("abcdefghijklmnopqrst"), 20, 1, 0);
    test(S("abcde"), 2, 0, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcde"), 2, 1, S(""), 0, 0, 1);
    test(S("abcde"), 2, 1, S(""), 0, 1, 1);
    test(S("abcde"), 2, 1, S(""), 1, 0, 0);
    test(S("abcde"), 2, 1, S("abcde"), 0, 0, 1);
    test(S("abcde"), 2, 1, S("abcde"), 0, 1, 2);
    test(S("abcde"), 2, 1, S("abcde"), 0, 2, 2);
    test(S("abcde"), 2, 1, S("abcde"), 0, 4, 2);
    test(S("abcde"), 2, 1, S("abcde"), 0, 5, 2);
    test(S("abcde"), 2, 1, S("abcde"), 0, 6, 2);
    test(S("abcde"), 2, 1, S("abcde"), 1, 0, 1);
    test(S("abcde"), 2, 1, S("abcde"), 1, 1, 1);
    test(S("abcde"), 2, 1, S("abcde"), 1, 2, 1);
    test(S("abcde"), 2, 1, S("abcde"), 1, 3, 1);
    test(S("abcde"), 2, 1, S("abcde"), 1, 4, 1);
    test(S("abcde"), 2, 1, S("abcde"), 1, 5, 1);
    test(S("abcde"), 2, 1, S("abcde"), 2, 0, 1);
}

template <class S>
void test12()
{
    test(S("abcde"), 2, 1, S("abcde"), 2, 1, 0);
    test(S("abcde"), 2, 1, S("abcde"), 2, 2, -1);
    test(S("abcde"), 2, 1, S("abcde"), 2, 3, -2);
    test(S("abcde"), 2, 1, S("abcde"), 2, 4, -2);
    test(S("abcde"), 2, 1, S("abcde"), 4, 0, 1);
    test(S("abcde"), 2, 1, S("abcde"), 4, 1, -2);
    test(S("abcde"), 2, 1, S("abcde"), 4, 2, -2);
    test(S("abcde"), 2, 1, S("abcde"), 5, 0, 1);
    test(S("abcde"), 2, 1, S("abcde"), 5, 1, 1);
    test(S("abcde"), 2, 1, S("abcde"), 6, 0, 0);
    test(S("abcde"), 2, 1, S("abcdefghij"), 0, 0, 1);
    test(S("abcde"), 2, 1, S("abcdefghij"), 0, 1, 2);
    test(S("abcde"), 2, 1, S("abcdefghij"), 0, 5, 2);
    test(S("abcde"), 2, 1, S("abcdefghij"), 0, 9, 2);
    test(S("abcde"), 2, 1, S("abcdefghij"), 0, 10, 2);
    test(S("abcde"), 2, 1, S("abcdefghij"), 0, 11, 2);
    test(S("abcde"), 2, 1, S("abcdefghij"), 1, 0, 1);
    test(S("abcde"), 2, 1, S("abcdefghij"), 1, 1, 1);
    test(S("abcde"), 2, 1, S("abcdefghij"), 1, 4, 1);
    test(S("abcde"), 2, 1, S("abcdefghij"), 1, 8, 1);
    test(S("abcde"), 2, 1, S("abcdefghij"), 1, 9, 1);
    test(S("abcde"), 2, 1, S("abcdefghij"), 1, 10, 1);
    test(S("abcde"), 2, 1, S("abcdefghij"), 5, 0, 1);
    test(S("abcde"), 2, 1, S("abcdefghij"), 5, 1, -3);
    test(S("abcde"), 2, 1, S("abcdefghij"), 5, 2, -3);
    test(S("abcde"), 2, 1, S("abcdefghij"), 5, 4, -3);
    test(S("abcde"), 2, 1, S("abcdefghij"), 5, 5, -3);
    test(S("abcde"), 2, 1, S("abcdefghij"), 5, 6, -3);
    test(S("abcde"), 2, 1, S("abcdefghij"), 9, 0, 1);
    test(S("abcde"), 2, 1, S("abcdefghij"), 9, 1, -7);
    test(S("abcde"), 2, 1, S("abcdefghij"), 9, 2, -7);
    test(S("abcde"), 2, 1, S("abcdefghij"), 10, 0, 1);
    test(S("abcde"), 2, 1, S("abcdefghij"), 10, 1, 1);
    test(S("abcde"), 2, 1, S("abcdefghij"), 11, 0, 0);
    test(S("abcde"), 2, 1, S("abcdefghijklmnopqrst"), 0, 0, 1);
    test(S("abcde"), 2, 1, S("abcdefghijklmnopqrst"), 0, 1, 2);
    test(S("abcde"), 2, 1, S("abcdefghijklmnopqrst"), 0, 10, 2);
    test(S("abcde"), 2, 1, S("abcdefghijklmnopqrst"), 0, 19, 2);
    test(S("abcde"), 2, 1, S("abcdefghijklmnopqrst"), 0, 20, 2);
    test(S("abcde"), 2, 1, S("abcdefghijklmnopqrst"), 0, 21, 2);
    test(S("abcde"), 2, 1, S("abcdefghijklmnopqrst"), 1, 0, 1);
    test(S("abcde"), 2, 1, S("abcdefghijklmnopqrst"), 1, 1, 1);
    test(S("abcde"), 2, 1, S("abcdefghijklmnopqrst"), 1, 9, 1);
    test(S("abcde"), 2, 1, S("abcdefghijklmnopqrst"), 1, 18, 1);
    test(S("abcde"), 2, 1, S("abcdefghijklmnopqrst"), 1, 19, 1);
    test(S("abcde"), 2, 1, S("abcdefghijklmnopqrst"), 1, 20, 1);
    test(S("abcde"), 2, 1, S("abcdefghijklmnopqrst"), 10, 0, 1);
    test(S("abcde"), 2, 1, S("abcdefghijklmnopqrst"), 10, 1, -8);
    test(S("abcde"), 2, 1, S("abcdefghijklmnopqrst"), 10, 5, -8);
    test(S("abcde"), 2, 1, S("abcdefghijklmnopqrst"), 10, 9, -8);
    test(S("abcde"), 2, 1, S("abcdefghijklmnopqrst"), 10, 10, -8);
    test(S("abcde"), 2, 1, S("abcdefghijklmnopqrst"), 10, 11, -8);
    test(S("abcde"), 2, 1, S("abcdefghijklmnopqrst"), 19, 0, 1);
    test(S("abcde"), 2, 1, S("abcdefghijklmnopqrst"), 19, 1, -17);
    test(S("abcde"), 2, 1, S("abcdefghijklmnopqrst"), 19, 2, -17);
    test(S("abcde"), 2, 1, S("abcdefghijklmnopqrst"), 20, 0, 1);
    test(S("abcde"), 2, 1, S("abcdefghijklmnopqrst"), 20, 1, 1);
    test(S("abcde"), 2, 1, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcde"), 2, 2, S(""), 0, 0, 2);
    test(S("abcde"), 2, 2, S(""), 0, 1, 2);
    test(S("abcde"), 2, 2, S(""), 1, 0, 0);
    test(S("abcde"), 2, 2, S("abcde"), 0, 0, 2);
    test(S("abcde"), 2, 2, S("abcde"), 0, 1, 2);
    test(S("abcde"), 2, 2, S("abcde"), 0, 2, 2);
    test(S("abcde"), 2, 2, S("abcde"), 0, 4, 2);
    test(S("abcde"), 2, 2, S("abcde"), 0, 5, 2);
    test(S("abcde"), 2, 2, S("abcde"), 0, 6, 2);
    test(S("abcde"), 2, 2, S("abcde"), 1, 0, 2);
    test(S("abcde"), 2, 2, S("abcde"), 1, 1, 1);
    test(S("abcde"), 2, 2, S("abcde"), 1, 2, 1);
    test(S("abcde"), 2, 2, S("abcde"), 1, 3, 1);
    test(S("abcde"), 2, 2, S("abcde"), 1, 4, 1);
    test(S("abcde"), 2, 2, S("abcde"), 1, 5, 1);
    test(S("abcde"), 2, 2, S("abcde"), 2, 0, 2);
    test(S("abcde"), 2, 2, S("abcde"), 2, 1, 1);
    test(S("abcde"), 2, 2, S("abcde"), 2, 2, 0);
    test(S("abcde"), 2, 2, S("abcde"), 2, 3, -1);
    test(S("abcde"), 2, 2, S("abcde"), 2, 4, -1);
    test(S("abcde"), 2, 2, S("abcde"), 4, 0, 2);
    test(S("abcde"), 2, 2, S("abcde"), 4, 1, -2);
    test(S("abcde"), 2, 2, S("abcde"), 4, 2, -2);
    test(S("abcde"), 2, 2, S("abcde"), 5, 0, 2);
    test(S("abcde"), 2, 2, S("abcde"), 5, 1, 2);
    test(S("abcde"), 2, 2, S("abcde"), 6, 0, 0);
    test(S("abcde"), 2, 2, S("abcdefghij"), 0, 0, 2);
    test(S("abcde"), 2, 2, S("abcdefghij"), 0, 1, 2);
    test(S("abcde"), 2, 2, S("abcdefghij"), 0, 5, 2);
    test(S("abcde"), 2, 2, S("abcdefghij"), 0, 9, 2);
    test(S("abcde"), 2, 2, S("abcdefghij"), 0, 10, 2);
    test(S("abcde"), 2, 2, S("abcdefghij"), 0, 11, 2);
    test(S("abcde"), 2, 2, S("abcdefghij"), 1, 0, 2);
    test(S("abcde"), 2, 2, S("abcdefghij"), 1, 1, 1);
    test(S("abcde"), 2, 2, S("abcdefghij"), 1, 4, 1);
    test(S("abcde"), 2, 2, S("abcdefghij"), 1, 8, 1);
    test(S("abcde"), 2, 2, S("abcdefghij"), 1, 9, 1);
    test(S("abcde"), 2, 2, S("abcdefghij"), 1, 10, 1);
    test(S("abcde"), 2, 2, S("abcdefghij"), 5, 0, 2);
    test(S("abcde"), 2, 2, S("abcdefghij"), 5, 1, -3);
    test(S("abcde"), 2, 2, S("abcdefghij"), 5, 2, -3);
    test(S("abcde"), 2, 2, S("abcdefghij"), 5, 4, -3);
}

template <class S>
void test13()
{
    test(S("abcde"), 2, 2, S("abcdefghij"), 5, 5, -3);
    test(S("abcde"), 2, 2, S("abcdefghij"), 5, 6, -3);
    test(S("abcde"), 2, 2, S("abcdefghij"), 9, 0, 2);
    test(S("abcde"), 2, 2, S("abcdefghij"), 9, 1, -7);
    test(S("abcde"), 2, 2, S("abcdefghij"), 9, 2, -7);
    test(S("abcde"), 2, 2, S("abcdefghij"), 10, 0, 2);
    test(S("abcde"), 2, 2, S("abcdefghij"), 10, 1, 2);
    test(S("abcde"), 2, 2, S("abcdefghij"), 11, 0, 0);
    test(S("abcde"), 2, 2, S("abcdefghijklmnopqrst"), 0, 0, 2);
    test(S("abcde"), 2, 2, S("abcdefghijklmnopqrst"), 0, 1, 2);
    test(S("abcde"), 2, 2, S("abcdefghijklmnopqrst"), 0, 10, 2);
    test(S("abcde"), 2, 2, S("abcdefghijklmnopqrst"), 0, 19, 2);
    test(S("abcde"), 2, 2, S("abcdefghijklmnopqrst"), 0, 20, 2);
    test(S("abcde"), 2, 2, S("abcdefghijklmnopqrst"), 0, 21, 2);
    test(S("abcde"), 2, 2, S("abcdefghijklmnopqrst"), 1, 0, 2);
    test(S("abcde"), 2, 2, S("abcdefghijklmnopqrst"), 1, 1, 1);
    test(S("abcde"), 2, 2, S("abcdefghijklmnopqrst"), 1, 9, 1);
    test(S("abcde"), 2, 2, S("abcdefghijklmnopqrst"), 1, 18, 1);
    test(S("abcde"), 2, 2, S("abcdefghijklmnopqrst"), 1, 19, 1);
    test(S("abcde"), 2, 2, S("abcdefghijklmnopqrst"), 1, 20, 1);
    test(S("abcde"), 2, 2, S("abcdefghijklmnopqrst"), 10, 0, 2);
    test(S("abcde"), 2, 2, S("abcdefghijklmnopqrst"), 10, 1, -8);
    test(S("abcde"), 2, 2, S("abcdefghijklmnopqrst"), 10, 5, -8);
    test(S("abcde"), 2, 2, S("abcdefghijklmnopqrst"), 10, 9, -8);
    test(S("abcde"), 2, 2, S("abcdefghijklmnopqrst"), 10, 10, -8);
    test(S("abcde"), 2, 2, S("abcdefghijklmnopqrst"), 10, 11, -8);
    test(S("abcde"), 2, 2, S("abcdefghijklmnopqrst"), 19, 0, 2);
    test(S("abcde"), 2, 2, S("abcdefghijklmnopqrst"), 19, 1, -17);
    test(S("abcde"), 2, 2, S("abcdefghijklmnopqrst"), 19, 2, -17);
    test(S("abcde"), 2, 2, S("abcdefghijklmnopqrst"), 20, 0, 2);
    test(S("abcde"), 2, 2, S("abcdefghijklmnopqrst"), 20, 1, 2);
    test(S("abcde"), 2, 2, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcde"), 2, 3, S(""), 0, 0, 3);
    test(S("abcde"), 2, 3, S(""), 0, 1, 3);
    test(S("abcde"), 2, 3, S(""), 1, 0, 0);
    test(S("abcde"), 2, 3, S("abcde"), 0, 0, 3);
    test(S("abcde"), 2, 3, S("abcde"), 0, 1, 2);
    test(S("abcde"), 2, 3, S("abcde"), 0, 2, 2);
    test(S("abcde"), 2, 3, S("abcde"), 0, 4, 2);
    test(S("abcde"), 2, 3, S("abcde"), 0, 5, 2);
    test(S("abcde"), 2, 3, S("abcde"), 0, 6, 2);
    test(S("abcde"), 2, 3, S("abcde"), 1, 0, 3);
    test(S("abcde"), 2, 3, S("abcde"), 1, 1, 1);
    test(S("abcde"), 2, 3, S("abcde"), 1, 2, 1);
    test(S("abcde"), 2, 3, S("abcde"), 1, 3, 1);
    test(S("abcde"), 2, 3, S("abcde"), 1, 4, 1);
    test(S("abcde"), 2, 3, S("abcde"), 1, 5, 1);
    test(S("abcde"), 2, 3, S("abcde"), 2, 0, 3);
    test(S("abcde"), 2, 3, S("abcde"), 2, 1, 2);
    test(S("abcde"), 2, 3, S("abcde"), 2, 2, 1);
    test(S("abcde"), 2, 3, S("abcde"), 2, 3, 0);
    test(S("abcde"), 2, 3, S("abcde"), 2, 4, 0);
    test(S("abcde"), 2, 3, S("abcde"), 4, 0, 3);
    test(S("abcde"), 2, 3, S("abcde"), 4, 1, -2);
    test(S("abcde"), 2, 3, S("abcde"), 4, 2, -2);
    test(S("abcde"), 2, 3, S("abcde"), 5, 0, 3);
    test(S("abcde"), 2, 3, S("abcde"), 5, 1, 3);
    test(S("abcde"), 2, 3, S("abcde"), 6, 0, 0);
    test(S("abcde"), 2, 3, S("abcdefghij"), 0, 0, 3);
    test(S("abcde"), 2, 3, S("abcdefghij"), 0, 1, 2);
    test(S("abcde"), 2, 3, S("abcdefghij"), 0, 5, 2);
    test(S("abcde"), 2, 3, S("abcdefghij"), 0, 9, 2);
    test(S("abcde"), 2, 3, S("abcdefghij"), 0, 10, 2);
    test(S("abcde"), 2, 3, S("abcdefghij"), 0, 11, 2);
    test(S("abcde"), 2, 3, S("abcdefghij"), 1, 0, 3);
    test(S("abcde"), 2, 3, S("abcdefghij"), 1, 1, 1);
    test(S("abcde"), 2, 3, S("abcdefghij"), 1, 4, 1);
    test(S("abcde"), 2, 3, S("abcdefghij"), 1, 8, 1);
    test(S("abcde"), 2, 3, S("abcdefghij"), 1, 9, 1);
    test(S("abcde"), 2, 3, S("abcdefghij"), 1, 10, 1);
    test(S("abcde"), 2, 3, S("abcdefghij"), 5, 0, 3);
    test(S("abcde"), 2, 3, S("abcdefghij"), 5, 1, -3);
    test(S("abcde"), 2, 3, S("abcdefghij"), 5, 2, -3);
    test(S("abcde"), 2, 3, S("abcdefghij"), 5, 4, -3);
    test(S("abcde"), 2, 3, S("abcdefghij"), 5, 5, -3);
    test(S("abcde"), 2, 3, S("abcdefghij"), 5, 6, -3);
    test(S("abcde"), 2, 3, S("abcdefghij"), 9, 0, 3);
    test(S("abcde"), 2, 3, S("abcdefghij"), 9, 1, -7);
    test(S("abcde"), 2, 3, S("abcdefghij"), 9, 2, -7);
    test(S("abcde"), 2, 3, S("abcdefghij"), 10, 0, 3);
    test(S("abcde"), 2, 3, S("abcdefghij"), 10, 1, 3);
    test(S("abcde"), 2, 3, S("abcdefghij"), 11, 0, 0);
    test(S("abcde"), 2, 3, S("abcdefghijklmnopqrst"), 0, 0, 3);
    test(S("abcde"), 2, 3, S("abcdefghijklmnopqrst"), 0, 1, 2);
    test(S("abcde"), 2, 3, S("abcdefghijklmnopqrst"), 0, 10, 2);
    test(S("abcde"), 2, 3, S("abcdefghijklmnopqrst"), 0, 19, 2);
    test(S("abcde"), 2, 3, S("abcdefghijklmnopqrst"), 0, 20, 2);
    test(S("abcde"), 2, 3, S("abcdefghijklmnopqrst"), 0, 21, 2);
    test(S("abcde"), 2, 3, S("abcdefghijklmnopqrst"), 1, 0, 3);
    test(S("abcde"), 2, 3, S("abcdefghijklmnopqrst"), 1, 1, 1);
    test(S("abcde"), 2, 3, S("abcdefghijklmnopqrst"), 1, 9, 1);
    test(S("abcde"), 2, 3, S("abcdefghijklmnopqrst"), 1, 18, 1);
    test(S("abcde"), 2, 3, S("abcdefghijklmnopqrst"), 1, 19, 1);
    test(S("abcde"), 2, 3, S("abcdefghijklmnopqrst"), 1, 20, 1);
    test(S("abcde"), 2, 3, S("abcdefghijklmnopqrst"), 10, 0, 3);
    test(S("abcde"), 2, 3, S("abcdefghijklmnopqrst"), 10, 1, -8);
    test(S("abcde"), 2, 3, S("abcdefghijklmnopqrst"), 10, 5, -8);
    test(S("abcde"), 2, 3, S("abcdefghijklmnopqrst"), 10, 9, -8);
    test(S("abcde"), 2, 3, S("abcdefghijklmnopqrst"), 10, 10, -8);
    test(S("abcde"), 2, 3, S("abcdefghijklmnopqrst"), 10, 11, -8);
}

template <class S>
void test14()
{
    test(S("abcde"), 2, 3, S("abcdefghijklmnopqrst"), 19, 0, 3);
    test(S("abcde"), 2, 3, S("abcdefghijklmnopqrst"), 19, 1, -17);
    test(S("abcde"), 2, 3, S("abcdefghijklmnopqrst"), 19, 2, -17);
    test(S("abcde"), 2, 3, S("abcdefghijklmnopqrst"), 20, 0, 3);
    test(S("abcde"), 2, 3, S("abcdefghijklmnopqrst"), 20, 1, 3);
    test(S("abcde"), 2, 3, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcde"), 2, 4, S(""), 0, 0, 3);
    test(S("abcde"), 2, 4, S(""), 0, 1, 3);
    test(S("abcde"), 2, 4, S(""), 1, 0, 0);
    test(S("abcde"), 2, 4, S("abcde"), 0, 0, 3);
    test(S("abcde"), 2, 4, S("abcde"), 0, 1, 2);
    test(S("abcde"), 2, 4, S("abcde"), 0, 2, 2);
    test(S("abcde"), 2, 4, S("abcde"), 0, 4, 2);
    test(S("abcde"), 2, 4, S("abcde"), 0, 5, 2);
    test(S("abcde"), 2, 4, S("abcde"), 0, 6, 2);
    test(S("abcde"), 2, 4, S("abcde"), 1, 0, 3);
    test(S("abcde"), 2, 4, S("abcde"), 1, 1, 1);
    test(S("abcde"), 2, 4, S("abcde"), 1, 2, 1);
    test(S("abcde"), 2, 4, S("abcde"), 1, 3, 1);
    test(S("abcde"), 2, 4, S("abcde"), 1, 4, 1);
    test(S("abcde"), 2, 4, S("abcde"), 1, 5, 1);
    test(S("abcde"), 2, 4, S("abcde"), 2, 0, 3);
    test(S("abcde"), 2, 4, S("abcde"), 2, 1, 2);
    test(S("abcde"), 2, 4, S("abcde"), 2, 2, 1);
    test(S("abcde"), 2, 4, S("abcde"), 2, 3, 0);
    test(S("abcde"), 2, 4, S("abcde"), 2, 4, 0);
    test(S("abcde"), 2, 4, S("abcde"), 4, 0, 3);
    test(S("abcde"), 2, 4, S("abcde"), 4, 1, -2);
    test(S("abcde"), 2, 4, S("abcde"), 4, 2, -2);
    test(S("abcde"), 2, 4, S("abcde"), 5, 0, 3);
    test(S("abcde"), 2, 4, S("abcde"), 5, 1, 3);
    test(S("abcde"), 2, 4, S("abcde"), 6, 0, 0);
    test(S("abcde"), 2, 4, S("abcdefghij"), 0, 0, 3);
    test(S("abcde"), 2, 4, S("abcdefghij"), 0, 1, 2);
    test(S("abcde"), 2, 4, S("abcdefghij"), 0, 5, 2);
    test(S("abcde"), 2, 4, S("abcdefghij"), 0, 9, 2);
    test(S("abcde"), 2, 4, S("abcdefghij"), 0, 10, 2);
    test(S("abcde"), 2, 4, S("abcdefghij"), 0, 11, 2);
    test(S("abcde"), 2, 4, S("abcdefghij"), 1, 0, 3);
    test(S("abcde"), 2, 4, S("abcdefghij"), 1, 1, 1);
    test(S("abcde"), 2, 4, S("abcdefghij"), 1, 4, 1);
    test(S("abcde"), 2, 4, S("abcdefghij"), 1, 8, 1);
    test(S("abcde"), 2, 4, S("abcdefghij"), 1, 9, 1);
    test(S("abcde"), 2, 4, S("abcdefghij"), 1, 10, 1);
    test(S("abcde"), 2, 4, S("abcdefghij"), 5, 0, 3);
    test(S("abcde"), 2, 4, S("abcdefghij"), 5, 1, -3);
    test(S("abcde"), 2, 4, S("abcdefghij"), 5, 2, -3);
    test(S("abcde"), 2, 4, S("abcdefghij"), 5, 4, -3);
    test(S("abcde"), 2, 4, S("abcdefghij"), 5, 5, -3);
    test(S("abcde"), 2, 4, S("abcdefghij"), 5, 6, -3);
    test(S("abcde"), 2, 4, S("abcdefghij"), 9, 0, 3);
    test(S("abcde"), 2, 4, S("abcdefghij"), 9, 1, -7);
    test(S("abcde"), 2, 4, S("abcdefghij"), 9, 2, -7);
    test(S("abcde"), 2, 4, S("abcdefghij"), 10, 0, 3);
    test(S("abcde"), 2, 4, S("abcdefghij"), 10, 1, 3);
    test(S("abcde"), 2, 4, S("abcdefghij"), 11, 0, 0);
    test(S("abcde"), 2, 4, S("abcdefghijklmnopqrst"), 0, 0, 3);
    test(S("abcde"), 2, 4, S("abcdefghijklmnopqrst"), 0, 1, 2);
    test(S("abcde"), 2, 4, S("abcdefghijklmnopqrst"), 0, 10, 2);
    test(S("abcde"), 2, 4, S("abcdefghijklmnopqrst"), 0, 19, 2);
    test(S("abcde"), 2, 4, S("abcdefghijklmnopqrst"), 0, 20, 2);
    test(S("abcde"), 2, 4, S("abcdefghijklmnopqrst"), 0, 21, 2);
    test(S("abcde"), 2, 4, S("abcdefghijklmnopqrst"), 1, 0, 3);
    test(S("abcde"), 2, 4, S("abcdefghijklmnopqrst"), 1, 1, 1);
    test(S("abcde"), 2, 4, S("abcdefghijklmnopqrst"), 1, 9, 1);
    test(S("abcde"), 2, 4, S("abcdefghijklmnopqrst"), 1, 18, 1);
    test(S("abcde"), 2, 4, S("abcdefghijklmnopqrst"), 1, 19, 1);
    test(S("abcde"), 2, 4, S("abcdefghijklmnopqrst"), 1, 20, 1);
    test(S("abcde"), 2, 4, S("abcdefghijklmnopqrst"), 10, 0, 3);
    test(S("abcde"), 2, 4, S("abcdefghijklmnopqrst"), 10, 1, -8);
    test(S("abcde"), 2, 4, S("abcdefghijklmnopqrst"), 10, 5, -8);
    test(S("abcde"), 2, 4, S("abcdefghijklmnopqrst"), 10, 9, -8);
    test(S("abcde"), 2, 4, S("abcdefghijklmnopqrst"), 10, 10, -8);
    test(S("abcde"), 2, 4, S("abcdefghijklmnopqrst"), 10, 11, -8);
    test(S("abcde"), 2, 4, S("abcdefghijklmnopqrst"), 19, 0, 3);
    test(S("abcde"), 2, 4, S("abcdefghijklmnopqrst"), 19, 1, -17);
    test(S("abcde"), 2, 4, S("abcdefghijklmnopqrst"), 19, 2, -17);
    test(S("abcde"), 2, 4, S("abcdefghijklmnopqrst"), 20, 0, 3);
    test(S("abcde"), 2, 4, S("abcdefghijklmnopqrst"), 20, 1, 3);
    test(S("abcde"), 2, 4, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcde"), 4, 0, S(""), 0, 0, 0);
    test(S("abcde"), 4, 0, S(""), 0, 1, 0);
    test(S("abcde"), 4, 0, S(""), 1, 0, 0);
    test(S("abcde"), 4, 0, S("abcde"), 0, 0, 0);
    test(S("abcde"), 4, 0, S("abcde"), 0, 1, -1);
    test(S("abcde"), 4, 0, S("abcde"), 0, 2, -2);
    test(S("abcde"), 4, 0, S("abcde"), 0, 4, -4);
    test(S("abcde"), 4, 0, S("abcde"), 0, 5, -5);
    test(S("abcde"), 4, 0, S("abcde"), 0, 6, -5);
    test(S("abcde"), 4, 0, S("abcde"), 1, 0, 0);
    test(S("abcde"), 4, 0, S("abcde"), 1, 1, -1);
    test(S("abcde"), 4, 0, S("abcde"), 1, 2, -2);
    test(S("abcde"), 4, 0, S("abcde"), 1, 3, -3);
    test(S("abcde"), 4, 0, S("abcde"), 1, 4, -4);
    test(S("abcde"), 4, 0, S("abcde"), 1, 5, -4);
    test(S("abcde"), 4, 0, S("abcde"), 2, 0, 0);
    test(S("abcde"), 4, 0, S("abcde"), 2, 1, -1);
    test(S("abcde"), 4, 0, S("abcde"), 2, 2, -2);
    test(S("abcde"), 4, 0, S("abcde"), 2, 3, -3);
    test(S("abcde"), 4, 0, S("abcde"), 2, 4, -3);
}

template <class S>
void test15()
{
    test(S("abcde"), 4, 0, S("abcde"), 4, 0, 0);
    test(S("abcde"), 4, 0, S("abcde"), 4, 1, -1);
    test(S("abcde"), 4, 0, S("abcde"), 4, 2, -1);
    test(S("abcde"), 4, 0, S("abcde"), 5, 0, 0);
    test(S("abcde"), 4, 0, S("abcde"), 5, 1, 0);
    test(S("abcde"), 4, 0, S("abcde"), 6, 0, 0);
    test(S("abcde"), 4, 0, S("abcdefghij"), 0, 0, 0);
    test(S("abcde"), 4, 0, S("abcdefghij"), 0, 1, -1);
    test(S("abcde"), 4, 0, S("abcdefghij"), 0, 5, -5);
    test(S("abcde"), 4, 0, S("abcdefghij"), 0, 9, -9);
    test(S("abcde"), 4, 0, S("abcdefghij"), 0, 10, -10);
    test(S("abcde"), 4, 0, S("abcdefghij"), 0, 11, -10);
    test(S("abcde"), 4, 0, S("abcdefghij"), 1, 0, 0);
    test(S("abcde"), 4, 0, S("abcdefghij"), 1, 1, -1);
    test(S("abcde"), 4, 0, S("abcdefghij"), 1, 4, -4);
    test(S("abcde"), 4, 0, S("abcdefghij"), 1, 8, -8);
    test(S("abcde"), 4, 0, S("abcdefghij"), 1, 9, -9);
    test(S("abcde"), 4, 0, S("abcdefghij"), 1, 10, -9);
    test(S("abcde"), 4, 0, S("abcdefghij"), 5, 0, 0);
    test(S("abcde"), 4, 0, S("abcdefghij"), 5, 1, -1);
    test(S("abcde"), 4, 0, S("abcdefghij"), 5, 2, -2);
    test(S("abcde"), 4, 0, S("abcdefghij"), 5, 4, -4);
    test(S("abcde"), 4, 0, S("abcdefghij"), 5, 5, -5);
    test(S("abcde"), 4, 0, S("abcdefghij"), 5, 6, -5);
    test(S("abcde"), 4, 0, S("abcdefghij"), 9, 0, 0);
    test(S("abcde"), 4, 0, S("abcdefghij"), 9, 1, -1);
    test(S("abcde"), 4, 0, S("abcdefghij"), 9, 2, -1);
    test(S("abcde"), 4, 0, S("abcdefghij"), 10, 0, 0);
    test(S("abcde"), 4, 0, S("abcdefghij"), 10, 1, 0);
    test(S("abcde"), 4, 0, S("abcdefghij"), 11, 0, 0);
    test(S("abcde"), 4, 0, S("abcdefghijklmnopqrst"), 0, 0, 0);
    test(S("abcde"), 4, 0, S("abcdefghijklmnopqrst"), 0, 1, -1);
    test(S("abcde"), 4, 0, S("abcdefghijklmnopqrst"), 0, 10, -10);
    test(S("abcde"), 4, 0, S("abcdefghijklmnopqrst"), 0, 19, -19);
    test(S("abcde"), 4, 0, S("abcdefghijklmnopqrst"), 0, 20, -20);
    test(S("abcde"), 4, 0, S("abcdefghijklmnopqrst"), 0, 21, -20);
    test(S("abcde"), 4, 0, S("abcdefghijklmnopqrst"), 1, 0, 0);
    test(S("abcde"), 4, 0, S("abcdefghijklmnopqrst"), 1, 1, -1);
    test(S("abcde"), 4, 0, S("abcdefghijklmnopqrst"), 1, 9, -9);
    test(S("abcde"), 4, 0, S("abcdefghijklmnopqrst"), 1, 18, -18);
    test(S("abcde"), 4, 0, S("abcdefghijklmnopqrst"), 1, 19, -19);
    test(S("abcde"), 4, 0, S("abcdefghijklmnopqrst"), 1, 20, -19);
    test(S("abcde"), 4, 0, S("abcdefghijklmnopqrst"), 10, 0, 0);
    test(S("abcde"), 4, 0, S("abcdefghijklmnopqrst"), 10, 1, -1);
    test(S("abcde"), 4, 0, S("abcdefghijklmnopqrst"), 10, 5, -5);
    test(S("abcde"), 4, 0, S("abcdefghijklmnopqrst"), 10, 9, -9);
    test(S("abcde"), 4, 0, S("abcdefghijklmnopqrst"), 10, 10, -10);
    test(S("abcde"), 4, 0, S("abcdefghijklmnopqrst"), 10, 11, -10);
    test(S("abcde"), 4, 0, S("abcdefghijklmnopqrst"), 19, 0, 0);
    test(S("abcde"), 4, 0, S("abcdefghijklmnopqrst"), 19, 1, -1);
    test(S("abcde"), 4, 0, S("abcdefghijklmnopqrst"), 19, 2, -1);
    test(S("abcde"), 4, 0, S("abcdefghijklmnopqrst"), 20, 0, 0);
    test(S("abcde"), 4, 0, S("abcdefghijklmnopqrst"), 20, 1, 0);
    test(S("abcde"), 4, 0, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcde"), 4, 1, S(""), 0, 0, 1);
    test(S("abcde"), 4, 1, S(""), 0, 1, 1);
    test(S("abcde"), 4, 1, S(""), 1, 0, 0);
    test(S("abcde"), 4, 1, S("abcde"), 0, 0, 1);
    test(S("abcde"), 4, 1, S("abcde"), 0, 1, 4);
    test(S("abcde"), 4, 1, S("abcde"), 0, 2, 4);
    test(S("abcde"), 4, 1, S("abcde"), 0, 4, 4);
    test(S("abcde"), 4, 1, S("abcde"), 0, 5, 4);
    test(S("abcde"), 4, 1, S("abcde"), 0, 6, 4);
    test(S("abcde"), 4, 1, S("abcde"), 1, 0, 1);
    test(S("abcde"), 4, 1, S("abcde"), 1, 1, 3);
    test(S("abcde"), 4, 1, S("abcde"), 1, 2, 3);
    test(S("abcde"), 4, 1, S("abcde"), 1, 3, 3);
    test(S("abcde"), 4, 1, S("abcde"), 1, 4, 3);
    test(S("abcde"), 4, 1, S("abcde"), 1, 5, 3);
    test(S("abcde"), 4, 1, S("abcde"), 2, 0, 1);
    test(S("abcde"), 4, 1, S("abcde"), 2, 1, 2);
    test(S("abcde"), 4, 1, S("abcde"), 2, 2, 2);
    test(S("abcde"), 4, 1, S("abcde"), 2, 3, 2);
    test(S("abcde"), 4, 1, S("abcde"), 2, 4, 2);
    test(S("abcde"), 4, 1, S("abcde"), 4, 0, 1);
    test(S("abcde"), 4, 1, S("abcde"), 4, 1, 0);
    test(S("abcde"), 4, 1, S("abcde"), 4, 2, 0);
    test(S("abcde"), 4, 1, S("abcde"), 5, 0, 1);
    test(S("abcde"), 4, 1, S("abcde"), 5, 1, 1);
    test(S("abcde"), 4, 1, S("abcde"), 6, 0, 0);
    test(S("abcde"), 4, 1, S("abcdefghij"), 0, 0, 1);
    test(S("abcde"), 4, 1, S("abcdefghij"), 0, 1, 4);
    test(S("abcde"), 4, 1, S("abcdefghij"), 0, 5, 4);
    test(S("abcde"), 4, 1, S("abcdefghij"), 0, 9, 4);
    test(S("abcde"), 4, 1, S("abcdefghij"), 0, 10, 4);
    test(S("abcde"), 4, 1, S("abcdefghij"), 0, 11, 4);
    test(S("abcde"), 4, 1, S("abcdefghij"), 1, 0, 1);
    test(S("abcde"), 4, 1, S("abcdefghij"), 1, 1, 3);
    test(S("abcde"), 4, 1, S("abcdefghij"), 1, 4, 3);
    test(S("abcde"), 4, 1, S("abcdefghij"), 1, 8, 3);
    test(S("abcde"), 4, 1, S("abcdefghij"), 1, 9, 3);
    test(S("abcde"), 4, 1, S("abcdefghij"), 1, 10, 3);
    test(S("abcde"), 4, 1, S("abcdefghij"), 5, 0, 1);
    test(S("abcde"), 4, 1, S("abcdefghij"), 5, 1, -1);
    test(S("abcde"), 4, 1, S("abcdefghij"), 5, 2, -1);
    test(S("abcde"), 4, 1, S("abcdefghij"), 5, 4, -1);
    test(S("abcde"), 4, 1, S("abcdefghij"), 5, 5, -1);
    test(S("abcde"), 4, 1, S("abcdefghij"), 5, 6, -1);
    test(S("abcde"), 4, 1, S("abcdefghij"), 9, 0, 1);
    test(S("abcde"), 4, 1, S("abcdefghij"), 9, 1, -5);
}

template <class S>
void test16()
{
    test(S("abcde"), 4, 1, S("abcdefghij"), 9, 2, -5);
    test(S("abcde"), 4, 1, S("abcdefghij"), 10, 0, 1);
    test(S("abcde"), 4, 1, S("abcdefghij"), 10, 1, 1);
    test(S("abcde"), 4, 1, S("abcdefghij"), 11, 0, 0);
    test(S("abcde"), 4, 1, S("abcdefghijklmnopqrst"), 0, 0, 1);
    test(S("abcde"), 4, 1, S("abcdefghijklmnopqrst"), 0, 1, 4);
    test(S("abcde"), 4, 1, S("abcdefghijklmnopqrst"), 0, 10, 4);
    test(S("abcde"), 4, 1, S("abcdefghijklmnopqrst"), 0, 19, 4);
    test(S("abcde"), 4, 1, S("abcdefghijklmnopqrst"), 0, 20, 4);
    test(S("abcde"), 4, 1, S("abcdefghijklmnopqrst"), 0, 21, 4);
    test(S("abcde"), 4, 1, S("abcdefghijklmnopqrst"), 1, 0, 1);
    test(S("abcde"), 4, 1, S("abcdefghijklmnopqrst"), 1, 1, 3);
    test(S("abcde"), 4, 1, S("abcdefghijklmnopqrst"), 1, 9, 3);
    test(S("abcde"), 4, 1, S("abcdefghijklmnopqrst"), 1, 18, 3);
    test(S("abcde"), 4, 1, S("abcdefghijklmnopqrst"), 1, 19, 3);
    test(S("abcde"), 4, 1, S("abcdefghijklmnopqrst"), 1, 20, 3);
    test(S("abcde"), 4, 1, S("abcdefghijklmnopqrst"), 10, 0, 1);
    test(S("abcde"), 4, 1, S("abcdefghijklmnopqrst"), 10, 1, -6);
    test(S("abcde"), 4, 1, S("abcdefghijklmnopqrst"), 10, 5, -6);
    test(S("abcde"), 4, 1, S("abcdefghijklmnopqrst"), 10, 9, -6);
    test(S("abcde"), 4, 1, S("abcdefghijklmnopqrst"), 10, 10, -6);
    test(S("abcde"), 4, 1, S("abcdefghijklmnopqrst"), 10, 11, -6);
    test(S("abcde"), 4, 1, S("abcdefghijklmnopqrst"), 19, 0, 1);
    test(S("abcde"), 4, 1, S("abcdefghijklmnopqrst"), 19, 1, -15);
    test(S("abcde"), 4, 1, S("abcdefghijklmnopqrst"), 19, 2, -15);
    test(S("abcde"), 4, 1, S("abcdefghijklmnopqrst"), 20, 0, 1);
    test(S("abcde"), 4, 1, S("abcdefghijklmnopqrst"), 20, 1, 1);
    test(S("abcde"), 4, 1, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcde"), 4, 2, S(""), 0, 0, 1);
    test(S("abcde"), 4, 2, S(""), 0, 1, 1);
    test(S("abcde"), 4, 2, S(""), 1, 0, 0);
    test(S("abcde"), 4, 2, S("abcde"), 0, 0, 1);
    test(S("abcde"), 4, 2, S("abcde"), 0, 1, 4);
    test(S("abcde"), 4, 2, S("abcde"), 0, 2, 4);
    test(S("abcde"), 4, 2, S("abcde"), 0, 4, 4);
    test(S("abcde"), 4, 2, S("abcde"), 0, 5, 4);
    test(S("abcde"), 4, 2, S("abcde"), 0, 6, 4);
    test(S("abcde"), 4, 2, S("abcde"), 1, 0, 1);
    test(S("abcde"), 4, 2, S("abcde"), 1, 1, 3);
    test(S("abcde"), 4, 2, S("abcde"), 1, 2, 3);
    test(S("abcde"), 4, 2, S("abcde"), 1, 3, 3);
    test(S("abcde"), 4, 2, S("abcde"), 1, 4, 3);
    test(S("abcde"), 4, 2, S("abcde"), 1, 5, 3);
    test(S("abcde"), 4, 2, S("abcde"), 2, 0, 1);
    test(S("abcde"), 4, 2, S("abcde"), 2, 1, 2);
    test(S("abcde"), 4, 2, S("abcde"), 2, 2, 2);
    test(S("abcde"), 4, 2, S("abcde"), 2, 3, 2);
    test(S("abcde"), 4, 2, S("abcde"), 2, 4, 2);
    test(S("abcde"), 4, 2, S("abcde"), 4, 0, 1);
    test(S("abcde"), 4, 2, S("abcde"), 4, 1, 0);
    test(S("abcde"), 4, 2, S("abcde"), 4, 2, 0);
    test(S("abcde"), 4, 2, S("abcde"), 5, 0, 1);
    test(S("abcde"), 4, 2, S("abcde"), 5, 1, 1);
    test(S("abcde"), 4, 2, S("abcde"), 6, 0, 0);
    test(S("abcde"), 4, 2, S("abcdefghij"), 0, 0, 1);
    test(S("abcde"), 4, 2, S("abcdefghij"), 0, 1, 4);
    test(S("abcde"), 4, 2, S("abcdefghij"), 0, 5, 4);
    test(S("abcde"), 4, 2, S("abcdefghij"), 0, 9, 4);
    test(S("abcde"), 4, 2, S("abcdefghij"), 0, 10, 4);
    test(S("abcde"), 4, 2, S("abcdefghij"), 0, 11, 4);
    test(S("abcde"), 4, 2, S("abcdefghij"), 1, 0, 1);
    test(S("abcde"), 4, 2, S("abcdefghij"), 1, 1, 3);
    test(S("abcde"), 4, 2, S("abcdefghij"), 1, 4, 3);
    test(S("abcde"), 4, 2, S("abcdefghij"), 1, 8, 3);
    test(S("abcde"), 4, 2, S("abcdefghij"), 1, 9, 3);
    test(S("abcde"), 4, 2, S("abcdefghij"), 1, 10, 3);
    test(S("abcde"), 4, 2, S("abcdefghij"), 5, 0, 1);
    test(S("abcde"), 4, 2, S("abcdefghij"), 5, 1, -1);
    test(S("abcde"), 4, 2, S("abcdefghij"), 5, 2, -1);
    test(S("abcde"), 4, 2, S("abcdefghij"), 5, 4, -1);
    test(S("abcde"), 4, 2, S("abcdefghij"), 5, 5, -1);
    test(S("abcde"), 4, 2, S("abcdefghij"), 5, 6, -1);
    test(S("abcde"), 4, 2, S("abcdefghij"), 9, 0, 1);
    test(S("abcde"), 4, 2, S("abcdefghij"), 9, 1, -5);
    test(S("abcde"), 4, 2, S("abcdefghij"), 9, 2, -5);
    test(S("abcde"), 4, 2, S("abcdefghij"), 10, 0, 1);
    test(S("abcde"), 4, 2, S("abcdefghij"), 10, 1, 1);
    test(S("abcde"), 4, 2, S("abcdefghij"), 11, 0, 0);
    test(S("abcde"), 4, 2, S("abcdefghijklmnopqrst"), 0, 0, 1);
    test(S("abcde"), 4, 2, S("abcdefghijklmnopqrst"), 0, 1, 4);
    test(S("abcde"), 4, 2, S("abcdefghijklmnopqrst"), 0, 10, 4);
    test(S("abcde"), 4, 2, S("abcdefghijklmnopqrst"), 0, 19, 4);
    test(S("abcde"), 4, 2, S("abcdefghijklmnopqrst"), 0, 20, 4);
    test(S("abcde"), 4, 2, S("abcdefghijklmnopqrst"), 0, 21, 4);
    test(S("abcde"), 4, 2, S("abcdefghijklmnopqrst"), 1, 0, 1);
    test(S("abcde"), 4, 2, S("abcdefghijklmnopqrst"), 1, 1, 3);
    test(S("abcde"), 4, 2, S("abcdefghijklmnopqrst"), 1, 9, 3);
    test(S("abcde"), 4, 2, S("abcdefghijklmnopqrst"), 1, 18, 3);
    test(S("abcde"), 4, 2, S("abcdefghijklmnopqrst"), 1, 19, 3);
    test(S("abcde"), 4, 2, S("abcdefghijklmnopqrst"), 1, 20, 3);
    test(S("abcde"), 4, 2, S("abcdefghijklmnopqrst"), 10, 0, 1);
    test(S("abcde"), 4, 2, S("abcdefghijklmnopqrst"), 10, 1, -6);
    test(S("abcde"), 4, 2, S("abcdefghijklmnopqrst"), 10, 5, -6);
    test(S("abcde"), 4, 2, S("abcdefghijklmnopqrst"), 10, 9, -6);
    test(S("abcde"), 4, 2, S("abcdefghijklmnopqrst"), 10, 10, -6);
    test(S("abcde"), 4, 2, S("abcdefghijklmnopqrst"), 10, 11, -6);
    test(S("abcde"), 4, 2, S("abcdefghijklmnopqrst"), 19, 0, 1);
    test(S("abcde"), 4, 2, S("abcdefghijklmnopqrst"), 19, 1, -15);
    test(S("abcde"), 4, 2, S("abcdefghijklmnopqrst"), 19, 2, -15);
    test(S("abcde"), 4, 2, S("abcdefghijklmnopqrst"), 20, 0, 1);
}

template <class S>
void test17()
{
    test(S("abcde"), 4, 2, S("abcdefghijklmnopqrst"), 20, 1, 1);
    test(S("abcde"), 4, 2, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcde"), 5, 0, S(""), 0, 0, 0);
    test(S("abcde"), 5, 0, S(""), 0, 1, 0);
    test(S("abcde"), 5, 0, S(""), 1, 0, 0);
    test(S("abcde"), 5, 0, S("abcde"), 0, 0, 0);
    test(S("abcde"), 5, 0, S("abcde"), 0, 1, -1);
    test(S("abcde"), 5, 0, S("abcde"), 0, 2, -2);
    test(S("abcde"), 5, 0, S("abcde"), 0, 4, -4);
    test(S("abcde"), 5, 0, S("abcde"), 0, 5, -5);
    test(S("abcde"), 5, 0, S("abcde"), 0, 6, -5);
    test(S("abcde"), 5, 0, S("abcde"), 1, 0, 0);
    test(S("abcde"), 5, 0, S("abcde"), 1, 1, -1);
    test(S("abcde"), 5, 0, S("abcde"), 1, 2, -2);
    test(S("abcde"), 5, 0, S("abcde"), 1, 3, -3);
    test(S("abcde"), 5, 0, S("abcde"), 1, 4, -4);
    test(S("abcde"), 5, 0, S("abcde"), 1, 5, -4);
    test(S("abcde"), 5, 0, S("abcde"), 2, 0, 0);
    test(S("abcde"), 5, 0, S("abcde"), 2, 1, -1);
    test(S("abcde"), 5, 0, S("abcde"), 2, 2, -2);
    test(S("abcde"), 5, 0, S("abcde"), 2, 3, -3);
    test(S("abcde"), 5, 0, S("abcde"), 2, 4, -3);
    test(S("abcde"), 5, 0, S("abcde"), 4, 0, 0);
    test(S("abcde"), 5, 0, S("abcde"), 4, 1, -1);
    test(S("abcde"), 5, 0, S("abcde"), 4, 2, -1);
    test(S("abcde"), 5, 0, S("abcde"), 5, 0, 0);
    test(S("abcde"), 5, 0, S("abcde"), 5, 1, 0);
    test(S("abcde"), 5, 0, S("abcde"), 6, 0, 0);
    test(S("abcde"), 5, 0, S("abcdefghij"), 0, 0, 0);
    test(S("abcde"), 5, 0, S("abcdefghij"), 0, 1, -1);
    test(S("abcde"), 5, 0, S("abcdefghij"), 0, 5, -5);
    test(S("abcde"), 5, 0, S("abcdefghij"), 0, 9, -9);
    test(S("abcde"), 5, 0, S("abcdefghij"), 0, 10, -10);
    test(S("abcde"), 5, 0, S("abcdefghij"), 0, 11, -10);
    test(S("abcde"), 5, 0, S("abcdefghij"), 1, 0, 0);
    test(S("abcde"), 5, 0, S("abcdefghij"), 1, 1, -1);
    test(S("abcde"), 5, 0, S("abcdefghij"), 1, 4, -4);
    test(S("abcde"), 5, 0, S("abcdefghij"), 1, 8, -8);
    test(S("abcde"), 5, 0, S("abcdefghij"), 1, 9, -9);
    test(S("abcde"), 5, 0, S("abcdefghij"), 1, 10, -9);
    test(S("abcde"), 5, 0, S("abcdefghij"), 5, 0, 0);
    test(S("abcde"), 5, 0, S("abcdefghij"), 5, 1, -1);
    test(S("abcde"), 5, 0, S("abcdefghij"), 5, 2, -2);
    test(S("abcde"), 5, 0, S("abcdefghij"), 5, 4, -4);
    test(S("abcde"), 5, 0, S("abcdefghij"), 5, 5, -5);
    test(S("abcde"), 5, 0, S("abcdefghij"), 5, 6, -5);
    test(S("abcde"), 5, 0, S("abcdefghij"), 9, 0, 0);
    test(S("abcde"), 5, 0, S("abcdefghij"), 9, 1, -1);
    test(S("abcde"), 5, 0, S("abcdefghij"), 9, 2, -1);
    test(S("abcde"), 5, 0, S("abcdefghij"), 10, 0, 0);
    test(S("abcde"), 5, 0, S("abcdefghij"), 10, 1, 0);
    test(S("abcde"), 5, 0, S("abcdefghij"), 11, 0, 0);
    test(S("abcde"), 5, 0, S("abcdefghijklmnopqrst"), 0, 0, 0);
    test(S("abcde"), 5, 0, S("abcdefghijklmnopqrst"), 0, 1, -1);
    test(S("abcde"), 5, 0, S("abcdefghijklmnopqrst"), 0, 10, -10);
    test(S("abcde"), 5, 0, S("abcdefghijklmnopqrst"), 0, 19, -19);
    test(S("abcde"), 5, 0, S("abcdefghijklmnopqrst"), 0, 20, -20);
    test(S("abcde"), 5, 0, S("abcdefghijklmnopqrst"), 0, 21, -20);
    test(S("abcde"), 5, 0, S("abcdefghijklmnopqrst"), 1, 0, 0);
    test(S("abcde"), 5, 0, S("abcdefghijklmnopqrst"), 1, 1, -1);
    test(S("abcde"), 5, 0, S("abcdefghijklmnopqrst"), 1, 9, -9);
    test(S("abcde"), 5, 0, S("abcdefghijklmnopqrst"), 1, 18, -18);
    test(S("abcde"), 5, 0, S("abcdefghijklmnopqrst"), 1, 19, -19);
    test(S("abcde"), 5, 0, S("abcdefghijklmnopqrst"), 1, 20, -19);
    test(S("abcde"), 5, 0, S("abcdefghijklmnopqrst"), 10, 0, 0);
    test(S("abcde"), 5, 0, S("abcdefghijklmnopqrst"), 10, 1, -1);
    test(S("abcde"), 5, 0, S("abcdefghijklmnopqrst"), 10, 5, -5);
    test(S("abcde"), 5, 0, S("abcdefghijklmnopqrst"), 10, 9, -9);
    test(S("abcde"), 5, 0, S("abcdefghijklmnopqrst"), 10, 10, -10);
    test(S("abcde"), 5, 0, S("abcdefghijklmnopqrst"), 10, 11, -10);
    test(S("abcde"), 5, 0, S("abcdefghijklmnopqrst"), 19, 0, 0);
    test(S("abcde"), 5, 0, S("abcdefghijklmnopqrst"), 19, 1, -1);
    test(S("abcde"), 5, 0, S("abcdefghijklmnopqrst"), 19, 2, -1);
    test(S("abcde"), 5, 0, S("abcdefghijklmnopqrst"), 20, 0, 0);
    test(S("abcde"), 5, 0, S("abcdefghijklmnopqrst"), 20, 1, 0);
    test(S("abcde"), 5, 0, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcde"), 5, 1, S(""), 0, 0, 0);
    test(S("abcde"), 5, 1, S(""), 0, 1, 0);
    test(S("abcde"), 5, 1, S(""), 1, 0, 0);
    test(S("abcde"), 5, 1, S("abcde"), 0, 0, 0);
    test(S("abcde"), 5, 1, S("abcde"), 0, 1, -1);
    test(S("abcde"), 5, 1, S("abcde"), 0, 2, -2);
    test(S("abcde"), 5, 1, S("abcde"), 0, 4, -4);
    test(S("abcde"), 5, 1, S("abcde"), 0, 5, -5);
    test(S("abcde"), 5, 1, S("abcde"), 0, 6, -5);
    test(S("abcde"), 5, 1, S("abcde"), 1, 0, 0);
    test(S("abcde"), 5, 1, S("abcde"), 1, 1, -1);
    test(S("abcde"), 5, 1, S("abcde"), 1, 2, -2);
    test(S("abcde"), 5, 1, S("abcde"), 1, 3, -3);
    test(S("abcde"), 5, 1, S("abcde"), 1, 4, -4);
    test(S("abcde"), 5, 1, S("abcde"), 1, 5, -4);
    test(S("abcde"), 5, 1, S("abcde"), 2, 0, 0);
    test(S("abcde"), 5, 1, S("abcde"), 2, 1, -1);
    test(S("abcde"), 5, 1, S("abcde"), 2, 2, -2);
    test(S("abcde"), 5, 1, S("abcde"), 2, 3, -3);
    test(S("abcde"), 5, 1, S("abcde"), 2, 4, -3);
    test(S("abcde"), 5, 1, S("abcde"), 4, 0, 0);
    test(S("abcde"), 5, 1, S("abcde"), 4, 1, -1);
    test(S("abcde"), 5, 1, S("abcde"), 4, 2, -1);
    test(S("abcde"), 5, 1, S("abcde"), 5, 0, 0);
}

template <class S>
void test18()
{
    test(S("abcde"), 5, 1, S("abcde"), 5, 1, 0);
    test(S("abcde"), 5, 1, S("abcde"), 6, 0, 0);
    test(S("abcde"), 5, 1, S("abcdefghij"), 0, 0, 0);
    test(S("abcde"), 5, 1, S("abcdefghij"), 0, 1, -1);
    test(S("abcde"), 5, 1, S("abcdefghij"), 0, 5, -5);
    test(S("abcde"), 5, 1, S("abcdefghij"), 0, 9, -9);
    test(S("abcde"), 5, 1, S("abcdefghij"), 0, 10, -10);
    test(S("abcde"), 5, 1, S("abcdefghij"), 0, 11, -10);
    test(S("abcde"), 5, 1, S("abcdefghij"), 1, 0, 0);
    test(S("abcde"), 5, 1, S("abcdefghij"), 1, 1, -1);
    test(S("abcde"), 5, 1, S("abcdefghij"), 1, 4, -4);
    test(S("abcde"), 5, 1, S("abcdefghij"), 1, 8, -8);
    test(S("abcde"), 5, 1, S("abcdefghij"), 1, 9, -9);
    test(S("abcde"), 5, 1, S("abcdefghij"), 1, 10, -9);
    test(S("abcde"), 5, 1, S("abcdefghij"), 5, 0, 0);
    test(S("abcde"), 5, 1, S("abcdefghij"), 5, 1, -1);
    test(S("abcde"), 5, 1, S("abcdefghij"), 5, 2, -2);
    test(S("abcde"), 5, 1, S("abcdefghij"), 5, 4, -4);
    test(S("abcde"), 5, 1, S("abcdefghij"), 5, 5, -5);
    test(S("abcde"), 5, 1, S("abcdefghij"), 5, 6, -5);
    test(S("abcde"), 5, 1, S("abcdefghij"), 9, 0, 0);
    test(S("abcde"), 5, 1, S("abcdefghij"), 9, 1, -1);
    test(S("abcde"), 5, 1, S("abcdefghij"), 9, 2, -1);
    test(S("abcde"), 5, 1, S("abcdefghij"), 10, 0, 0);
    test(S("abcde"), 5, 1, S("abcdefghij"), 10, 1, 0);
    test(S("abcde"), 5, 1, S("abcdefghij"), 11, 0, 0);
    test(S("abcde"), 5, 1, S("abcdefghijklmnopqrst"), 0, 0, 0);
    test(S("abcde"), 5, 1, S("abcdefghijklmnopqrst"), 0, 1, -1);
    test(S("abcde"), 5, 1, S("abcdefghijklmnopqrst"), 0, 10, -10);
    test(S("abcde"), 5, 1, S("abcdefghijklmnopqrst"), 0, 19, -19);
    test(S("abcde"), 5, 1, S("abcdefghijklmnopqrst"), 0, 20, -20);
    test(S("abcde"), 5, 1, S("abcdefghijklmnopqrst"), 0, 21, -20);
    test(S("abcde"), 5, 1, S("abcdefghijklmnopqrst"), 1, 0, 0);
    test(S("abcde"), 5, 1, S("abcdefghijklmnopqrst"), 1, 1, -1);
    test(S("abcde"), 5, 1, S("abcdefghijklmnopqrst"), 1, 9, -9);
    test(S("abcde"), 5, 1, S("abcdefghijklmnopqrst"), 1, 18, -18);
    test(S("abcde"), 5, 1, S("abcdefghijklmnopqrst"), 1, 19, -19);
    test(S("abcde"), 5, 1, S("abcdefghijklmnopqrst"), 1, 20, -19);
    test(S("abcde"), 5, 1, S("abcdefghijklmnopqrst"), 10, 0, 0);
    test(S("abcde"), 5, 1, S("abcdefghijklmnopqrst"), 10, 1, -1);
    test(S("abcde"), 5, 1, S("abcdefghijklmnopqrst"), 10, 5, -5);
    test(S("abcde"), 5, 1, S("abcdefghijklmnopqrst"), 10, 9, -9);
    test(S("abcde"), 5, 1, S("abcdefghijklmnopqrst"), 10, 10, -10);
    test(S("abcde"), 5, 1, S("abcdefghijklmnopqrst"), 10, 11, -10);
    test(S("abcde"), 5, 1, S("abcdefghijklmnopqrst"), 19, 0, 0);
    test(S("abcde"), 5, 1, S("abcdefghijklmnopqrst"), 19, 1, -1);
    test(S("abcde"), 5, 1, S("abcdefghijklmnopqrst"), 19, 2, -1);
    test(S("abcde"), 5, 1, S("abcdefghijklmnopqrst"), 20, 0, 0);
    test(S("abcde"), 5, 1, S("abcdefghijklmnopqrst"), 20, 1, 0);
    test(S("abcde"), 5, 1, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcde"), 6, 0, S(""), 0, 0, 0);
    test(S("abcde"), 6, 0, S(""), 0, 1, 0);
    test(S("abcde"), 6, 0, S(""), 1, 0, 0);
    test(S("abcde"), 6, 0, S("abcde"), 0, 0, 0);
    test(S("abcde"), 6, 0, S("abcde"), 0, 1, 0);
    test(S("abcde"), 6, 0, S("abcde"), 0, 2, 0);
    test(S("abcde"), 6, 0, S("abcde"), 0, 4, 0);
    test(S("abcde"), 6, 0, S("abcde"), 0, 5, 0);
    test(S("abcde"), 6, 0, S("abcde"), 0, 6, 0);
    test(S("abcde"), 6, 0, S("abcde"), 1, 0, 0);
    test(S("abcde"), 6, 0, S("abcde"), 1, 1, 0);
    test(S("abcde"), 6, 0, S("abcde"), 1, 2, 0);
    test(S("abcde"), 6, 0, S("abcde"), 1, 3, 0);
    test(S("abcde"), 6, 0, S("abcde"), 1, 4, 0);
    test(S("abcde"), 6, 0, S("abcde"), 1, 5, 0);
    test(S("abcde"), 6, 0, S("abcde"), 2, 0, 0);
    test(S("abcde"), 6, 0, S("abcde"), 2, 1, 0);
    test(S("abcde"), 6, 0, S("abcde"), 2, 2, 0);
    test(S("abcde"), 6, 0, S("abcde"), 2, 3, 0);
    test(S("abcde"), 6, 0, S("abcde"), 2, 4, 0);
    test(S("abcde"), 6, 0, S("abcde"), 4, 0, 0);
    test(S("abcde"), 6, 0, S("abcde"), 4, 1, 0);
    test(S("abcde"), 6, 0, S("abcde"), 4, 2, 0);
    test(S("abcde"), 6, 0, S("abcde"), 5, 0, 0);
    test(S("abcde"), 6, 0, S("abcde"), 5, 1, 0);
    test(S("abcde"), 6, 0, S("abcde"), 6, 0, 0);
    test(S("abcde"), 6, 0, S("abcdefghij"), 0, 0, 0);
    test(S("abcde"), 6, 0, S("abcdefghij"), 0, 1, 0);
    test(S("abcde"), 6, 0, S("abcdefghij"), 0, 5, 0);
    test(S("abcde"), 6, 0, S("abcdefghij"), 0, 9, 0);
    test(S("abcde"), 6, 0, S("abcdefghij"), 0, 10, 0);
    test(S("abcde"), 6, 0, S("abcdefghij"), 0, 11, 0);
    test(S("abcde"), 6, 0, S("abcdefghij"), 1, 0, 0);
    test(S("abcde"), 6, 0, S("abcdefghij"), 1, 1, 0);
    test(S("abcde"), 6, 0, S("abcdefghij"), 1, 4, 0);
    test(S("abcde"), 6, 0, S("abcdefghij"), 1, 8, 0);
    test(S("abcde"), 6, 0, S("abcdefghij"), 1, 9, 0);
    test(S("abcde"), 6, 0, S("abcdefghij"), 1, 10, 0);
    test(S("abcde"), 6, 0, S("abcdefghij"), 5, 0, 0);
    test(S("abcde"), 6, 0, S("abcdefghij"), 5, 1, 0);
    test(S("abcde"), 6, 0, S("abcdefghij"), 5, 2, 0);
    test(S("abcde"), 6, 0, S("abcdefghij"), 5, 4, 0);
    test(S("abcde"), 6, 0, S("abcdefghij"), 5, 5, 0);
    test(S("abcde"), 6, 0, S("abcdefghij"), 5, 6, 0);
    test(S("abcde"), 6, 0, S("abcdefghij"), 9, 0, 0);
    test(S("abcde"), 6, 0, S("abcdefghij"), 9, 1, 0);
    test(S("abcde"), 6, 0, S("abcdefghij"), 9, 2, 0);
    test(S("abcde"), 6, 0, S("abcdefghij"), 10, 0, 0);
    test(S("abcde"), 6, 0, S("abcdefghij"), 10, 1, 0);
    test(S("abcde"), 6, 0, S("abcdefghij"), 11, 0, 0);
}

template <class S>
void test19()
{
    test(S("abcde"), 6, 0, S("abcdefghijklmnopqrst"), 0, 0, 0);
    test(S("abcde"), 6, 0, S("abcdefghijklmnopqrst"), 0, 1, 0);
    test(S("abcde"), 6, 0, S("abcdefghijklmnopqrst"), 0, 10, 0);
    test(S("abcde"), 6, 0, S("abcdefghijklmnopqrst"), 0, 19, 0);
    test(S("abcde"), 6, 0, S("abcdefghijklmnopqrst"), 0, 20, 0);
    test(S("abcde"), 6, 0, S("abcdefghijklmnopqrst"), 0, 21, 0);
    test(S("abcde"), 6, 0, S("abcdefghijklmnopqrst"), 1, 0, 0);
    test(S("abcde"), 6, 0, S("abcdefghijklmnopqrst"), 1, 1, 0);
    test(S("abcde"), 6, 0, S("abcdefghijklmnopqrst"), 1, 9, 0);
    test(S("abcde"), 6, 0, S("abcdefghijklmnopqrst"), 1, 18, 0);
    test(S("abcde"), 6, 0, S("abcdefghijklmnopqrst"), 1, 19, 0);
    test(S("abcde"), 6, 0, S("abcdefghijklmnopqrst"), 1, 20, 0);
    test(S("abcde"), 6, 0, S("abcdefghijklmnopqrst"), 10, 0, 0);
    test(S("abcde"), 6, 0, S("abcdefghijklmnopqrst"), 10, 1, 0);
    test(S("abcde"), 6, 0, S("abcdefghijklmnopqrst"), 10, 5, 0);
    test(S("abcde"), 6, 0, S("abcdefghijklmnopqrst"), 10, 9, 0);
    test(S("abcde"), 6, 0, S("abcdefghijklmnopqrst"), 10, 10, 0);
    test(S("abcde"), 6, 0, S("abcdefghijklmnopqrst"), 10, 11, 0);
    test(S("abcde"), 6, 0, S("abcdefghijklmnopqrst"), 19, 0, 0);
    test(S("abcde"), 6, 0, S("abcdefghijklmnopqrst"), 19, 1, 0);
    test(S("abcde"), 6, 0, S("abcdefghijklmnopqrst"), 19, 2, 0);
    test(S("abcde"), 6, 0, S("abcdefghijklmnopqrst"), 20, 0, 0);
    test(S("abcde"), 6, 0, S("abcdefghijklmnopqrst"), 20, 1, 0);
    test(S("abcde"), 6, 0, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghij"), 0, 0, S(""), 0, 0, 0);
    test(S("abcdefghij"), 0, 0, S(""), 0, 1, 0);
    test(S("abcdefghij"), 0, 0, S(""), 1, 0, 0);
    test(S("abcdefghij"), 0, 0, S("abcde"), 0, 0, 0);
    test(S("abcdefghij"), 0, 0, S("abcde"), 0, 1, -1);
    test(S("abcdefghij"), 0, 0, S("abcde"), 0, 2, -2);
    test(S("abcdefghij"), 0, 0, S("abcde"), 0, 4, -4);
    test(S("abcdefghij"), 0, 0, S("abcde"), 0, 5, -5);
    test(S("abcdefghij"), 0, 0, S("abcde"), 0, 6, -5);
    test(S("abcdefghij"), 0, 0, S("abcde"), 1, 0, 0);
    test(S("abcdefghij"), 0, 0, S("abcde"), 1, 1, -1);
    test(S("abcdefghij"), 0, 0, S("abcde"), 1, 2, -2);
    test(S("abcdefghij"), 0, 0, S("abcde"), 1, 3, -3);
    test(S("abcdefghij"), 0, 0, S("abcde"), 1, 4, -4);
    test(S("abcdefghij"), 0, 0, S("abcde"), 1, 5, -4);
    test(S("abcdefghij"), 0, 0, S("abcde"), 2, 0, 0);
    test(S("abcdefghij"), 0, 0, S("abcde"), 2, 1, -1);
    test(S("abcdefghij"), 0, 0, S("abcde"), 2, 2, -2);
    test(S("abcdefghij"), 0, 0, S("abcde"), 2, 3, -3);
    test(S("abcdefghij"), 0, 0, S("abcde"), 2, 4, -3);
    test(S("abcdefghij"), 0, 0, S("abcde"), 4, 0, 0);
    test(S("abcdefghij"), 0, 0, S("abcde"), 4, 1, -1);
    test(S("abcdefghij"), 0, 0, S("abcde"), 4, 2, -1);
    test(S("abcdefghij"), 0, 0, S("abcde"), 5, 0, 0);
    test(S("abcdefghij"), 0, 0, S("abcde"), 5, 1, 0);
    test(S("abcdefghij"), 0, 0, S("abcde"), 6, 0, 0);
    test(S("abcdefghij"), 0, 0, S("abcdefghij"), 0, 0, 0);
    test(S("abcdefghij"), 0, 0, S("abcdefghij"), 0, 1, -1);
    test(S("abcdefghij"), 0, 0, S("abcdefghij"), 0, 5, -5);
    test(S("abcdefghij"), 0, 0, S("abcdefghij"), 0, 9, -9);
    test(S("abcdefghij"), 0, 0, S("abcdefghij"), 0, 10, -10);
    test(S("abcdefghij"), 0, 0, S("abcdefghij"), 0, 11, -10);
    test(S("abcdefghij"), 0, 0, S("abcdefghij"), 1, 0, 0);
    test(S("abcdefghij"), 0, 0, S("abcdefghij"), 1, 1, -1);
    test(S("abcdefghij"), 0, 0, S("abcdefghij"), 1, 4, -4);
    test(S("abcdefghij"), 0, 0, S("abcdefghij"), 1, 8, -8);
    test(S("abcdefghij"), 0, 0, S("abcdefghij"), 1, 9, -9);
    test(S("abcdefghij"), 0, 0, S("abcdefghij"), 1, 10, -9);
    test(S("abcdefghij"), 0, 0, S("abcdefghij"), 5, 0, 0);
    test(S("abcdefghij"), 0, 0, S("abcdefghij"), 5, 1, -1);
    test(S("abcdefghij"), 0, 0, S("abcdefghij"), 5, 2, -2);
    test(S("abcdefghij"), 0, 0, S("abcdefghij"), 5, 4, -4);
    test(S("abcdefghij"), 0, 0, S("abcdefghij"), 5, 5, -5);
    test(S("abcdefghij"), 0, 0, S("abcdefghij"), 5, 6, -5);
    test(S("abcdefghij"), 0, 0, S("abcdefghij"), 9, 0, 0);
    test(S("abcdefghij"), 0, 0, S("abcdefghij"), 9, 1, -1);
    test(S("abcdefghij"), 0, 0, S("abcdefghij"), 9, 2, -1);
    test(S("abcdefghij"), 0, 0, S("abcdefghij"), 10, 0, 0);
    test(S("abcdefghij"), 0, 0, S("abcdefghij"), 10, 1, 0);
    test(S("abcdefghij"), 0, 0, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghij"), 0, 0, S("abcdefghijklmnopqrst"), 0, 0, 0);
    test(S("abcdefghij"), 0, 0, S("abcdefghijklmnopqrst"), 0, 1, -1);
    test(S("abcdefghij"), 0, 0, S("abcdefghijklmnopqrst"), 0, 10, -10);
    test(S("abcdefghij"), 0, 0, S("abcdefghijklmnopqrst"), 0, 19, -19);
    test(S("abcdefghij"), 0, 0, S("abcdefghijklmnopqrst"), 0, 20, -20);
    test(S("abcdefghij"), 0, 0, S("abcdefghijklmnopqrst"), 0, 21, -20);
    test(S("abcdefghij"), 0, 0, S("abcdefghijklmnopqrst"), 1, 0, 0);
    test(S("abcdefghij"), 0, 0, S("abcdefghijklmnopqrst"), 1, 1, -1);
    test(S("abcdefghij"), 0, 0, S("abcdefghijklmnopqrst"), 1, 9, -9);
    test(S("abcdefghij"), 0, 0, S("abcdefghijklmnopqrst"), 1, 18, -18);
    test(S("abcdefghij"), 0, 0, S("abcdefghijklmnopqrst"), 1, 19, -19);
    test(S("abcdefghij"), 0, 0, S("abcdefghijklmnopqrst"), 1, 20, -19);
    test(S("abcdefghij"), 0, 0, S("abcdefghijklmnopqrst"), 10, 0, 0);
    test(S("abcdefghij"), 0, 0, S("abcdefghijklmnopqrst"), 10, 1, -1);
    test(S("abcdefghij"), 0, 0, S("abcdefghijklmnopqrst"), 10, 5, -5);
    test(S("abcdefghij"), 0, 0, S("abcdefghijklmnopqrst"), 10, 9, -9);
    test(S("abcdefghij"), 0, 0, S("abcdefghijklmnopqrst"), 10, 10, -10);
    test(S("abcdefghij"), 0, 0, S("abcdefghijklmnopqrst"), 10, 11, -10);
    test(S("abcdefghij"), 0, 0, S("abcdefghijklmnopqrst"), 19, 0, 0);
    test(S("abcdefghij"), 0, 0, S("abcdefghijklmnopqrst"), 19, 1, -1);
    test(S("abcdefghij"), 0, 0, S("abcdefghijklmnopqrst"), 19, 2, -1);
    test(S("abcdefghij"), 0, 0, S("abcdefghijklmnopqrst"), 20, 0, 0);
    test(S("abcdefghij"), 0, 0, S("abcdefghijklmnopqrst"), 20, 1, 0);
    test(S("abcdefghij"), 0, 0, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghij"), 0, 1, S(""), 0, 0, 1);
    test(S("abcdefghij"), 0, 1, S(""), 0, 1, 1);
}

template <class S>
void test20()
{
    test(S("abcdefghij"), 0, 1, S(""), 1, 0, 0);
    test(S("abcdefghij"), 0, 1, S("abcde"), 0, 0, 1);
    test(S("abcdefghij"), 0, 1, S("abcde"), 0, 1, 0);
    test(S("abcdefghij"), 0, 1, S("abcde"), 0, 2, -1);
    test(S("abcdefghij"), 0, 1, S("abcde"), 0, 4, -3);
    test(S("abcdefghij"), 0, 1, S("abcde"), 0, 5, -4);
    test(S("abcdefghij"), 0, 1, S("abcde"), 0, 6, -4);
    test(S("abcdefghij"), 0, 1, S("abcde"), 1, 0, 1);
    test(S("abcdefghij"), 0, 1, S("abcde"), 1, 1, -1);
    test(S("abcdefghij"), 0, 1, S("abcde"), 1, 2, -1);
    test(S("abcdefghij"), 0, 1, S("abcde"), 1, 3, -1);
    test(S("abcdefghij"), 0, 1, S("abcde"), 1, 4, -1);
    test(S("abcdefghij"), 0, 1, S("abcde"), 1, 5, -1);
    test(S("abcdefghij"), 0, 1, S("abcde"), 2, 0, 1);
    test(S("abcdefghij"), 0, 1, S("abcde"), 2, 1, -2);
    test(S("abcdefghij"), 0, 1, S("abcde"), 2, 2, -2);
    test(S("abcdefghij"), 0, 1, S("abcde"), 2, 3, -2);
    test(S("abcdefghij"), 0, 1, S("abcde"), 2, 4, -2);
    test(S("abcdefghij"), 0, 1, S("abcde"), 4, 0, 1);
    test(S("abcdefghij"), 0, 1, S("abcde"), 4, 1, -4);
    test(S("abcdefghij"), 0, 1, S("abcde"), 4, 2, -4);
    test(S("abcdefghij"), 0, 1, S("abcde"), 5, 0, 1);
    test(S("abcdefghij"), 0, 1, S("abcde"), 5, 1, 1);
    test(S("abcdefghij"), 0, 1, S("abcde"), 6, 0, 0);
    test(S("abcdefghij"), 0, 1, S("abcdefghij"), 0, 0, 1);
    test(S("abcdefghij"), 0, 1, S("abcdefghij"), 0, 1, 0);
    test(S("abcdefghij"), 0, 1, S("abcdefghij"), 0, 5, -4);
    test(S("abcdefghij"), 0, 1, S("abcdefghij"), 0, 9, -8);
    test(S("abcdefghij"), 0, 1, S("abcdefghij"), 0, 10, -9);
    test(S("abcdefghij"), 0, 1, S("abcdefghij"), 0, 11, -9);
    test(S("abcdefghij"), 0, 1, S("abcdefghij"), 1, 0, 1);
    test(S("abcdefghij"), 0, 1, S("abcdefghij"), 1, 1, -1);
    test(S("abcdefghij"), 0, 1, S("abcdefghij"), 1, 4, -1);
    test(S("abcdefghij"), 0, 1, S("abcdefghij"), 1, 8, -1);
    test(S("abcdefghij"), 0, 1, S("abcdefghij"), 1, 9, -1);
    test(S("abcdefghij"), 0, 1, S("abcdefghij"), 1, 10, -1);
    test(S("abcdefghij"), 0, 1, S("abcdefghij"), 5, 0, 1);
    test(S("abcdefghij"), 0, 1, S("abcdefghij"), 5, 1, -5);
    test(S("abcdefghij"), 0, 1, S("abcdefghij"), 5, 2, -5);
    test(S("abcdefghij"), 0, 1, S("abcdefghij"), 5, 4, -5);
    test(S("abcdefghij"), 0, 1, S("abcdefghij"), 5, 5, -5);
    test(S("abcdefghij"), 0, 1, S("abcdefghij"), 5, 6, -5);
    test(S("abcdefghij"), 0, 1, S("abcdefghij"), 9, 0, 1);
    test(S("abcdefghij"), 0, 1, S("abcdefghij"), 9, 1, -9);
    test(S("abcdefghij"), 0, 1, S("abcdefghij"), 9, 2, -9);
    test(S("abcdefghij"), 0, 1, S("abcdefghij"), 10, 0, 1);
    test(S("abcdefghij"), 0, 1, S("abcdefghij"), 10, 1, 1);
    test(S("abcdefghij"), 0, 1, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghij"), 0, 1, S("abcdefghijklmnopqrst"), 0, 0, 1);
    test(S("abcdefghij"), 0, 1, S("abcdefghijklmnopqrst"), 0, 1, 0);
    test(S("abcdefghij"), 0, 1, S("abcdefghijklmnopqrst"), 0, 10, -9);
    test(S("abcdefghij"), 0, 1, S("abcdefghijklmnopqrst"), 0, 19, -18);
    test(S("abcdefghij"), 0, 1, S("abcdefghijklmnopqrst"), 0, 20, -19);
    test(S("abcdefghij"), 0, 1, S("abcdefghijklmnopqrst"), 0, 21, -19);
    test(S("abcdefghij"), 0, 1, S("abcdefghijklmnopqrst"), 1, 0, 1);
    test(S("abcdefghij"), 0, 1, S("abcdefghijklmnopqrst"), 1, 1, -1);
    test(S("abcdefghij"), 0, 1, S("abcdefghijklmnopqrst"), 1, 9, -1);
    test(S("abcdefghij"), 0, 1, S("abcdefghijklmnopqrst"), 1, 18, -1);
    test(S("abcdefghij"), 0, 1, S("abcdefghijklmnopqrst"), 1, 19, -1);
    test(S("abcdefghij"), 0, 1, S("abcdefghijklmnopqrst"), 1, 20, -1);
    test(S("abcdefghij"), 0, 1, S("abcdefghijklmnopqrst"), 10, 0, 1);
    test(S("abcdefghij"), 0, 1, S("abcdefghijklmnopqrst"), 10, 1, -10);
    test(S("abcdefghij"), 0, 1, S("abcdefghijklmnopqrst"), 10, 5, -10);
    test(S("abcdefghij"), 0, 1, S("abcdefghijklmnopqrst"), 10, 9, -10);
    test(S("abcdefghij"), 0, 1, S("abcdefghijklmnopqrst"), 10, 10, -10);
    test(S("abcdefghij"), 0, 1, S("abcdefghijklmnopqrst"), 10, 11, -10);
    test(S("abcdefghij"), 0, 1, S("abcdefghijklmnopqrst"), 19, 0, 1);
    test(S("abcdefghij"), 0, 1, S("abcdefghijklmnopqrst"), 19, 1, -19);
    test(S("abcdefghij"), 0, 1, S("abcdefghijklmnopqrst"), 19, 2, -19);
    test(S("abcdefghij"), 0, 1, S("abcdefghijklmnopqrst"), 20, 0, 1);
    test(S("abcdefghij"), 0, 1, S("abcdefghijklmnopqrst"), 20, 1, 1);
    test(S("abcdefghij"), 0, 1, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghij"), 0, 5, S(""), 0, 0, 5);
    test(S("abcdefghij"), 0, 5, S(""), 0, 1, 5);
    test(S("abcdefghij"), 0, 5, S(""), 1, 0, 0);
    test(S("abcdefghij"), 0, 5, S("abcde"), 0, 0, 5);
    test(S("abcdefghij"), 0, 5, S("abcde"), 0, 1, 4);
    test(S("abcdefghij"), 0, 5, S("abcde"), 0, 2, 3);
    test(S("abcdefghij"), 0, 5, S("abcde"), 0, 4, 1);
    test(S("abcdefghij"), 0, 5, S("abcde"), 0, 5, 0);
    test(S("abcdefghij"), 0, 5, S("abcde"), 0, 6, 0);
    test(S("abcdefghij"), 0, 5, S("abcde"), 1, 0, 5);
    test(S("abcdefghij"), 0, 5, S("abcde"), 1, 1, -1);
    test(S("abcdefghij"), 0, 5, S("abcde"), 1, 2, -1);
    test(S("abcdefghij"), 0, 5, S("abcde"), 1, 3, -1);
    test(S("abcdefghij"), 0, 5, S("abcde"), 1, 4, -1);
    test(S("abcdefghij"), 0, 5, S("abcde"), 1, 5, -1);
    test(S("abcdefghij"), 0, 5, S("abcde"), 2, 0, 5);
    test(S("abcdefghij"), 0, 5, S("abcde"), 2, 1, -2);
    test(S("abcdefghij"), 0, 5, S("abcde"), 2, 2, -2);
    test(S("abcdefghij"), 0, 5, S("abcde"), 2, 3, -2);
    test(S("abcdefghij"), 0, 5, S("abcde"), 2, 4, -2);
    test(S("abcdefghij"), 0, 5, S("abcde"), 4, 0, 5);
    test(S("abcdefghij"), 0, 5, S("abcde"), 4, 1, -4);
    test(S("abcdefghij"), 0, 5, S("abcde"), 4, 2, -4);
    test(S("abcdefghij"), 0, 5, S("abcde"), 5, 0, 5);
    test(S("abcdefghij"), 0, 5, S("abcde"), 5, 1, 5);
    test(S("abcdefghij"), 0, 5, S("abcde"), 6, 0, 0);
    test(S("abcdefghij"), 0, 5, S("abcdefghij"), 0, 0, 5);
    test(S("abcdefghij"), 0, 5, S("abcdefghij"), 0, 1, 4);
}

template <class S>
void test21()
{
    test(S("abcdefghij"), 0, 5, S("abcdefghij"), 0, 5, 0);
    test(S("abcdefghij"), 0, 5, S("abcdefghij"), 0, 9, -4);
    test(S("abcdefghij"), 0, 5, S("abcdefghij"), 0, 10, -5);
    test(S("abcdefghij"), 0, 5, S("abcdefghij"), 0, 11, -5);
    test(S("abcdefghij"), 0, 5, S("abcdefghij"), 1, 0, 5);
    test(S("abcdefghij"), 0, 5, S("abcdefghij"), 1, 1, -1);
    test(S("abcdefghij"), 0, 5, S("abcdefghij"), 1, 4, -1);
    test(S("abcdefghij"), 0, 5, S("abcdefghij"), 1, 8, -1);
    test(S("abcdefghij"), 0, 5, S("abcdefghij"), 1, 9, -1);
    test(S("abcdefghij"), 0, 5, S("abcdefghij"), 1, 10, -1);
    test(S("abcdefghij"), 0, 5, S("abcdefghij"), 5, 0, 5);
    test(S("abcdefghij"), 0, 5, S("abcdefghij"), 5, 1, -5);
    test(S("abcdefghij"), 0, 5, S("abcdefghij"), 5, 2, -5);
    test(S("abcdefghij"), 0, 5, S("abcdefghij"), 5, 4, -5);
    test(S("abcdefghij"), 0, 5, S("abcdefghij"), 5, 5, -5);
    test(S("abcdefghij"), 0, 5, S("abcdefghij"), 5, 6, -5);
    test(S("abcdefghij"), 0, 5, S("abcdefghij"), 9, 0, 5);
    test(S("abcdefghij"), 0, 5, S("abcdefghij"), 9, 1, -9);
    test(S("abcdefghij"), 0, 5, S("abcdefghij"), 9, 2, -9);
    test(S("abcdefghij"), 0, 5, S("abcdefghij"), 10, 0, 5);
    test(S("abcdefghij"), 0, 5, S("abcdefghij"), 10, 1, 5);
    test(S("abcdefghij"), 0, 5, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghij"), 0, 5, S("abcdefghijklmnopqrst"), 0, 0, 5);
    test(S("abcdefghij"), 0, 5, S("abcdefghijklmnopqrst"), 0, 1, 4);
    test(S("abcdefghij"), 0, 5, S("abcdefghijklmnopqrst"), 0, 10, -5);
    test(S("abcdefghij"), 0, 5, S("abcdefghijklmnopqrst"), 0, 19, -14);
    test(S("abcdefghij"), 0, 5, S("abcdefghijklmnopqrst"), 0, 20, -15);
    test(S("abcdefghij"), 0, 5, S("abcdefghijklmnopqrst"), 0, 21, -15);
    test(S("abcdefghij"), 0, 5, S("abcdefghijklmnopqrst"), 1, 0, 5);
    test(S("abcdefghij"), 0, 5, S("abcdefghijklmnopqrst"), 1, 1, -1);
    test(S("abcdefghij"), 0, 5, S("abcdefghijklmnopqrst"), 1, 9, -1);
    test(S("abcdefghij"), 0, 5, S("abcdefghijklmnopqrst"), 1, 18, -1);
    test(S("abcdefghij"), 0, 5, S("abcdefghijklmnopqrst"), 1, 19, -1);
    test(S("abcdefghij"), 0, 5, S("abcdefghijklmnopqrst"), 1, 20, -1);
    test(S("abcdefghij"), 0, 5, S("abcdefghijklmnopqrst"), 10, 0, 5);
    test(S("abcdefghij"), 0, 5, S("abcdefghijklmnopqrst"), 10, 1, -10);
    test(S("abcdefghij"), 0, 5, S("abcdefghijklmnopqrst"), 10, 5, -10);
    test(S("abcdefghij"), 0, 5, S("abcdefghijklmnopqrst"), 10, 9, -10);
    test(S("abcdefghij"), 0, 5, S("abcdefghijklmnopqrst"), 10, 10, -10);
    test(S("abcdefghij"), 0, 5, S("abcdefghijklmnopqrst"), 10, 11, -10);
    test(S("abcdefghij"), 0, 5, S("abcdefghijklmnopqrst"), 19, 0, 5);
    test(S("abcdefghij"), 0, 5, S("abcdefghijklmnopqrst"), 19, 1, -19);
    test(S("abcdefghij"), 0, 5, S("abcdefghijklmnopqrst"), 19, 2, -19);
    test(S("abcdefghij"), 0, 5, S("abcdefghijklmnopqrst"), 20, 0, 5);
    test(S("abcdefghij"), 0, 5, S("abcdefghijklmnopqrst"), 20, 1, 5);
    test(S("abcdefghij"), 0, 5, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghij"), 0, 9, S(""), 0, 0, 9);
    test(S("abcdefghij"), 0, 9, S(""), 0, 1, 9);
    test(S("abcdefghij"), 0, 9, S(""), 1, 0, 0);
    test(S("abcdefghij"), 0, 9, S("abcde"), 0, 0, 9);
    test(S("abcdefghij"), 0, 9, S("abcde"), 0, 1, 8);
    test(S("abcdefghij"), 0, 9, S("abcde"), 0, 2, 7);
    test(S("abcdefghij"), 0, 9, S("abcde"), 0, 4, 5);
    test(S("abcdefghij"), 0, 9, S("abcde"), 0, 5, 4);
    test(S("abcdefghij"), 0, 9, S("abcde"), 0, 6, 4);
    test(S("abcdefghij"), 0, 9, S("abcde"), 1, 0, 9);
    test(S("abcdefghij"), 0, 9, S("abcde"), 1, 1, -1);
    test(S("abcdefghij"), 0, 9, S("abcde"), 1, 2, -1);
    test(S("abcdefghij"), 0, 9, S("abcde"), 1, 3, -1);
    test(S("abcdefghij"), 0, 9, S("abcde"), 1, 4, -1);
    test(S("abcdefghij"), 0, 9, S("abcde"), 1, 5, -1);
    test(S("abcdefghij"), 0, 9, S("abcde"), 2, 0, 9);
    test(S("abcdefghij"), 0, 9, S("abcde"), 2, 1, -2);
    test(S("abcdefghij"), 0, 9, S("abcde"), 2, 2, -2);
    test(S("abcdefghij"), 0, 9, S("abcde"), 2, 3, -2);
    test(S("abcdefghij"), 0, 9, S("abcde"), 2, 4, -2);
    test(S("abcdefghij"), 0, 9, S("abcde"), 4, 0, 9);
    test(S("abcdefghij"), 0, 9, S("abcde"), 4, 1, -4);
    test(S("abcdefghij"), 0, 9, S("abcde"), 4, 2, -4);
    test(S("abcdefghij"), 0, 9, S("abcde"), 5, 0, 9);
    test(S("abcdefghij"), 0, 9, S("abcde"), 5, 1, 9);
    test(S("abcdefghij"), 0, 9, S("abcde"), 6, 0, 0);
    test(S("abcdefghij"), 0, 9, S("abcdefghij"), 0, 0, 9);
    test(S("abcdefghij"), 0, 9, S("abcdefghij"), 0, 1, 8);
    test(S("abcdefghij"), 0, 9, S("abcdefghij"), 0, 5, 4);
    test(S("abcdefghij"), 0, 9, S("abcdefghij"), 0, 9, 0);
    test(S("abcdefghij"), 0, 9, S("abcdefghij"), 0, 10, -1);
    test(S("abcdefghij"), 0, 9, S("abcdefghij"), 0, 11, -1);
    test(S("abcdefghij"), 0, 9, S("abcdefghij"), 1, 0, 9);
    test(S("abcdefghij"), 0, 9, S("abcdefghij"), 1, 1, -1);
    test(S("abcdefghij"), 0, 9, S("abcdefghij"), 1, 4, -1);
    test(S("abcdefghij"), 0, 9, S("abcdefghij"), 1, 8, -1);
    test(S("abcdefghij"), 0, 9, S("abcdefghij"), 1, 9, -1);
    test(S("abcdefghij"), 0, 9, S("abcdefghij"), 1, 10, -1);
    test(S("abcdefghij"), 0, 9, S("abcdefghij"), 5, 0, 9);
    test(S("abcdefghij"), 0, 9, S("abcdefghij"), 5, 1, -5);
    test(S("abcdefghij"), 0, 9, S("abcdefghij"), 5, 2, -5);
    test(S("abcdefghij"), 0, 9, S("abcdefghij"), 5, 4, -5);
    test(S("abcdefghij"), 0, 9, S("abcdefghij"), 5, 5, -5);
    test(S("abcdefghij"), 0, 9, S("abcdefghij"), 5, 6, -5);
    test(S("abcdefghij"), 0, 9, S("abcdefghij"), 9, 0, 9);
    test(S("abcdefghij"), 0, 9, S("abcdefghij"), 9, 1, -9);
    test(S("abcdefghij"), 0, 9, S("abcdefghij"), 9, 2, -9);
    test(S("abcdefghij"), 0, 9, S("abcdefghij"), 10, 0, 9);
    test(S("abcdefghij"), 0, 9, S("abcdefghij"), 10, 1, 9);
    test(S("abcdefghij"), 0, 9, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghij"), 0, 9, S("abcdefghijklmnopqrst"), 0, 0, 9);
    test(S("abcdefghij"), 0, 9, S("abcdefghijklmnopqrst"), 0, 1, 8);
    test(S("abcdefghij"), 0, 9, S("abcdefghijklmnopqrst"), 0, 10, -1);
    test(S("abcdefghij"), 0, 9, S("abcdefghijklmnopqrst"), 0, 19, -10);
}

template <class S>
void test22()
{
    test(S("abcdefghij"), 0, 9, S("abcdefghijklmnopqrst"), 0, 20, -11);
    test(S("abcdefghij"), 0, 9, S("abcdefghijklmnopqrst"), 0, 21, -11);
    test(S("abcdefghij"), 0, 9, S("abcdefghijklmnopqrst"), 1, 0, 9);
    test(S("abcdefghij"), 0, 9, S("abcdefghijklmnopqrst"), 1, 1, -1);
    test(S("abcdefghij"), 0, 9, S("abcdefghijklmnopqrst"), 1, 9, -1);
    test(S("abcdefghij"), 0, 9, S("abcdefghijklmnopqrst"), 1, 18, -1);
    test(S("abcdefghij"), 0, 9, S("abcdefghijklmnopqrst"), 1, 19, -1);
    test(S("abcdefghij"), 0, 9, S("abcdefghijklmnopqrst"), 1, 20, -1);
    test(S("abcdefghij"), 0, 9, S("abcdefghijklmnopqrst"), 10, 0, 9);
    test(S("abcdefghij"), 0, 9, S("abcdefghijklmnopqrst"), 10, 1, -10);
    test(S("abcdefghij"), 0, 9, S("abcdefghijklmnopqrst"), 10, 5, -10);
    test(S("abcdefghij"), 0, 9, S("abcdefghijklmnopqrst"), 10, 9, -10);
    test(S("abcdefghij"), 0, 9, S("abcdefghijklmnopqrst"), 10, 10, -10);
    test(S("abcdefghij"), 0, 9, S("abcdefghijklmnopqrst"), 10, 11, -10);
    test(S("abcdefghij"), 0, 9, S("abcdefghijklmnopqrst"), 19, 0, 9);
    test(S("abcdefghij"), 0, 9, S("abcdefghijklmnopqrst"), 19, 1, -19);
    test(S("abcdefghij"), 0, 9, S("abcdefghijklmnopqrst"), 19, 2, -19);
    test(S("abcdefghij"), 0, 9, S("abcdefghijklmnopqrst"), 20, 0, 9);
    test(S("abcdefghij"), 0, 9, S("abcdefghijklmnopqrst"), 20, 1, 9);
    test(S("abcdefghij"), 0, 9, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghij"), 0, 10, S(""), 0, 0, 10);
    test(S("abcdefghij"), 0, 10, S(""), 0, 1, 10);
    test(S("abcdefghij"), 0, 10, S(""), 1, 0, 0);
    test(S("abcdefghij"), 0, 10, S("abcde"), 0, 0, 10);
    test(S("abcdefghij"), 0, 10, S("abcde"), 0, 1, 9);
    test(S("abcdefghij"), 0, 10, S("abcde"), 0, 2, 8);
    test(S("abcdefghij"), 0, 10, S("abcde"), 0, 4, 6);
    test(S("abcdefghij"), 0, 10, S("abcde"), 0, 5, 5);
    test(S("abcdefghij"), 0, 10, S("abcde"), 0, 6, 5);
    test(S("abcdefghij"), 0, 10, S("abcde"), 1, 0, 10);
    test(S("abcdefghij"), 0, 10, S("abcde"), 1, 1, -1);
    test(S("abcdefghij"), 0, 10, S("abcde"), 1, 2, -1);
    test(S("abcdefghij"), 0, 10, S("abcde"), 1, 3, -1);
    test(S("abcdefghij"), 0, 10, S("abcde"), 1, 4, -1);
    test(S("abcdefghij"), 0, 10, S("abcde"), 1, 5, -1);
    test(S("abcdefghij"), 0, 10, S("abcde"), 2, 0, 10);
    test(S("abcdefghij"), 0, 10, S("abcde"), 2, 1, -2);
    test(S("abcdefghij"), 0, 10, S("abcde"), 2, 2, -2);
    test(S("abcdefghij"), 0, 10, S("abcde"), 2, 3, -2);
    test(S("abcdefghij"), 0, 10, S("abcde"), 2, 4, -2);
    test(S("abcdefghij"), 0, 10, S("abcde"), 4, 0, 10);
    test(S("abcdefghij"), 0, 10, S("abcde"), 4, 1, -4);
    test(S("abcdefghij"), 0, 10, S("abcde"), 4, 2, -4);
    test(S("abcdefghij"), 0, 10, S("abcde"), 5, 0, 10);
    test(S("abcdefghij"), 0, 10, S("abcde"), 5, 1, 10);
    test(S("abcdefghij"), 0, 10, S("abcde"), 6, 0, 0);
    test(S("abcdefghij"), 0, 10, S("abcdefghij"), 0, 0, 10);
    test(S("abcdefghij"), 0, 10, S("abcdefghij"), 0, 1, 9);
    test(S("abcdefghij"), 0, 10, S("abcdefghij"), 0, 5, 5);
    test(S("abcdefghij"), 0, 10, S("abcdefghij"), 0, 9, 1);
    test(S("abcdefghij"), 0, 10, S("abcdefghij"), 0, 10, 0);
    test(S("abcdefghij"), 0, 10, S("abcdefghij"), 0, 11, 0);
    test(S("abcdefghij"), 0, 10, S("abcdefghij"), 1, 0, 10);
    test(S("abcdefghij"), 0, 10, S("abcdefghij"), 1, 1, -1);
    test(S("abcdefghij"), 0, 10, S("abcdefghij"), 1, 4, -1);
    test(S("abcdefghij"), 0, 10, S("abcdefghij"), 1, 8, -1);
    test(S("abcdefghij"), 0, 10, S("abcdefghij"), 1, 9, -1);
    test(S("abcdefghij"), 0, 10, S("abcdefghij"), 1, 10, -1);
    test(S("abcdefghij"), 0, 10, S("abcdefghij"), 5, 0, 10);
    test(S("abcdefghij"), 0, 10, S("abcdefghij"), 5, 1, -5);
    test(S("abcdefghij"), 0, 10, S("abcdefghij"), 5, 2, -5);
    test(S("abcdefghij"), 0, 10, S("abcdefghij"), 5, 4, -5);
    test(S("abcdefghij"), 0, 10, S("abcdefghij"), 5, 5, -5);
    test(S("abcdefghij"), 0, 10, S("abcdefghij"), 5, 6, -5);
    test(S("abcdefghij"), 0, 10, S("abcdefghij"), 9, 0, 10);
    test(S("abcdefghij"), 0, 10, S("abcdefghij"), 9, 1, -9);
    test(S("abcdefghij"), 0, 10, S("abcdefghij"), 9, 2, -9);
    test(S("abcdefghij"), 0, 10, S("abcdefghij"), 10, 0, 10);
    test(S("abcdefghij"), 0, 10, S("abcdefghij"), 10, 1, 10);
    test(S("abcdefghij"), 0, 10, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghij"), 0, 10, S("abcdefghijklmnopqrst"), 0, 0, 10);
    test(S("abcdefghij"), 0, 10, S("abcdefghijklmnopqrst"), 0, 1, 9);
    test(S("abcdefghij"), 0, 10, S("abcdefghijklmnopqrst"), 0, 10, 0);
    test(S("abcdefghij"), 0, 10, S("abcdefghijklmnopqrst"), 0, 19, -9);
    test(S("abcdefghij"), 0, 10, S("abcdefghijklmnopqrst"), 0, 20, -10);
    test(S("abcdefghij"), 0, 10, S("abcdefghijklmnopqrst"), 0, 21, -10);
    test(S("abcdefghij"), 0, 10, S("abcdefghijklmnopqrst"), 1, 0, 10);
    test(S("abcdefghij"), 0, 10, S("abcdefghijklmnopqrst"), 1, 1, -1);
    test(S("abcdefghij"), 0, 10, S("abcdefghijklmnopqrst"), 1, 9, -1);
    test(S("abcdefghij"), 0, 10, S("abcdefghijklmnopqrst"), 1, 18, -1);
    test(S("abcdefghij"), 0, 10, S("abcdefghijklmnopqrst"), 1, 19, -1);
    test(S("abcdefghij"), 0, 10, S("abcdefghijklmnopqrst"), 1, 20, -1);
    test(S("abcdefghij"), 0, 10, S("abcdefghijklmnopqrst"), 10, 0, 10);
    test(S("abcdefghij"), 0, 10, S("abcdefghijklmnopqrst"), 10, 1, -10);
    test(S("abcdefghij"), 0, 10, S("abcdefghijklmnopqrst"), 10, 5, -10);
    test(S("abcdefghij"), 0, 10, S("abcdefghijklmnopqrst"), 10, 9, -10);
    test(S("abcdefghij"), 0, 10, S("abcdefghijklmnopqrst"), 10, 10, -10);
    test(S("abcdefghij"), 0, 10, S("abcdefghijklmnopqrst"), 10, 11, -10);
    test(S("abcdefghij"), 0, 10, S("abcdefghijklmnopqrst"), 19, 0, 10);
    test(S("abcdefghij"), 0, 10, S("abcdefghijklmnopqrst"), 19, 1, -19);
    test(S("abcdefghij"), 0, 10, S("abcdefghijklmnopqrst"), 19, 2, -19);
    test(S("abcdefghij"), 0, 10, S("abcdefghijklmnopqrst"), 20, 0, 10);
    test(S("abcdefghij"), 0, 10, S("abcdefghijklmnopqrst"), 20, 1, 10);
    test(S("abcdefghij"), 0, 10, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghij"), 0, 11, S(""), 0, 0, 10);
    test(S("abcdefghij"), 0, 11, S(""), 0, 1, 10);
    test(S("abcdefghij"), 0, 11, S(""), 1, 0, 0);
    test(S("abcdefghij"), 0, 11, S("abcde"), 0, 0, 10);
    test(S("abcdefghij"), 0, 11, S("abcde"), 0, 1, 9);
    test(S("abcdefghij"), 0, 11, S("abcde"), 0, 2, 8);
}

template <class S>
void test23()
{
    test(S("abcdefghij"), 0, 11, S("abcde"), 0, 4, 6);
    test(S("abcdefghij"), 0, 11, S("abcde"), 0, 5, 5);
    test(S("abcdefghij"), 0, 11, S("abcde"), 0, 6, 5);
    test(S("abcdefghij"), 0, 11, S("abcde"), 1, 0, 10);
    test(S("abcdefghij"), 0, 11, S("abcde"), 1, 1, -1);
    test(S("abcdefghij"), 0, 11, S("abcde"), 1, 2, -1);
    test(S("abcdefghij"), 0, 11, S("abcde"), 1, 3, -1);
    test(S("abcdefghij"), 0, 11, S("abcde"), 1, 4, -1);
    test(S("abcdefghij"), 0, 11, S("abcde"), 1, 5, -1);
    test(S("abcdefghij"), 0, 11, S("abcde"), 2, 0, 10);
    test(S("abcdefghij"), 0, 11, S("abcde"), 2, 1, -2);
    test(S("abcdefghij"), 0, 11, S("abcde"), 2, 2, -2);
    test(S("abcdefghij"), 0, 11, S("abcde"), 2, 3, -2);
    test(S("abcdefghij"), 0, 11, S("abcde"), 2, 4, -2);
    test(S("abcdefghij"), 0, 11, S("abcde"), 4, 0, 10);
    test(S("abcdefghij"), 0, 11, S("abcde"), 4, 1, -4);
    test(S("abcdefghij"), 0, 11, S("abcde"), 4, 2, -4);
    test(S("abcdefghij"), 0, 11, S("abcde"), 5, 0, 10);
    test(S("abcdefghij"), 0, 11, S("abcde"), 5, 1, 10);
    test(S("abcdefghij"), 0, 11, S("abcde"), 6, 0, 0);
    test(S("abcdefghij"), 0, 11, S("abcdefghij"), 0, 0, 10);
    test(S("abcdefghij"), 0, 11, S("abcdefghij"), 0, 1, 9);
    test(S("abcdefghij"), 0, 11, S("abcdefghij"), 0, 5, 5);
    test(S("abcdefghij"), 0, 11, S("abcdefghij"), 0, 9, 1);
    test(S("abcdefghij"), 0, 11, S("abcdefghij"), 0, 10, 0);
    test(S("abcdefghij"), 0, 11, S("abcdefghij"), 0, 11, 0);
    test(S("abcdefghij"), 0, 11, S("abcdefghij"), 1, 0, 10);
    test(S("abcdefghij"), 0, 11, S("abcdefghij"), 1, 1, -1);
    test(S("abcdefghij"), 0, 11, S("abcdefghij"), 1, 4, -1);
    test(S("abcdefghij"), 0, 11, S("abcdefghij"), 1, 8, -1);
    test(S("abcdefghij"), 0, 11, S("abcdefghij"), 1, 9, -1);
    test(S("abcdefghij"), 0, 11, S("abcdefghij"), 1, 10, -1);
    test(S("abcdefghij"), 0, 11, S("abcdefghij"), 5, 0, 10);
    test(S("abcdefghij"), 0, 11, S("abcdefghij"), 5, 1, -5);
    test(S("abcdefghij"), 0, 11, S("abcdefghij"), 5, 2, -5);
    test(S("abcdefghij"), 0, 11, S("abcdefghij"), 5, 4, -5);
    test(S("abcdefghij"), 0, 11, S("abcdefghij"), 5, 5, -5);
    test(S("abcdefghij"), 0, 11, S("abcdefghij"), 5, 6, -5);
    test(S("abcdefghij"), 0, 11, S("abcdefghij"), 9, 0, 10);
    test(S("abcdefghij"), 0, 11, S("abcdefghij"), 9, 1, -9);
    test(S("abcdefghij"), 0, 11, S("abcdefghij"), 9, 2, -9);
    test(S("abcdefghij"), 0, 11, S("abcdefghij"), 10, 0, 10);
    test(S("abcdefghij"), 0, 11, S("abcdefghij"), 10, 1, 10);
    test(S("abcdefghij"), 0, 11, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghij"), 0, 11, S("abcdefghijklmnopqrst"), 0, 0, 10);
    test(S("abcdefghij"), 0, 11, S("abcdefghijklmnopqrst"), 0, 1, 9);
    test(S("abcdefghij"), 0, 11, S("abcdefghijklmnopqrst"), 0, 10, 0);
    test(S("abcdefghij"), 0, 11, S("abcdefghijklmnopqrst"), 0, 19, -9);
    test(S("abcdefghij"), 0, 11, S("abcdefghijklmnopqrst"), 0, 20, -10);
    test(S("abcdefghij"), 0, 11, S("abcdefghijklmnopqrst"), 0, 21, -10);
    test(S("abcdefghij"), 0, 11, S("abcdefghijklmnopqrst"), 1, 0, 10);
    test(S("abcdefghij"), 0, 11, S("abcdefghijklmnopqrst"), 1, 1, -1);
    test(S("abcdefghij"), 0, 11, S("abcdefghijklmnopqrst"), 1, 9, -1);
    test(S("abcdefghij"), 0, 11, S("abcdefghijklmnopqrst"), 1, 18, -1);
    test(S("abcdefghij"), 0, 11, S("abcdefghijklmnopqrst"), 1, 19, -1);
    test(S("abcdefghij"), 0, 11, S("abcdefghijklmnopqrst"), 1, 20, -1);
    test(S("abcdefghij"), 0, 11, S("abcdefghijklmnopqrst"), 10, 0, 10);
    test(S("abcdefghij"), 0, 11, S("abcdefghijklmnopqrst"), 10, 1, -10);
    test(S("abcdefghij"), 0, 11, S("abcdefghijklmnopqrst"), 10, 5, -10);
    test(S("abcdefghij"), 0, 11, S("abcdefghijklmnopqrst"), 10, 9, -10);
    test(S("abcdefghij"), 0, 11, S("abcdefghijklmnopqrst"), 10, 10, -10);
    test(S("abcdefghij"), 0, 11, S("abcdefghijklmnopqrst"), 10, 11, -10);
    test(S("abcdefghij"), 0, 11, S("abcdefghijklmnopqrst"), 19, 0, 10);
    test(S("abcdefghij"), 0, 11, S("abcdefghijklmnopqrst"), 19, 1, -19);
    test(S("abcdefghij"), 0, 11, S("abcdefghijklmnopqrst"), 19, 2, -19);
    test(S("abcdefghij"), 0, 11, S("abcdefghijklmnopqrst"), 20, 0, 10);
    test(S("abcdefghij"), 0, 11, S("abcdefghijklmnopqrst"), 20, 1, 10);
    test(S("abcdefghij"), 0, 11, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghij"), 1, 0, S(""), 0, 0, 0);
    test(S("abcdefghij"), 1, 0, S(""), 0, 1, 0);
    test(S("abcdefghij"), 1, 0, S(""), 1, 0, 0);
    test(S("abcdefghij"), 1, 0, S("abcde"), 0, 0, 0);
    test(S("abcdefghij"), 1, 0, S("abcde"), 0, 1, -1);
    test(S("abcdefghij"), 1, 0, S("abcde"), 0, 2, -2);
    test(S("abcdefghij"), 1, 0, S("abcde"), 0, 4, -4);
    test(S("abcdefghij"), 1, 0, S("abcde"), 0, 5, -5);
    test(S("abcdefghij"), 1, 0, S("abcde"), 0, 6, -5);
    test(S("abcdefghij"), 1, 0, S("abcde"), 1, 0, 0);
    test(S("abcdefghij"), 1, 0, S("abcde"), 1, 1, -1);
    test(S("abcdefghij"), 1, 0, S("abcde"), 1, 2, -2);
    test(S("abcdefghij"), 1, 0, S("abcde"), 1, 3, -3);
    test(S("abcdefghij"), 1, 0, S("abcde"), 1, 4, -4);
    test(S("abcdefghij"), 1, 0, S("abcde"), 1, 5, -4);
    test(S("abcdefghij"), 1, 0, S("abcde"), 2, 0, 0);
    test(S("abcdefghij"), 1, 0, S("abcde"), 2, 1, -1);
    test(S("abcdefghij"), 1, 0, S("abcde"), 2, 2, -2);
    test(S("abcdefghij"), 1, 0, S("abcde"), 2, 3, -3);
    test(S("abcdefghij"), 1, 0, S("abcde"), 2, 4, -3);
    test(S("abcdefghij"), 1, 0, S("abcde"), 4, 0, 0);
    test(S("abcdefghij"), 1, 0, S("abcde"), 4, 1, -1);
    test(S("abcdefghij"), 1, 0, S("abcde"), 4, 2, -1);
    test(S("abcdefghij"), 1, 0, S("abcde"), 5, 0, 0);
    test(S("abcdefghij"), 1, 0, S("abcde"), 5, 1, 0);
    test(S("abcdefghij"), 1, 0, S("abcde"), 6, 0, 0);
    test(S("abcdefghij"), 1, 0, S("abcdefghij"), 0, 0, 0);
    test(S("abcdefghij"), 1, 0, S("abcdefghij"), 0, 1, -1);
    test(S("abcdefghij"), 1, 0, S("abcdefghij"), 0, 5, -5);
    test(S("abcdefghij"), 1, 0, S("abcdefghij"), 0, 9, -9);
    test(S("abcdefghij"), 1, 0, S("abcdefghij"), 0, 10, -10);
    test(S("abcdefghij"), 1, 0, S("abcdefghij"), 0, 11, -10);
}

template <class S>
void test24()
{
    test(S("abcdefghij"), 1, 0, S("abcdefghij"), 1, 0, 0);
    test(S("abcdefghij"), 1, 0, S("abcdefghij"), 1, 1, -1);
    test(S("abcdefghij"), 1, 0, S("abcdefghij"), 1, 4, -4);
    test(S("abcdefghij"), 1, 0, S("abcdefghij"), 1, 8, -8);
    test(S("abcdefghij"), 1, 0, S("abcdefghij"), 1, 9, -9);
    test(S("abcdefghij"), 1, 0, S("abcdefghij"), 1, 10, -9);
    test(S("abcdefghij"), 1, 0, S("abcdefghij"), 5, 0, 0);
    test(S("abcdefghij"), 1, 0, S("abcdefghij"), 5, 1, -1);
    test(S("abcdefghij"), 1, 0, S("abcdefghij"), 5, 2, -2);
    test(S("abcdefghij"), 1, 0, S("abcdefghij"), 5, 4, -4);
    test(S("abcdefghij"), 1, 0, S("abcdefghij"), 5, 5, -5);
    test(S("abcdefghij"), 1, 0, S("abcdefghij"), 5, 6, -5);
    test(S("abcdefghij"), 1, 0, S("abcdefghij"), 9, 0, 0);
    test(S("abcdefghij"), 1, 0, S("abcdefghij"), 9, 1, -1);
    test(S("abcdefghij"), 1, 0, S("abcdefghij"), 9, 2, -1);
    test(S("abcdefghij"), 1, 0, S("abcdefghij"), 10, 0, 0);
    test(S("abcdefghij"), 1, 0, S("abcdefghij"), 10, 1, 0);
    test(S("abcdefghij"), 1, 0, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghij"), 1, 0, S("abcdefghijklmnopqrst"), 0, 0, 0);
    test(S("abcdefghij"), 1, 0, S("abcdefghijklmnopqrst"), 0, 1, -1);
    test(S("abcdefghij"), 1, 0, S("abcdefghijklmnopqrst"), 0, 10, -10);
    test(S("abcdefghij"), 1, 0, S("abcdefghijklmnopqrst"), 0, 19, -19);
    test(S("abcdefghij"), 1, 0, S("abcdefghijklmnopqrst"), 0, 20, -20);
    test(S("abcdefghij"), 1, 0, S("abcdefghijklmnopqrst"), 0, 21, -20);
    test(S("abcdefghij"), 1, 0, S("abcdefghijklmnopqrst"), 1, 0, 0);
    test(S("abcdefghij"), 1, 0, S("abcdefghijklmnopqrst"), 1, 1, -1);
    test(S("abcdefghij"), 1, 0, S("abcdefghijklmnopqrst"), 1, 9, -9);
    test(S("abcdefghij"), 1, 0, S("abcdefghijklmnopqrst"), 1, 18, -18);
    test(S("abcdefghij"), 1, 0, S("abcdefghijklmnopqrst"), 1, 19, -19);
    test(S("abcdefghij"), 1, 0, S("abcdefghijklmnopqrst"), 1, 20, -19);
    test(S("abcdefghij"), 1, 0, S("abcdefghijklmnopqrst"), 10, 0, 0);
    test(S("abcdefghij"), 1, 0, S("abcdefghijklmnopqrst"), 10, 1, -1);
    test(S("abcdefghij"), 1, 0, S("abcdefghijklmnopqrst"), 10, 5, -5);
    test(S("abcdefghij"), 1, 0, S("abcdefghijklmnopqrst"), 10, 9, -9);
    test(S("abcdefghij"), 1, 0, S("abcdefghijklmnopqrst"), 10, 10, -10);
    test(S("abcdefghij"), 1, 0, S("abcdefghijklmnopqrst"), 10, 11, -10);
    test(S("abcdefghij"), 1, 0, S("abcdefghijklmnopqrst"), 19, 0, 0);
    test(S("abcdefghij"), 1, 0, S("abcdefghijklmnopqrst"), 19, 1, -1);
    test(S("abcdefghij"), 1, 0, S("abcdefghijklmnopqrst"), 19, 2, -1);
    test(S("abcdefghij"), 1, 0, S("abcdefghijklmnopqrst"), 20, 0, 0);
    test(S("abcdefghij"), 1, 0, S("abcdefghijklmnopqrst"), 20, 1, 0);
    test(S("abcdefghij"), 1, 0, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghij"), 1, 1, S(""), 0, 0, 1);
    test(S("abcdefghij"), 1, 1, S(""), 0, 1, 1);
    test(S("abcdefghij"), 1, 1, S(""), 1, 0, 0);
    test(S("abcdefghij"), 1, 1, S("abcde"), 0, 0, 1);
    test(S("abcdefghij"), 1, 1, S("abcde"), 0, 1, 1);
    test(S("abcdefghij"), 1, 1, S("abcde"), 0, 2, 1);
    test(S("abcdefghij"), 1, 1, S("abcde"), 0, 4, 1);
    test(S("abcdefghij"), 1, 1, S("abcde"), 0, 5, 1);
    test(S("abcdefghij"), 1, 1, S("abcde"), 0, 6, 1);
    test(S("abcdefghij"), 1, 1, S("abcde"), 1, 0, 1);
    test(S("abcdefghij"), 1, 1, S("abcde"), 1, 1, 0);
    test(S("abcdefghij"), 1, 1, S("abcde"), 1, 2, -1);
    test(S("abcdefghij"), 1, 1, S("abcde"), 1, 3, -2);
    test(S("abcdefghij"), 1, 1, S("abcde"), 1, 4, -3);
    test(S("abcdefghij"), 1, 1, S("abcde"), 1, 5, -3);
    test(S("abcdefghij"), 1, 1, S("abcde"), 2, 0, 1);
    test(S("abcdefghij"), 1, 1, S("abcde"), 2, 1, -1);
    test(S("abcdefghij"), 1, 1, S("abcde"), 2, 2, -1);
    test(S("abcdefghij"), 1, 1, S("abcde"), 2, 3, -1);
    test(S("abcdefghij"), 1, 1, S("abcde"), 2, 4, -1);
    test(S("abcdefghij"), 1, 1, S("abcde"), 4, 0, 1);
    test(S("abcdefghij"), 1, 1, S("abcde"), 4, 1, -3);
    test(S("abcdefghij"), 1, 1, S("abcde"), 4, 2, -3);
    test(S("abcdefghij"), 1, 1, S("abcde"), 5, 0, 1);
    test(S("abcdefghij"), 1, 1, S("abcde"), 5, 1, 1);
    test(S("abcdefghij"), 1, 1, S("abcde"), 6, 0, 0);
    test(S("abcdefghij"), 1, 1, S("abcdefghij"), 0, 0, 1);
    test(S("abcdefghij"), 1, 1, S("abcdefghij"), 0, 1, 1);
    test(S("abcdefghij"), 1, 1, S("abcdefghij"), 0, 5, 1);
    test(S("abcdefghij"), 1, 1, S("abcdefghij"), 0, 9, 1);
    test(S("abcdefghij"), 1, 1, S("abcdefghij"), 0, 10, 1);
    test(S("abcdefghij"), 1, 1, S("abcdefghij"), 0, 11, 1);
    test(S("abcdefghij"), 1, 1, S("abcdefghij"), 1, 0, 1);
    test(S("abcdefghij"), 1, 1, S("abcdefghij"), 1, 1, 0);
    test(S("abcdefghij"), 1, 1, S("abcdefghij"), 1, 4, -3);
    test(S("abcdefghij"), 1, 1, S("abcdefghij"), 1, 8, -7);
    test(S("abcdefghij"), 1, 1, S("abcdefghij"), 1, 9, -8);
    test(S("abcdefghij"), 1, 1, S("abcdefghij"), 1, 10, -8);
    test(S("abcdefghij"), 1, 1, S("abcdefghij"), 5, 0, 1);
    test(S("abcdefghij"), 1, 1, S("abcdefghij"), 5, 1, -4);
    test(S("abcdefghij"), 1, 1, S("abcdefghij"), 5, 2, -4);
    test(S("abcdefghij"), 1, 1, S("abcdefghij"), 5, 4, -4);
    test(S("abcdefghij"), 1, 1, S("abcdefghij"), 5, 5, -4);
    test(S("abcdefghij"), 1, 1, S("abcdefghij"), 5, 6, -4);
    test(S("abcdefghij"), 1, 1, S("abcdefghij"), 9, 0, 1);
    test(S("abcdefghij"), 1, 1, S("abcdefghij"), 9, 1, -8);
    test(S("abcdefghij"), 1, 1, S("abcdefghij"), 9, 2, -8);
    test(S("abcdefghij"), 1, 1, S("abcdefghij"), 10, 0, 1);
    test(S("abcdefghij"), 1, 1, S("abcdefghij"), 10, 1, 1);
    test(S("abcdefghij"), 1, 1, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghij"), 1, 1, S("abcdefghijklmnopqrst"), 0, 0, 1);
    test(S("abcdefghij"), 1, 1, S("abcdefghijklmnopqrst"), 0, 1, 1);
    test(S("abcdefghij"), 1, 1, S("abcdefghijklmnopqrst"), 0, 10, 1);
    test(S("abcdefghij"), 1, 1, S("abcdefghijklmnopqrst"), 0, 19, 1);
    test(S("abcdefghij"), 1, 1, S("abcdefghijklmnopqrst"), 0, 20, 1);
    test(S("abcdefghij"), 1, 1, S("abcdefghijklmnopqrst"), 0, 21, 1);
    test(S("abcdefghij"), 1, 1, S("abcdefghijklmnopqrst"), 1, 0, 1);
    test(S("abcdefghij"), 1, 1, S("abcdefghijklmnopqrst"), 1, 1, 0);
}

template <class S>
void test25()
{
    test(S("abcdefghij"), 1, 1, S("abcdefghijklmnopqrst"), 1, 9, -8);
    test(S("abcdefghij"), 1, 1, S("abcdefghijklmnopqrst"), 1, 18, -17);
    test(S("abcdefghij"), 1, 1, S("abcdefghijklmnopqrst"), 1, 19, -18);
    test(S("abcdefghij"), 1, 1, S("abcdefghijklmnopqrst"), 1, 20, -18);
    test(S("abcdefghij"), 1, 1, S("abcdefghijklmnopqrst"), 10, 0, 1);
    test(S("abcdefghij"), 1, 1, S("abcdefghijklmnopqrst"), 10, 1, -9);
    test(S("abcdefghij"), 1, 1, S("abcdefghijklmnopqrst"), 10, 5, -9);
    test(S("abcdefghij"), 1, 1, S("abcdefghijklmnopqrst"), 10, 9, -9);
    test(S("abcdefghij"), 1, 1, S("abcdefghijklmnopqrst"), 10, 10, -9);
    test(S("abcdefghij"), 1, 1, S("abcdefghijklmnopqrst"), 10, 11, -9);
    test(S("abcdefghij"), 1, 1, S("abcdefghijklmnopqrst"), 19, 0, 1);
    test(S("abcdefghij"), 1, 1, S("abcdefghijklmnopqrst"), 19, 1, -18);
    test(S("abcdefghij"), 1, 1, S("abcdefghijklmnopqrst"), 19, 2, -18);
    test(S("abcdefghij"), 1, 1, S("abcdefghijklmnopqrst"), 20, 0, 1);
    test(S("abcdefghij"), 1, 1, S("abcdefghijklmnopqrst"), 20, 1, 1);
    test(S("abcdefghij"), 1, 1, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghij"), 1, 4, S(""), 0, 0, 4);
    test(S("abcdefghij"), 1, 4, S(""), 0, 1, 4);
    test(S("abcdefghij"), 1, 4, S(""), 1, 0, 0);
    test(S("abcdefghij"), 1, 4, S("abcde"), 0, 0, 4);
    test(S("abcdefghij"), 1, 4, S("abcde"), 0, 1, 1);
    test(S("abcdefghij"), 1, 4, S("abcde"), 0, 2, 1);
    test(S("abcdefghij"), 1, 4, S("abcde"), 0, 4, 1);
    test(S("abcdefghij"), 1, 4, S("abcde"), 0, 5, 1);
    test(S("abcdefghij"), 1, 4, S("abcde"), 0, 6, 1);
    test(S("abcdefghij"), 1, 4, S("abcde"), 1, 0, 4);
    test(S("abcdefghij"), 1, 4, S("abcde"), 1, 1, 3);
    test(S("abcdefghij"), 1, 4, S("abcde"), 1, 2, 2);
    test(S("abcdefghij"), 1, 4, S("abcde"), 1, 3, 1);
    test(S("abcdefghij"), 1, 4, S("abcde"), 1, 4, 0);
    test(S("abcdefghij"), 1, 4, S("abcde"), 1, 5, 0);
    test(S("abcdefghij"), 1, 4, S("abcde"), 2, 0, 4);
    test(S("abcdefghij"), 1, 4, S("abcde"), 2, 1, -1);
    test(S("abcdefghij"), 1, 4, S("abcde"), 2, 2, -1);
    test(S("abcdefghij"), 1, 4, S("abcde"), 2, 3, -1);
    test(S("abcdefghij"), 1, 4, S("abcde"), 2, 4, -1);
    test(S("abcdefghij"), 1, 4, S("abcde"), 4, 0, 4);
    test(S("abcdefghij"), 1, 4, S("abcde"), 4, 1, -3);
    test(S("abcdefghij"), 1, 4, S("abcde"), 4, 2, -3);
    test(S("abcdefghij"), 1, 4, S("abcde"), 5, 0, 4);
    test(S("abcdefghij"), 1, 4, S("abcde"), 5, 1, 4);
    test(S("abcdefghij"), 1, 4, S("abcde"), 6, 0, 0);
    test(S("abcdefghij"), 1, 4, S("abcdefghij"), 0, 0, 4);
    test(S("abcdefghij"), 1, 4, S("abcdefghij"), 0, 1, 1);
    test(S("abcdefghij"), 1, 4, S("abcdefghij"), 0, 5, 1);
    test(S("abcdefghij"), 1, 4, S("abcdefghij"), 0, 9, 1);
    test(S("abcdefghij"), 1, 4, S("abcdefghij"), 0, 10, 1);
    test(S("abcdefghij"), 1, 4, S("abcdefghij"), 0, 11, 1);
    test(S("abcdefghij"), 1, 4, S("abcdefghij"), 1, 0, 4);
    test(S("abcdefghij"), 1, 4, S("abcdefghij"), 1, 1, 3);
    test(S("abcdefghij"), 1, 4, S("abcdefghij"), 1, 4, 0);
    test(S("abcdefghij"), 1, 4, S("abcdefghij"), 1, 8, -4);
    test(S("abcdefghij"), 1, 4, S("abcdefghij"), 1, 9, -5);
    test(S("abcdefghij"), 1, 4, S("abcdefghij"), 1, 10, -5);
    test(S("abcdefghij"), 1, 4, S("abcdefghij"), 5, 0, 4);
    test(S("abcdefghij"), 1, 4, S("abcdefghij"), 5, 1, -4);
    test(S("abcdefghij"), 1, 4, S("abcdefghij"), 5, 2, -4);
    test(S("abcdefghij"), 1, 4, S("abcdefghij"), 5, 4, -4);
    test(S("abcdefghij"), 1, 4, S("abcdefghij"), 5, 5, -4);
    test(S("abcdefghij"), 1, 4, S("abcdefghij"), 5, 6, -4);
    test(S("abcdefghij"), 1, 4, S("abcdefghij"), 9, 0, 4);
    test(S("abcdefghij"), 1, 4, S("abcdefghij"), 9, 1, -8);
    test(S("abcdefghij"), 1, 4, S("abcdefghij"), 9, 2, -8);
    test(S("abcdefghij"), 1, 4, S("abcdefghij"), 10, 0, 4);
    test(S("abcdefghij"), 1, 4, S("abcdefghij"), 10, 1, 4);
    test(S("abcdefghij"), 1, 4, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghij"), 1, 4, S("abcdefghijklmnopqrst"), 0, 0, 4);
    test(S("abcdefghij"), 1, 4, S("abcdefghijklmnopqrst"), 0, 1, 1);
    test(S("abcdefghij"), 1, 4, S("abcdefghijklmnopqrst"), 0, 10, 1);
    test(S("abcdefghij"), 1, 4, S("abcdefghijklmnopqrst"), 0, 19, 1);
    test(S("abcdefghij"), 1, 4, S("abcdefghijklmnopqrst"), 0, 20, 1);
    test(S("abcdefghij"), 1, 4, S("abcdefghijklmnopqrst"), 0, 21, 1);
    test(S("abcdefghij"), 1, 4, S("abcdefghijklmnopqrst"), 1, 0, 4);
    test(S("abcdefghij"), 1, 4, S("abcdefghijklmnopqrst"), 1, 1, 3);
    test(S("abcdefghij"), 1, 4, S("abcdefghijklmnopqrst"), 1, 9, -5);
    test(S("abcdefghij"), 1, 4, S("abcdefghijklmnopqrst"), 1, 18, -14);
    test(S("abcdefghij"), 1, 4, S("abcdefghijklmnopqrst"), 1, 19, -15);
    test(S("abcdefghij"), 1, 4, S("abcdefghijklmnopqrst"), 1, 20, -15);
    test(S("abcdefghij"), 1, 4, S("abcdefghijklmnopqrst"), 10, 0, 4);
    test(S("abcdefghij"), 1, 4, S("abcdefghijklmnopqrst"), 10, 1, -9);
    test(S("abcdefghij"), 1, 4, S("abcdefghijklmnopqrst"), 10, 5, -9);
    test(S("abcdefghij"), 1, 4, S("abcdefghijklmnopqrst"), 10, 9, -9);
    test(S("abcdefghij"), 1, 4, S("abcdefghijklmnopqrst"), 10, 10, -9);
    test(S("abcdefghij"), 1, 4, S("abcdefghijklmnopqrst"), 10, 11, -9);
    test(S("abcdefghij"), 1, 4, S("abcdefghijklmnopqrst"), 19, 0, 4);
    test(S("abcdefghij"), 1, 4, S("abcdefghijklmnopqrst"), 19, 1, -18);
    test(S("abcdefghij"), 1, 4, S("abcdefghijklmnopqrst"), 19, 2, -18);
    test(S("abcdefghij"), 1, 4, S("abcdefghijklmnopqrst"), 20, 0, 4);
    test(S("abcdefghij"), 1, 4, S("abcdefghijklmnopqrst"), 20, 1, 4);
    test(S("abcdefghij"), 1, 4, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghij"), 1, 8, S(""), 0, 0, 8);
    test(S("abcdefghij"), 1, 8, S(""), 0, 1, 8);
    test(S("abcdefghij"), 1, 8, S(""), 1, 0, 0);
    test(S("abcdefghij"), 1, 8, S("abcde"), 0, 0, 8);
    test(S("abcdefghij"), 1, 8, S("abcde"), 0, 1, 1);
    test(S("abcdefghij"), 1, 8, S("abcde"), 0, 2, 1);
    test(S("abcdefghij"), 1, 8, S("abcde"), 0, 4, 1);
    test(S("abcdefghij"), 1, 8, S("abcde"), 0, 5, 1);
    test(S("abcdefghij"), 1, 8, S("abcde"), 0, 6, 1);
    test(S("abcdefghij"), 1, 8, S("abcde"), 1, 0, 8);
}

template <class S>
void test26()
{
    test(S("abcdefghij"), 1, 8, S("abcde"), 1, 1, 7);
    test(S("abcdefghij"), 1, 8, S("abcde"), 1, 2, 6);
    test(S("abcdefghij"), 1, 8, S("abcde"), 1, 3, 5);
    test(S("abcdefghij"), 1, 8, S("abcde"), 1, 4, 4);
    test(S("abcdefghij"), 1, 8, S("abcde"), 1, 5, 4);
    test(S("abcdefghij"), 1, 8, S("abcde"), 2, 0, 8);
    test(S("abcdefghij"), 1, 8, S("abcde"), 2, 1, -1);
    test(S("abcdefghij"), 1, 8, S("abcde"), 2, 2, -1);
    test(S("abcdefghij"), 1, 8, S("abcde"), 2, 3, -1);
    test(S("abcdefghij"), 1, 8, S("abcde"), 2, 4, -1);
    test(S("abcdefghij"), 1, 8, S("abcde"), 4, 0, 8);
    test(S("abcdefghij"), 1, 8, S("abcde"), 4, 1, -3);
    test(S("abcdefghij"), 1, 8, S("abcde"), 4, 2, -3);
    test(S("abcdefghij"), 1, 8, S("abcde"), 5, 0, 8);
    test(S("abcdefghij"), 1, 8, S("abcde"), 5, 1, 8);
    test(S("abcdefghij"), 1, 8, S("abcde"), 6, 0, 0);
    test(S("abcdefghij"), 1, 8, S("abcdefghij"), 0, 0, 8);
    test(S("abcdefghij"), 1, 8, S("abcdefghij"), 0, 1, 1);
    test(S("abcdefghij"), 1, 8, S("abcdefghij"), 0, 5, 1);
    test(S("abcdefghij"), 1, 8, S("abcdefghij"), 0, 9, 1);
    test(S("abcdefghij"), 1, 8, S("abcdefghij"), 0, 10, 1);
    test(S("abcdefghij"), 1, 8, S("abcdefghij"), 0, 11, 1);
    test(S("abcdefghij"), 1, 8, S("abcdefghij"), 1, 0, 8);
    test(S("abcdefghij"), 1, 8, S("abcdefghij"), 1, 1, 7);
    test(S("abcdefghij"), 1, 8, S("abcdefghij"), 1, 4, 4);
    test(S("abcdefghij"), 1, 8, S("abcdefghij"), 1, 8, 0);
    test(S("abcdefghij"), 1, 8, S("abcdefghij"), 1, 9, -1);
    test(S("abcdefghij"), 1, 8, S("abcdefghij"), 1, 10, -1);
    test(S("abcdefghij"), 1, 8, S("abcdefghij"), 5, 0, 8);
    test(S("abcdefghij"), 1, 8, S("abcdefghij"), 5, 1, -4);
    test(S("abcdefghij"), 1, 8, S("abcdefghij"), 5, 2, -4);
    test(S("abcdefghij"), 1, 8, S("abcdefghij"), 5, 4, -4);
    test(S("abcdefghij"), 1, 8, S("abcdefghij"), 5, 5, -4);
    test(S("abcdefghij"), 1, 8, S("abcdefghij"), 5, 6, -4);
    test(S("abcdefghij"), 1, 8, S("abcdefghij"), 9, 0, 8);
    test(S("abcdefghij"), 1, 8, S("abcdefghij"), 9, 1, -8);
    test(S("abcdefghij"), 1, 8, S("abcdefghij"), 9, 2, -8);
    test(S("abcdefghij"), 1, 8, S("abcdefghij"), 10, 0, 8);
    test(S("abcdefghij"), 1, 8, S("abcdefghij"), 10, 1, 8);
    test(S("abcdefghij"), 1, 8, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghij"), 1, 8, S("abcdefghijklmnopqrst"), 0, 0, 8);
    test(S("abcdefghij"), 1, 8, S("abcdefghijklmnopqrst"), 0, 1, 1);
    test(S("abcdefghij"), 1, 8, S("abcdefghijklmnopqrst"), 0, 10, 1);
    test(S("abcdefghij"), 1, 8, S("abcdefghijklmnopqrst"), 0, 19, 1);
    test(S("abcdefghij"), 1, 8, S("abcdefghijklmnopqrst"), 0, 20, 1);
    test(S("abcdefghij"), 1, 8, S("abcdefghijklmnopqrst"), 0, 21, 1);
    test(S("abcdefghij"), 1, 8, S("abcdefghijklmnopqrst"), 1, 0, 8);
    test(S("abcdefghij"), 1, 8, S("abcdefghijklmnopqrst"), 1, 1, 7);
    test(S("abcdefghij"), 1, 8, S("abcdefghijklmnopqrst"), 1, 9, -1);
    test(S("abcdefghij"), 1, 8, S("abcdefghijklmnopqrst"), 1, 18, -10);
    test(S("abcdefghij"), 1, 8, S("abcdefghijklmnopqrst"), 1, 19, -11);
    test(S("abcdefghij"), 1, 8, S("abcdefghijklmnopqrst"), 1, 20, -11);
    test(S("abcdefghij"), 1, 8, S("abcdefghijklmnopqrst"), 10, 0, 8);
    test(S("abcdefghij"), 1, 8, S("abcdefghijklmnopqrst"), 10, 1, -9);
    test(S("abcdefghij"), 1, 8, S("abcdefghijklmnopqrst"), 10, 5, -9);
    test(S("abcdefghij"), 1, 8, S("abcdefghijklmnopqrst"), 10, 9, -9);
    test(S("abcdefghij"), 1, 8, S("abcdefghijklmnopqrst"), 10, 10, -9);
    test(S("abcdefghij"), 1, 8, S("abcdefghijklmnopqrst"), 10, 11, -9);
    test(S("abcdefghij"), 1, 8, S("abcdefghijklmnopqrst"), 19, 0, 8);
    test(S("abcdefghij"), 1, 8, S("abcdefghijklmnopqrst"), 19, 1, -18);
    test(S("abcdefghij"), 1, 8, S("abcdefghijklmnopqrst"), 19, 2, -18);
    test(S("abcdefghij"), 1, 8, S("abcdefghijklmnopqrst"), 20, 0, 8);
    test(S("abcdefghij"), 1, 8, S("abcdefghijklmnopqrst"), 20, 1, 8);
    test(S("abcdefghij"), 1, 8, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghij"), 1, 9, S(""), 0, 0, 9);
    test(S("abcdefghij"), 1, 9, S(""), 0, 1, 9);
    test(S("abcdefghij"), 1, 9, S(""), 1, 0, 0);
    test(S("abcdefghij"), 1, 9, S("abcde"), 0, 0, 9);
    test(S("abcdefghij"), 1, 9, S("abcde"), 0, 1, 1);
    test(S("abcdefghij"), 1, 9, S("abcde"), 0, 2, 1);
    test(S("abcdefghij"), 1, 9, S("abcde"), 0, 4, 1);
    test(S("abcdefghij"), 1, 9, S("abcde"), 0, 5, 1);
    test(S("abcdefghij"), 1, 9, S("abcde"), 0, 6, 1);
    test(S("abcdefghij"), 1, 9, S("abcde"), 1, 0, 9);
    test(S("abcdefghij"), 1, 9, S("abcde"), 1, 1, 8);
    test(S("abcdefghij"), 1, 9, S("abcde"), 1, 2, 7);
    test(S("abcdefghij"), 1, 9, S("abcde"), 1, 3, 6);
    test(S("abcdefghij"), 1, 9, S("abcde"), 1, 4, 5);
    test(S("abcdefghij"), 1, 9, S("abcde"), 1, 5, 5);
    test(S("abcdefghij"), 1, 9, S("abcde"), 2, 0, 9);
    test(S("abcdefghij"), 1, 9, S("abcde"), 2, 1, -1);
    test(S("abcdefghij"), 1, 9, S("abcde"), 2, 2, -1);
    test(S("abcdefghij"), 1, 9, S("abcde"), 2, 3, -1);
    test(S("abcdefghij"), 1, 9, S("abcde"), 2, 4, -1);
    test(S("abcdefghij"), 1, 9, S("abcde"), 4, 0, 9);
    test(S("abcdefghij"), 1, 9, S("abcde"), 4, 1, -3);
    test(S("abcdefghij"), 1, 9, S("abcde"), 4, 2, -3);
    test(S("abcdefghij"), 1, 9, S("abcde"), 5, 0, 9);
    test(S("abcdefghij"), 1, 9, S("abcde"), 5, 1, 9);
    test(S("abcdefghij"), 1, 9, S("abcde"), 6, 0, 0);
    test(S("abcdefghij"), 1, 9, S("abcdefghij"), 0, 0, 9);
    test(S("abcdefghij"), 1, 9, S("abcdefghij"), 0, 1, 1);
    test(S("abcdefghij"), 1, 9, S("abcdefghij"), 0, 5, 1);
    test(S("abcdefghij"), 1, 9, S("abcdefghij"), 0, 9, 1);
    test(S("abcdefghij"), 1, 9, S("abcdefghij"), 0, 10, 1);
    test(S("abcdefghij"), 1, 9, S("abcdefghij"), 0, 11, 1);
    test(S("abcdefghij"), 1, 9, S("abcdefghij"), 1, 0, 9);
    test(S("abcdefghij"), 1, 9, S("abcdefghij"), 1, 1, 8);
    test(S("abcdefghij"), 1, 9, S("abcdefghij"), 1, 4, 5);
    test(S("abcdefghij"), 1, 9, S("abcdefghij"), 1, 8, 1);
}

template <class S>
void test27()
{
    test(S("abcdefghij"), 1, 9, S("abcdefghij"), 1, 9, 0);
    test(S("abcdefghij"), 1, 9, S("abcdefghij"), 1, 10, 0);
    test(S("abcdefghij"), 1, 9, S("abcdefghij"), 5, 0, 9);
    test(S("abcdefghij"), 1, 9, S("abcdefghij"), 5, 1, -4);
    test(S("abcdefghij"), 1, 9, S("abcdefghij"), 5, 2, -4);
    test(S("abcdefghij"), 1, 9, S("abcdefghij"), 5, 4, -4);
    test(S("abcdefghij"), 1, 9, S("abcdefghij"), 5, 5, -4);
    test(S("abcdefghij"), 1, 9, S("abcdefghij"), 5, 6, -4);
    test(S("abcdefghij"), 1, 9, S("abcdefghij"), 9, 0, 9);
    test(S("abcdefghij"), 1, 9, S("abcdefghij"), 9, 1, -8);
    test(S("abcdefghij"), 1, 9, S("abcdefghij"), 9, 2, -8);
    test(S("abcdefghij"), 1, 9, S("abcdefghij"), 10, 0, 9);
    test(S("abcdefghij"), 1, 9, S("abcdefghij"), 10, 1, 9);
    test(S("abcdefghij"), 1, 9, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghij"), 1, 9, S("abcdefghijklmnopqrst"), 0, 0, 9);
    test(S("abcdefghij"), 1, 9, S("abcdefghijklmnopqrst"), 0, 1, 1);
    test(S("abcdefghij"), 1, 9, S("abcdefghijklmnopqrst"), 0, 10, 1);
    test(S("abcdefghij"), 1, 9, S("abcdefghijklmnopqrst"), 0, 19, 1);
    test(S("abcdefghij"), 1, 9, S("abcdefghijklmnopqrst"), 0, 20, 1);
    test(S("abcdefghij"), 1, 9, S("abcdefghijklmnopqrst"), 0, 21, 1);
    test(S("abcdefghij"), 1, 9, S("abcdefghijklmnopqrst"), 1, 0, 9);
    test(S("abcdefghij"), 1, 9, S("abcdefghijklmnopqrst"), 1, 1, 8);
    test(S("abcdefghij"), 1, 9, S("abcdefghijklmnopqrst"), 1, 9, 0);
    test(S("abcdefghij"), 1, 9, S("abcdefghijklmnopqrst"), 1, 18, -9);
    test(S("abcdefghij"), 1, 9, S("abcdefghijklmnopqrst"), 1, 19, -10);
    test(S("abcdefghij"), 1, 9, S("abcdefghijklmnopqrst"), 1, 20, -10);
    test(S("abcdefghij"), 1, 9, S("abcdefghijklmnopqrst"), 10, 0, 9);
    test(S("abcdefghij"), 1, 9, S("abcdefghijklmnopqrst"), 10, 1, -9);
    test(S("abcdefghij"), 1, 9, S("abcdefghijklmnopqrst"), 10, 5, -9);
    test(S("abcdefghij"), 1, 9, S("abcdefghijklmnopqrst"), 10, 9, -9);
    test(S("abcdefghij"), 1, 9, S("abcdefghijklmnopqrst"), 10, 10, -9);
    test(S("abcdefghij"), 1, 9, S("abcdefghijklmnopqrst"), 10, 11, -9);
    test(S("abcdefghij"), 1, 9, S("abcdefghijklmnopqrst"), 19, 0, 9);
    test(S("abcdefghij"), 1, 9, S("abcdefghijklmnopqrst"), 19, 1, -18);
    test(S("abcdefghij"), 1, 9, S("abcdefghijklmnopqrst"), 19, 2, -18);
    test(S("abcdefghij"), 1, 9, S("abcdefghijklmnopqrst"), 20, 0, 9);
    test(S("abcdefghij"), 1, 9, S("abcdefghijklmnopqrst"), 20, 1, 9);
    test(S("abcdefghij"), 1, 9, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghij"), 1, 10, S(""), 0, 0, 9);
    test(S("abcdefghij"), 1, 10, S(""), 0, 1, 9);
    test(S("abcdefghij"), 1, 10, S(""), 1, 0, 0);
    test(S("abcdefghij"), 1, 10, S("abcde"), 0, 0, 9);
    test(S("abcdefghij"), 1, 10, S("abcde"), 0, 1, 1);
    test(S("abcdefghij"), 1, 10, S("abcde"), 0, 2, 1);
    test(S("abcdefghij"), 1, 10, S("abcde"), 0, 4, 1);
    test(S("abcdefghij"), 1, 10, S("abcde"), 0, 5, 1);
    test(S("abcdefghij"), 1, 10, S("abcde"), 0, 6, 1);
    test(S("abcdefghij"), 1, 10, S("abcde"), 1, 0, 9);
    test(S("abcdefghij"), 1, 10, S("abcde"), 1, 1, 8);
    test(S("abcdefghij"), 1, 10, S("abcde"), 1, 2, 7);
    test(S("abcdefghij"), 1, 10, S("abcde"), 1, 3, 6);
    test(S("abcdefghij"), 1, 10, S("abcde"), 1, 4, 5);
    test(S("abcdefghij"), 1, 10, S("abcde"), 1, 5, 5);
    test(S("abcdefghij"), 1, 10, S("abcde"), 2, 0, 9);
    test(S("abcdefghij"), 1, 10, S("abcde"), 2, 1, -1);
    test(S("abcdefghij"), 1, 10, S("abcde"), 2, 2, -1);
    test(S("abcdefghij"), 1, 10, S("abcde"), 2, 3, -1);
    test(S("abcdefghij"), 1, 10, S("abcde"), 2, 4, -1);
    test(S("abcdefghij"), 1, 10, S("abcde"), 4, 0, 9);
    test(S("abcdefghij"), 1, 10, S("abcde"), 4, 1, -3);
    test(S("abcdefghij"), 1, 10, S("abcde"), 4, 2, -3);
    test(S("abcdefghij"), 1, 10, S("abcde"), 5, 0, 9);
    test(S("abcdefghij"), 1, 10, S("abcde"), 5, 1, 9);
    test(S("abcdefghij"), 1, 10, S("abcde"), 6, 0, 0);
    test(S("abcdefghij"), 1, 10, S("abcdefghij"), 0, 0, 9);
    test(S("abcdefghij"), 1, 10, S("abcdefghij"), 0, 1, 1);
    test(S("abcdefghij"), 1, 10, S("abcdefghij"), 0, 5, 1);
    test(S("abcdefghij"), 1, 10, S("abcdefghij"), 0, 9, 1);
    test(S("abcdefghij"), 1, 10, S("abcdefghij"), 0, 10, 1);
    test(S("abcdefghij"), 1, 10, S("abcdefghij"), 0, 11, 1);
    test(S("abcdefghij"), 1, 10, S("abcdefghij"), 1, 0, 9);
    test(S("abcdefghij"), 1, 10, S("abcdefghij"), 1, 1, 8);
    test(S("abcdefghij"), 1, 10, S("abcdefghij"), 1, 4, 5);
    test(S("abcdefghij"), 1, 10, S("abcdefghij"), 1, 8, 1);
    test(S("abcdefghij"), 1, 10, S("abcdefghij"), 1, 9, 0);
    test(S("abcdefghij"), 1, 10, S("abcdefghij"), 1, 10, 0);
    test(S("abcdefghij"), 1, 10, S("abcdefghij"), 5, 0, 9);
    test(S("abcdefghij"), 1, 10, S("abcdefghij"), 5, 1, -4);
    test(S("abcdefghij"), 1, 10, S("abcdefghij"), 5, 2, -4);
    test(S("abcdefghij"), 1, 10, S("abcdefghij"), 5, 4, -4);
    test(S("abcdefghij"), 1, 10, S("abcdefghij"), 5, 5, -4);
    test(S("abcdefghij"), 1, 10, S("abcdefghij"), 5, 6, -4);
    test(S("abcdefghij"), 1, 10, S("abcdefghij"), 9, 0, 9);
    test(S("abcdefghij"), 1, 10, S("abcdefghij"), 9, 1, -8);
    test(S("abcdefghij"), 1, 10, S("abcdefghij"), 9, 2, -8);
    test(S("abcdefghij"), 1, 10, S("abcdefghij"), 10, 0, 9);
    test(S("abcdefghij"), 1, 10, S("abcdefghij"), 10, 1, 9);
    test(S("abcdefghij"), 1, 10, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghij"), 1, 10, S("abcdefghijklmnopqrst"), 0, 0, 9);
    test(S("abcdefghij"), 1, 10, S("abcdefghijklmnopqrst"), 0, 1, 1);
    test(S("abcdefghij"), 1, 10, S("abcdefghijklmnopqrst"), 0, 10, 1);
    test(S("abcdefghij"), 1, 10, S("abcdefghijklmnopqrst"), 0, 19, 1);
    test(S("abcdefghij"), 1, 10, S("abcdefghijklmnopqrst"), 0, 20, 1);
    test(S("abcdefghij"), 1, 10, S("abcdefghijklmnopqrst"), 0, 21, 1);
    test(S("abcdefghij"), 1, 10, S("abcdefghijklmnopqrst"), 1, 0, 9);
    test(S("abcdefghij"), 1, 10, S("abcdefghijklmnopqrst"), 1, 1, 8);
    test(S("abcdefghij"), 1, 10, S("abcdefghijklmnopqrst"), 1, 9, 0);
    test(S("abcdefghij"), 1, 10, S("abcdefghijklmnopqrst"), 1, 18, -9);
    test(S("abcdefghij"), 1, 10, S("abcdefghijklmnopqrst"), 1, 19, -10);
    test(S("abcdefghij"), 1, 10, S("abcdefghijklmnopqrst"), 1, 20, -10);
}

template <class S>
void test28()
{
    test(S("abcdefghij"), 1, 10, S("abcdefghijklmnopqrst"), 10, 0, 9);
    test(S("abcdefghij"), 1, 10, S("abcdefghijklmnopqrst"), 10, 1, -9);
    test(S("abcdefghij"), 1, 10, S("abcdefghijklmnopqrst"), 10, 5, -9);
    test(S("abcdefghij"), 1, 10, S("abcdefghijklmnopqrst"), 10, 9, -9);
    test(S("abcdefghij"), 1, 10, S("abcdefghijklmnopqrst"), 10, 10, -9);
    test(S("abcdefghij"), 1, 10, S("abcdefghijklmnopqrst"), 10, 11, -9);
    test(S("abcdefghij"), 1, 10, S("abcdefghijklmnopqrst"), 19, 0, 9);
    test(S("abcdefghij"), 1, 10, S("abcdefghijklmnopqrst"), 19, 1, -18);
    test(S("abcdefghij"), 1, 10, S("abcdefghijklmnopqrst"), 19, 2, -18);
    test(S("abcdefghij"), 1, 10, S("abcdefghijklmnopqrst"), 20, 0, 9);
    test(S("abcdefghij"), 1, 10, S("abcdefghijklmnopqrst"), 20, 1, 9);
    test(S("abcdefghij"), 1, 10, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghij"), 5, 0, S(""), 0, 0, 0);
    test(S("abcdefghij"), 5, 0, S(""), 0, 1, 0);
    test(S("abcdefghij"), 5, 0, S(""), 1, 0, 0);
    test(S("abcdefghij"), 5, 0, S("abcde"), 0, 0, 0);
    test(S("abcdefghij"), 5, 0, S("abcde"), 0, 1, -1);
    test(S("abcdefghij"), 5, 0, S("abcde"), 0, 2, -2);
    test(S("abcdefghij"), 5, 0, S("abcde"), 0, 4, -4);
    test(S("abcdefghij"), 5, 0, S("abcde"), 0, 5, -5);
    test(S("abcdefghij"), 5, 0, S("abcde"), 0, 6, -5);
    test(S("abcdefghij"), 5, 0, S("abcde"), 1, 0, 0);
    test(S("abcdefghij"), 5, 0, S("abcde"), 1, 1, -1);
    test(S("abcdefghij"), 5, 0, S("abcde"), 1, 2, -2);
    test(S("abcdefghij"), 5, 0, S("abcde"), 1, 3, -3);
    test(S("abcdefghij"), 5, 0, S("abcde"), 1, 4, -4);
    test(S("abcdefghij"), 5, 0, S("abcde"), 1, 5, -4);
    test(S("abcdefghij"), 5, 0, S("abcde"), 2, 0, 0);
    test(S("abcdefghij"), 5, 0, S("abcde"), 2, 1, -1);
    test(S("abcdefghij"), 5, 0, S("abcde"), 2, 2, -2);
    test(S("abcdefghij"), 5, 0, S("abcde"), 2, 3, -3);
    test(S("abcdefghij"), 5, 0, S("abcde"), 2, 4, -3);
    test(S("abcdefghij"), 5, 0, S("abcde"), 4, 0, 0);
    test(S("abcdefghij"), 5, 0, S("abcde"), 4, 1, -1);
    test(S("abcdefghij"), 5, 0, S("abcde"), 4, 2, -1);
    test(S("abcdefghij"), 5, 0, S("abcde"), 5, 0, 0);
    test(S("abcdefghij"), 5, 0, S("abcde"), 5, 1, 0);
    test(S("abcdefghij"), 5, 0, S("abcde"), 6, 0, 0);
    test(S("abcdefghij"), 5, 0, S("abcdefghij"), 0, 0, 0);
    test(S("abcdefghij"), 5, 0, S("abcdefghij"), 0, 1, -1);
    test(S("abcdefghij"), 5, 0, S("abcdefghij"), 0, 5, -5);
    test(S("abcdefghij"), 5, 0, S("abcdefghij"), 0, 9, -9);
    test(S("abcdefghij"), 5, 0, S("abcdefghij"), 0, 10, -10);
    test(S("abcdefghij"), 5, 0, S("abcdefghij"), 0, 11, -10);
    test(S("abcdefghij"), 5, 0, S("abcdefghij"), 1, 0, 0);
    test(S("abcdefghij"), 5, 0, S("abcdefghij"), 1, 1, -1);
    test(S("abcdefghij"), 5, 0, S("abcdefghij"), 1, 4, -4);
    test(S("abcdefghij"), 5, 0, S("abcdefghij"), 1, 8, -8);
    test(S("abcdefghij"), 5, 0, S("abcdefghij"), 1, 9, -9);
    test(S("abcdefghij"), 5, 0, S("abcdefghij"), 1, 10, -9);
    test(S("abcdefghij"), 5, 0, S("abcdefghij"), 5, 0, 0);
    test(S("abcdefghij"), 5, 0, S("abcdefghij"), 5, 1, -1);
    test(S("abcdefghij"), 5, 0, S("abcdefghij"), 5, 2, -2);
    test(S("abcdefghij"), 5, 0, S("abcdefghij"), 5, 4, -4);
    test(S("abcdefghij"), 5, 0, S("abcdefghij"), 5, 5, -5);
    test(S("abcdefghij"), 5, 0, S("abcdefghij"), 5, 6, -5);
    test(S("abcdefghij"), 5, 0, S("abcdefghij"), 9, 0, 0);
    test(S("abcdefghij"), 5, 0, S("abcdefghij"), 9, 1, -1);
    test(S("abcdefghij"), 5, 0, S("abcdefghij"), 9, 2, -1);
    test(S("abcdefghij"), 5, 0, S("abcdefghij"), 10, 0, 0);
    test(S("abcdefghij"), 5, 0, S("abcdefghij"), 10, 1, 0);
    test(S("abcdefghij"), 5, 0, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghij"), 5, 0, S("abcdefghijklmnopqrst"), 0, 0, 0);
    test(S("abcdefghij"), 5, 0, S("abcdefghijklmnopqrst"), 0, 1, -1);
    test(S("abcdefghij"), 5, 0, S("abcdefghijklmnopqrst"), 0, 10, -10);
    test(S("abcdefghij"), 5, 0, S("abcdefghijklmnopqrst"), 0, 19, -19);
    test(S("abcdefghij"), 5, 0, S("abcdefghijklmnopqrst"), 0, 20, -20);
    test(S("abcdefghij"), 5, 0, S("abcdefghijklmnopqrst"), 0, 21, -20);
    test(S("abcdefghij"), 5, 0, S("abcdefghijklmnopqrst"), 1, 0, 0);
    test(S("abcdefghij"), 5, 0, S("abcdefghijklmnopqrst"), 1, 1, -1);
    test(S("abcdefghij"), 5, 0, S("abcdefghijklmnopqrst"), 1, 9, -9);
    test(S("abcdefghij"), 5, 0, S("abcdefghijklmnopqrst"), 1, 18, -18);
    test(S("abcdefghij"), 5, 0, S("abcdefghijklmnopqrst"), 1, 19, -19);
    test(S("abcdefghij"), 5, 0, S("abcdefghijklmnopqrst"), 1, 20, -19);
    test(S("abcdefghij"), 5, 0, S("abcdefghijklmnopqrst"), 10, 0, 0);
    test(S("abcdefghij"), 5, 0, S("abcdefghijklmnopqrst"), 10, 1, -1);
    test(S("abcdefghij"), 5, 0, S("abcdefghijklmnopqrst"), 10, 5, -5);
    test(S("abcdefghij"), 5, 0, S("abcdefghijklmnopqrst"), 10, 9, -9);
    test(S("abcdefghij"), 5, 0, S("abcdefghijklmnopqrst"), 10, 10, -10);
    test(S("abcdefghij"), 5, 0, S("abcdefghijklmnopqrst"), 10, 11, -10);
    test(S("abcdefghij"), 5, 0, S("abcdefghijklmnopqrst"), 19, 0, 0);
    test(S("abcdefghij"), 5, 0, S("abcdefghijklmnopqrst"), 19, 1, -1);
    test(S("abcdefghij"), 5, 0, S("abcdefghijklmnopqrst"), 19, 2, -1);
    test(S("abcdefghij"), 5, 0, S("abcdefghijklmnopqrst"), 20, 0, 0);
    test(S("abcdefghij"), 5, 0, S("abcdefghijklmnopqrst"), 20, 1, 0);
    test(S("abcdefghij"), 5, 0, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghij"), 5, 1, S(""), 0, 0, 1);
    test(S("abcdefghij"), 5, 1, S(""), 0, 1, 1);
    test(S("abcdefghij"), 5, 1, S(""), 1, 0, 0);
    test(S("abcdefghij"), 5, 1, S("abcde"), 0, 0, 1);
    test(S("abcdefghij"), 5, 1, S("abcde"), 0, 1, 5);
    test(S("abcdefghij"), 5, 1, S("abcde"), 0, 2, 5);
    test(S("abcdefghij"), 5, 1, S("abcde"), 0, 4, 5);
    test(S("abcdefghij"), 5, 1, S("abcde"), 0, 5, 5);
    test(S("abcdefghij"), 5, 1, S("abcde"), 0, 6, 5);
    test(S("abcdefghij"), 5, 1, S("abcde"), 1, 0, 1);
    test(S("abcdefghij"), 5, 1, S("abcde"), 1, 1, 4);
    test(S("abcdefghij"), 5, 1, S("abcde"), 1, 2, 4);
    test(S("abcdefghij"), 5, 1, S("abcde"), 1, 3, 4);
    test(S("abcdefghij"), 5, 1, S("abcde"), 1, 4, 4);
}

template <class S>
void test29()
{
    test(S("abcdefghij"), 5, 1, S("abcde"), 1, 5, 4);
    test(S("abcdefghij"), 5, 1, S("abcde"), 2, 0, 1);
    test(S("abcdefghij"), 5, 1, S("abcde"), 2, 1, 3);
    test(S("abcdefghij"), 5, 1, S("abcde"), 2, 2, 3);
    test(S("abcdefghij"), 5, 1, S("abcde"), 2, 3, 3);
    test(S("abcdefghij"), 5, 1, S("abcde"), 2, 4, 3);
    test(S("abcdefghij"), 5, 1, S("abcde"), 4, 0, 1);
    test(S("abcdefghij"), 5, 1, S("abcde"), 4, 1, 1);
    test(S("abcdefghij"), 5, 1, S("abcde"), 4, 2, 1);
    test(S("abcdefghij"), 5, 1, S("abcde"), 5, 0, 1);
    test(S("abcdefghij"), 5, 1, S("abcde"), 5, 1, 1);
    test(S("abcdefghij"), 5, 1, S("abcde"), 6, 0, 0);
    test(S("abcdefghij"), 5, 1, S("abcdefghij"), 0, 0, 1);
    test(S("abcdefghij"), 5, 1, S("abcdefghij"), 0, 1, 5);
    test(S("abcdefghij"), 5, 1, S("abcdefghij"), 0, 5, 5);
    test(S("abcdefghij"), 5, 1, S("abcdefghij"), 0, 9, 5);
    test(S("abcdefghij"), 5, 1, S("abcdefghij"), 0, 10, 5);
    test(S("abcdefghij"), 5, 1, S("abcdefghij"), 0, 11, 5);
    test(S("abcdefghij"), 5, 1, S("abcdefghij"), 1, 0, 1);
    test(S("abcdefghij"), 5, 1, S("abcdefghij"), 1, 1, 4);
    test(S("abcdefghij"), 5, 1, S("abcdefghij"), 1, 4, 4);
    test(S("abcdefghij"), 5, 1, S("abcdefghij"), 1, 8, 4);
    test(S("abcdefghij"), 5, 1, S("abcdefghij"), 1, 9, 4);
    test(S("abcdefghij"), 5, 1, S("abcdefghij"), 1, 10, 4);
    test(S("abcdefghij"), 5, 1, S("abcdefghij"), 5, 0, 1);
    test(S("abcdefghij"), 5, 1, S("abcdefghij"), 5, 1, 0);
    test(S("abcdefghij"), 5, 1, S("abcdefghij"), 5, 2, -1);
    test(S("abcdefghij"), 5, 1, S("abcdefghij"), 5, 4, -3);
    test(S("abcdefghij"), 5, 1, S("abcdefghij"), 5, 5, -4);
    test(S("abcdefghij"), 5, 1, S("abcdefghij"), 5, 6, -4);
    test(S("abcdefghij"), 5, 1, S("abcdefghij"), 9, 0, 1);
    test(S("abcdefghij"), 5, 1, S("abcdefghij"), 9, 1, -4);
    test(S("abcdefghij"), 5, 1, S("abcdefghij"), 9, 2, -4);
    test(S("abcdefghij"), 5, 1, S("abcdefghij"), 10, 0, 1);
    test(S("abcdefghij"), 5, 1, S("abcdefghij"), 10, 1, 1);
    test(S("abcdefghij"), 5, 1, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghij"), 5, 1, S("abcdefghijklmnopqrst"), 0, 0, 1);
    test(S("abcdefghij"), 5, 1, S("abcdefghijklmnopqrst"), 0, 1, 5);
    test(S("abcdefghij"), 5, 1, S("abcdefghijklmnopqrst"), 0, 10, 5);
    test(S("abcdefghij"), 5, 1, S("abcdefghijklmnopqrst"), 0, 19, 5);
    test(S("abcdefghij"), 5, 1, S("abcdefghijklmnopqrst"), 0, 20, 5);
    test(S("abcdefghij"), 5, 1, S("abcdefghijklmnopqrst"), 0, 21, 5);
    test(S("abcdefghij"), 5, 1, S("abcdefghijklmnopqrst"), 1, 0, 1);
    test(S("abcdefghij"), 5, 1, S("abcdefghijklmnopqrst"), 1, 1, 4);
    test(S("abcdefghij"), 5, 1, S("abcdefghijklmnopqrst"), 1, 9, 4);
    test(S("abcdefghij"), 5, 1, S("abcdefghijklmnopqrst"), 1, 18, 4);
    test(S("abcdefghij"), 5, 1, S("abcdefghijklmnopqrst"), 1, 19, 4);
    test(S("abcdefghij"), 5, 1, S("abcdefghijklmnopqrst"), 1, 20, 4);
    test(S("abcdefghij"), 5, 1, S("abcdefghijklmnopqrst"), 10, 0, 1);
    test(S("abcdefghij"), 5, 1, S("abcdefghijklmnopqrst"), 10, 1, -5);
    test(S("abcdefghij"), 5, 1, S("abcdefghijklmnopqrst"), 10, 5, -5);
    test(S("abcdefghij"), 5, 1, S("abcdefghijklmnopqrst"), 10, 9, -5);
    test(S("abcdefghij"), 5, 1, S("abcdefghijklmnopqrst"), 10, 10, -5);
    test(S("abcdefghij"), 5, 1, S("abcdefghijklmnopqrst"), 10, 11, -5);
    test(S("abcdefghij"), 5, 1, S("abcdefghijklmnopqrst"), 19, 0, 1);
    test(S("abcdefghij"), 5, 1, S("abcdefghijklmnopqrst"), 19, 1, -14);
    test(S("abcdefghij"), 5, 1, S("abcdefghijklmnopqrst"), 19, 2, -14);
    test(S("abcdefghij"), 5, 1, S("abcdefghijklmnopqrst"), 20, 0, 1);
    test(S("abcdefghij"), 5, 1, S("abcdefghijklmnopqrst"), 20, 1, 1);
    test(S("abcdefghij"), 5, 1, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghij"), 5, 2, S(""), 0, 0, 2);
    test(S("abcdefghij"), 5, 2, S(""), 0, 1, 2);
    test(S("abcdefghij"), 5, 2, S(""), 1, 0, 0);
    test(S("abcdefghij"), 5, 2, S("abcde"), 0, 0, 2);
    test(S("abcdefghij"), 5, 2, S("abcde"), 0, 1, 5);
    test(S("abcdefghij"), 5, 2, S("abcde"), 0, 2, 5);
    test(S("abcdefghij"), 5, 2, S("abcde"), 0, 4, 5);
    test(S("abcdefghij"), 5, 2, S("abcde"), 0, 5, 5);
    test(S("abcdefghij"), 5, 2, S("abcde"), 0, 6, 5);
    test(S("abcdefghij"), 5, 2, S("abcde"), 1, 0, 2);
    test(S("abcdefghij"), 5, 2, S("abcde"), 1, 1, 4);
    test(S("abcdefghij"), 5, 2, S("abcde"), 1, 2, 4);
    test(S("abcdefghij"), 5, 2, S("abcde"), 1, 3, 4);
    test(S("abcdefghij"), 5, 2, S("abcde"), 1, 4, 4);
    test(S("abcdefghij"), 5, 2, S("abcde"), 1, 5, 4);
    test(S("abcdefghij"), 5, 2, S("abcde"), 2, 0, 2);
    test(S("abcdefghij"), 5, 2, S("abcde"), 2, 1, 3);
    test(S("abcdefghij"), 5, 2, S("abcde"), 2, 2, 3);
    test(S("abcdefghij"), 5, 2, S("abcde"), 2, 3, 3);
    test(S("abcdefghij"), 5, 2, S("abcde"), 2, 4, 3);
    test(S("abcdefghij"), 5, 2, S("abcde"), 4, 0, 2);
    test(S("abcdefghij"), 5, 2, S("abcde"), 4, 1, 1);
    test(S("abcdefghij"), 5, 2, S("abcde"), 4, 2, 1);
    test(S("abcdefghij"), 5, 2, S("abcde"), 5, 0, 2);
    test(S("abcdefghij"), 5, 2, S("abcde"), 5, 1, 2);
    test(S("abcdefghij"), 5, 2, S("abcde"), 6, 0, 0);
    test(S("abcdefghij"), 5, 2, S("abcdefghij"), 0, 0, 2);
    test(S("abcdefghij"), 5, 2, S("abcdefghij"), 0, 1, 5);
    test(S("abcdefghij"), 5, 2, S("abcdefghij"), 0, 5, 5);
    test(S("abcdefghij"), 5, 2, S("abcdefghij"), 0, 9, 5);
    test(S("abcdefghij"), 5, 2, S("abcdefghij"), 0, 10, 5);
    test(S("abcdefghij"), 5, 2, S("abcdefghij"), 0, 11, 5);
    test(S("abcdefghij"), 5, 2, S("abcdefghij"), 1, 0, 2);
    test(S("abcdefghij"), 5, 2, S("abcdefghij"), 1, 1, 4);
    test(S("abcdefghij"), 5, 2, S("abcdefghij"), 1, 4, 4);
    test(S("abcdefghij"), 5, 2, S("abcdefghij"), 1, 8, 4);
    test(S("abcdefghij"), 5, 2, S("abcdefghij"), 1, 9, 4);
    test(S("abcdefghij"), 5, 2, S("abcdefghij"), 1, 10, 4);
    test(S("abcdefghij"), 5, 2, S("abcdefghij"), 5, 0, 2);
    test(S("abcdefghij"), 5, 2, S("abcdefghij"), 5, 1, 1);
}

template <class S>
void test30()
{
    test(S("abcdefghij"), 5, 2, S("abcdefghij"), 5, 2, 0);
    test(S("abcdefghij"), 5, 2, S("abcdefghij"), 5, 4, -2);
    test(S("abcdefghij"), 5, 2, S("abcdefghij"), 5, 5, -3);
    test(S("abcdefghij"), 5, 2, S("abcdefghij"), 5, 6, -3);
    test(S("abcdefghij"), 5, 2, S("abcdefghij"), 9, 0, 2);
    test(S("abcdefghij"), 5, 2, S("abcdefghij"), 9, 1, -4);
    test(S("abcdefghij"), 5, 2, S("abcdefghij"), 9, 2, -4);
    test(S("abcdefghij"), 5, 2, S("abcdefghij"), 10, 0, 2);
    test(S("abcdefghij"), 5, 2, S("abcdefghij"), 10, 1, 2);
    test(S("abcdefghij"), 5, 2, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghij"), 5, 2, S("abcdefghijklmnopqrst"), 0, 0, 2);
    test(S("abcdefghij"), 5, 2, S("abcdefghijklmnopqrst"), 0, 1, 5);
    test(S("abcdefghij"), 5, 2, S("abcdefghijklmnopqrst"), 0, 10, 5);
    test(S("abcdefghij"), 5, 2, S("abcdefghijklmnopqrst"), 0, 19, 5);
    test(S("abcdefghij"), 5, 2, S("abcdefghijklmnopqrst"), 0, 20, 5);
    test(S("abcdefghij"), 5, 2, S("abcdefghijklmnopqrst"), 0, 21, 5);
    test(S("abcdefghij"), 5, 2, S("abcdefghijklmnopqrst"), 1, 0, 2);
    test(S("abcdefghij"), 5, 2, S("abcdefghijklmnopqrst"), 1, 1, 4);
    test(S("abcdefghij"), 5, 2, S("abcdefghijklmnopqrst"), 1, 9, 4);
    test(S("abcdefghij"), 5, 2, S("abcdefghijklmnopqrst"), 1, 18, 4);
    test(S("abcdefghij"), 5, 2, S("abcdefghijklmnopqrst"), 1, 19, 4);
    test(S("abcdefghij"), 5, 2, S("abcdefghijklmnopqrst"), 1, 20, 4);
    test(S("abcdefghij"), 5, 2, S("abcdefghijklmnopqrst"), 10, 0, 2);
    test(S("abcdefghij"), 5, 2, S("abcdefghijklmnopqrst"), 10, 1, -5);
    test(S("abcdefghij"), 5, 2, S("abcdefghijklmnopqrst"), 10, 5, -5);
    test(S("abcdefghij"), 5, 2, S("abcdefghijklmnopqrst"), 10, 9, -5);
    test(S("abcdefghij"), 5, 2, S("abcdefghijklmnopqrst"), 10, 10, -5);
    test(S("abcdefghij"), 5, 2, S("abcdefghijklmnopqrst"), 10, 11, -5);
    test(S("abcdefghij"), 5, 2, S("abcdefghijklmnopqrst"), 19, 0, 2);
    test(S("abcdefghij"), 5, 2, S("abcdefghijklmnopqrst"), 19, 1, -14);
    test(S("abcdefghij"), 5, 2, S("abcdefghijklmnopqrst"), 19, 2, -14);
    test(S("abcdefghij"), 5, 2, S("abcdefghijklmnopqrst"), 20, 0, 2);
    test(S("abcdefghij"), 5, 2, S("abcdefghijklmnopqrst"), 20, 1, 2);
    test(S("abcdefghij"), 5, 2, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghij"), 5, 4, S(""), 0, 0, 4);
    test(S("abcdefghij"), 5, 4, S(""), 0, 1, 4);
    test(S("abcdefghij"), 5, 4, S(""), 1, 0, 0);
    test(S("abcdefghij"), 5, 4, S("abcde"), 0, 0, 4);
    test(S("abcdefghij"), 5, 4, S("abcde"), 0, 1, 5);
    test(S("abcdefghij"), 5, 4, S("abcde"), 0, 2, 5);
    test(S("abcdefghij"), 5, 4, S("abcde"), 0, 4, 5);
    test(S("abcdefghij"), 5, 4, S("abcde"), 0, 5, 5);
    test(S("abcdefghij"), 5, 4, S("abcde"), 0, 6, 5);
    test(S("abcdefghij"), 5, 4, S("abcde"), 1, 0, 4);
    test(S("abcdefghij"), 5, 4, S("abcde"), 1, 1, 4);
    test(S("abcdefghij"), 5, 4, S("abcde"), 1, 2, 4);
    test(S("abcdefghij"), 5, 4, S("abcde"), 1, 3, 4);
    test(S("abcdefghij"), 5, 4, S("abcde"), 1, 4, 4);
    test(S("abcdefghij"), 5, 4, S("abcde"), 1, 5, 4);
    test(S("abcdefghij"), 5, 4, S("abcde"), 2, 0, 4);
    test(S("abcdefghij"), 5, 4, S("abcde"), 2, 1, 3);
    test(S("abcdefghij"), 5, 4, S("abcde"), 2, 2, 3);
    test(S("abcdefghij"), 5, 4, S("abcde"), 2, 3, 3);
    test(S("abcdefghij"), 5, 4, S("abcde"), 2, 4, 3);
    test(S("abcdefghij"), 5, 4, S("abcde"), 4, 0, 4);
    test(S("abcdefghij"), 5, 4, S("abcde"), 4, 1, 1);
    test(S("abcdefghij"), 5, 4, S("abcde"), 4, 2, 1);
    test(S("abcdefghij"), 5, 4, S("abcde"), 5, 0, 4);
    test(S("abcdefghij"), 5, 4, S("abcde"), 5, 1, 4);
    test(S("abcdefghij"), 5, 4, S("abcde"), 6, 0, 0);
    test(S("abcdefghij"), 5, 4, S("abcdefghij"), 0, 0, 4);
    test(S("abcdefghij"), 5, 4, S("abcdefghij"), 0, 1, 5);
    test(S("abcdefghij"), 5, 4, S("abcdefghij"), 0, 5, 5);
    test(S("abcdefghij"), 5, 4, S("abcdefghij"), 0, 9, 5);
    test(S("abcdefghij"), 5, 4, S("abcdefghij"), 0, 10, 5);
    test(S("abcdefghij"), 5, 4, S("abcdefghij"), 0, 11, 5);
    test(S("abcdefghij"), 5, 4, S("abcdefghij"), 1, 0, 4);
    test(S("abcdefghij"), 5, 4, S("abcdefghij"), 1, 1, 4);
    test(S("abcdefghij"), 5, 4, S("abcdefghij"), 1, 4, 4);
    test(S("abcdefghij"), 5, 4, S("abcdefghij"), 1, 8, 4);
    test(S("abcdefghij"), 5, 4, S("abcdefghij"), 1, 9, 4);
    test(S("abcdefghij"), 5, 4, S("abcdefghij"), 1, 10, 4);
    test(S("abcdefghij"), 5, 4, S("abcdefghij"), 5, 0, 4);
    test(S("abcdefghij"), 5, 4, S("abcdefghij"), 5, 1, 3);
    test(S("abcdefghij"), 5, 4, S("abcdefghij"), 5, 2, 2);
    test(S("abcdefghij"), 5, 4, S("abcdefghij"), 5, 4, 0);
    test(S("abcdefghij"), 5, 4, S("abcdefghij"), 5, 5, -1);
    test(S("abcdefghij"), 5, 4, S("abcdefghij"), 5, 6, -1);
    test(S("abcdefghij"), 5, 4, S("abcdefghij"), 9, 0, 4);
    test(S("abcdefghij"), 5, 4, S("abcdefghij"), 9, 1, -4);
    test(S("abcdefghij"), 5, 4, S("abcdefghij"), 9, 2, -4);
    test(S("abcdefghij"), 5, 4, S("abcdefghij"), 10, 0, 4);
    test(S("abcdefghij"), 5, 4, S("abcdefghij"), 10, 1, 4);
    test(S("abcdefghij"), 5, 4, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghij"), 5, 4, S("abcdefghijklmnopqrst"), 0, 0, 4);
    test(S("abcdefghij"), 5, 4, S("abcdefghijklmnopqrst"), 0, 1, 5);
    test(S("abcdefghij"), 5, 4, S("abcdefghijklmnopqrst"), 0, 10, 5);
    test(S("abcdefghij"), 5, 4, S("abcdefghijklmnopqrst"), 0, 19, 5);
    test(S("abcdefghij"), 5, 4, S("abcdefghijklmnopqrst"), 0, 20, 5);
    test(S("abcdefghij"), 5, 4, S("abcdefghijklmnopqrst"), 0, 21, 5);
    test(S("abcdefghij"), 5, 4, S("abcdefghijklmnopqrst"), 1, 0, 4);
    test(S("abcdefghij"), 5, 4, S("abcdefghijklmnopqrst"), 1, 1, 4);
    test(S("abcdefghij"), 5, 4, S("abcdefghijklmnopqrst"), 1, 9, 4);
    test(S("abcdefghij"), 5, 4, S("abcdefghijklmnopqrst"), 1, 18, 4);
    test(S("abcdefghij"), 5, 4, S("abcdefghijklmnopqrst"), 1, 19, 4);
    test(S("abcdefghij"), 5, 4, S("abcdefghijklmnopqrst"), 1, 20, 4);
    test(S("abcdefghij"), 5, 4, S("abcdefghijklmnopqrst"), 10, 0, 4);
    test(S("abcdefghij"), 5, 4, S("abcdefghijklmnopqrst"), 10, 1, -5);
    test(S("abcdefghij"), 5, 4, S("abcdefghijklmnopqrst"), 10, 5, -5);
    test(S("abcdefghij"), 5, 4, S("abcdefghijklmnopqrst"), 10, 9, -5);
}

template <class S>
void test31()
{
    test(S("abcdefghij"), 5, 4, S("abcdefghijklmnopqrst"), 10, 10, -5);
    test(S("abcdefghij"), 5, 4, S("abcdefghijklmnopqrst"), 10, 11, -5);
    test(S("abcdefghij"), 5, 4, S("abcdefghijklmnopqrst"), 19, 0, 4);
    test(S("abcdefghij"), 5, 4, S("abcdefghijklmnopqrst"), 19, 1, -14);
    test(S("abcdefghij"), 5, 4, S("abcdefghijklmnopqrst"), 19, 2, -14);
    test(S("abcdefghij"), 5, 4, S("abcdefghijklmnopqrst"), 20, 0, 4);
    test(S("abcdefghij"), 5, 4, S("abcdefghijklmnopqrst"), 20, 1, 4);
    test(S("abcdefghij"), 5, 4, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghij"), 5, 5, S(""), 0, 0, 5);
    test(S("abcdefghij"), 5, 5, S(""), 0, 1, 5);
    test(S("abcdefghij"), 5, 5, S(""), 1, 0, 0);
    test(S("abcdefghij"), 5, 5, S("abcde"), 0, 0, 5);
    test(S("abcdefghij"), 5, 5, S("abcde"), 0, 1, 5);
    test(S("abcdefghij"), 5, 5, S("abcde"), 0, 2, 5);
    test(S("abcdefghij"), 5, 5, S("abcde"), 0, 4, 5);
    test(S("abcdefghij"), 5, 5, S("abcde"), 0, 5, 5);
    test(S("abcdefghij"), 5, 5, S("abcde"), 0, 6, 5);
    test(S("abcdefghij"), 5, 5, S("abcde"), 1, 0, 5);
    test(S("abcdefghij"), 5, 5, S("abcde"), 1, 1, 4);
    test(S("abcdefghij"), 5, 5, S("abcde"), 1, 2, 4);
    test(S("abcdefghij"), 5, 5, S("abcde"), 1, 3, 4);
    test(S("abcdefghij"), 5, 5, S("abcde"), 1, 4, 4);
    test(S("abcdefghij"), 5, 5, S("abcde"), 1, 5, 4);
    test(S("abcdefghij"), 5, 5, S("abcde"), 2, 0, 5);
    test(S("abcdefghij"), 5, 5, S("abcde"), 2, 1, 3);
    test(S("abcdefghij"), 5, 5, S("abcde"), 2, 2, 3);
    test(S("abcdefghij"), 5, 5, S("abcde"), 2, 3, 3);
    test(S("abcdefghij"), 5, 5, S("abcde"), 2, 4, 3);
    test(S("abcdefghij"), 5, 5, S("abcde"), 4, 0, 5);
    test(S("abcdefghij"), 5, 5, S("abcde"), 4, 1, 1);
    test(S("abcdefghij"), 5, 5, S("abcde"), 4, 2, 1);
    test(S("abcdefghij"), 5, 5, S("abcde"), 5, 0, 5);
    test(S("abcdefghij"), 5, 5, S("abcde"), 5, 1, 5);
    test(S("abcdefghij"), 5, 5, S("abcde"), 6, 0, 0);
    test(S("abcdefghij"), 5, 5, S("abcdefghij"), 0, 0, 5);
    test(S("abcdefghij"), 5, 5, S("abcdefghij"), 0, 1, 5);
    test(S("abcdefghij"), 5, 5, S("abcdefghij"), 0, 5, 5);
    test(S("abcdefghij"), 5, 5, S("abcdefghij"), 0, 9, 5);
    test(S("abcdefghij"), 5, 5, S("abcdefghij"), 0, 10, 5);
    test(S("abcdefghij"), 5, 5, S("abcdefghij"), 0, 11, 5);
    test(S("abcdefghij"), 5, 5, S("abcdefghij"), 1, 0, 5);
    test(S("abcdefghij"), 5, 5, S("abcdefghij"), 1, 1, 4);
    test(S("abcdefghij"), 5, 5, S("abcdefghij"), 1, 4, 4);
    test(S("abcdefghij"), 5, 5, S("abcdefghij"), 1, 8, 4);
    test(S("abcdefghij"), 5, 5, S("abcdefghij"), 1, 9, 4);
    test(S("abcdefghij"), 5, 5, S("abcdefghij"), 1, 10, 4);
    test(S("abcdefghij"), 5, 5, S("abcdefghij"), 5, 0, 5);
    test(S("abcdefghij"), 5, 5, S("abcdefghij"), 5, 1, 4);
    test(S("abcdefghij"), 5, 5, S("abcdefghij"), 5, 2, 3);
    test(S("abcdefghij"), 5, 5, S("abcdefghij"), 5, 4, 1);
    test(S("abcdefghij"), 5, 5, S("abcdefghij"), 5, 5, 0);
    test(S("abcdefghij"), 5, 5, S("abcdefghij"), 5, 6, 0);
    test(S("abcdefghij"), 5, 5, S("abcdefghij"), 9, 0, 5);
    test(S("abcdefghij"), 5, 5, S("abcdefghij"), 9, 1, -4);
    test(S("abcdefghij"), 5, 5, S("abcdefghij"), 9, 2, -4);
    test(S("abcdefghij"), 5, 5, S("abcdefghij"), 10, 0, 5);
    test(S("abcdefghij"), 5, 5, S("abcdefghij"), 10, 1, 5);
    test(S("abcdefghij"), 5, 5, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghij"), 5, 5, S("abcdefghijklmnopqrst"), 0, 0, 5);
    test(S("abcdefghij"), 5, 5, S("abcdefghijklmnopqrst"), 0, 1, 5);
    test(S("abcdefghij"), 5, 5, S("abcdefghijklmnopqrst"), 0, 10, 5);
    test(S("abcdefghij"), 5, 5, S("abcdefghijklmnopqrst"), 0, 19, 5);
    test(S("abcdefghij"), 5, 5, S("abcdefghijklmnopqrst"), 0, 20, 5);
    test(S("abcdefghij"), 5, 5, S("abcdefghijklmnopqrst"), 0, 21, 5);
    test(S("abcdefghij"), 5, 5, S("abcdefghijklmnopqrst"), 1, 0, 5);
    test(S("abcdefghij"), 5, 5, S("abcdefghijklmnopqrst"), 1, 1, 4);
    test(S("abcdefghij"), 5, 5, S("abcdefghijklmnopqrst"), 1, 9, 4);
    test(S("abcdefghij"), 5, 5, S("abcdefghijklmnopqrst"), 1, 18, 4);
    test(S("abcdefghij"), 5, 5, S("abcdefghijklmnopqrst"), 1, 19, 4);
    test(S("abcdefghij"), 5, 5, S("abcdefghijklmnopqrst"), 1, 20, 4);
    test(S("abcdefghij"), 5, 5, S("abcdefghijklmnopqrst"), 10, 0, 5);
    test(S("abcdefghij"), 5, 5, S("abcdefghijklmnopqrst"), 10, 1, -5);
    test(S("abcdefghij"), 5, 5, S("abcdefghijklmnopqrst"), 10, 5, -5);
    test(S("abcdefghij"), 5, 5, S("abcdefghijklmnopqrst"), 10, 9, -5);
    test(S("abcdefghij"), 5, 5, S("abcdefghijklmnopqrst"), 10, 10, -5);
    test(S("abcdefghij"), 5, 5, S("abcdefghijklmnopqrst"), 10, 11, -5);
    test(S("abcdefghij"), 5, 5, S("abcdefghijklmnopqrst"), 19, 0, 5);
    test(S("abcdefghij"), 5, 5, S("abcdefghijklmnopqrst"), 19, 1, -14);
    test(S("abcdefghij"), 5, 5, S("abcdefghijklmnopqrst"), 19, 2, -14);
    test(S("abcdefghij"), 5, 5, S("abcdefghijklmnopqrst"), 20, 0, 5);
    test(S("abcdefghij"), 5, 5, S("abcdefghijklmnopqrst"), 20, 1, 5);
    test(S("abcdefghij"), 5, 5, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghij"), 5, 6, S(""), 0, 0, 5);
    test(S("abcdefghij"), 5, 6, S(""), 0, 1, 5);
    test(S("abcdefghij"), 5, 6, S(""), 1, 0, 0);
    test(S("abcdefghij"), 5, 6, S("abcde"), 0, 0, 5);
    test(S("abcdefghij"), 5, 6, S("abcde"), 0, 1, 5);
    test(S("abcdefghij"), 5, 6, S("abcde"), 0, 2, 5);
    test(S("abcdefghij"), 5, 6, S("abcde"), 0, 4, 5);
    test(S("abcdefghij"), 5, 6, S("abcde"), 0, 5, 5);
    test(S("abcdefghij"), 5, 6, S("abcde"), 0, 6, 5);
    test(S("abcdefghij"), 5, 6, S("abcde"), 1, 0, 5);
    test(S("abcdefghij"), 5, 6, S("abcde"), 1, 1, 4);
    test(S("abcdefghij"), 5, 6, S("abcde"), 1, 2, 4);
    test(S("abcdefghij"), 5, 6, S("abcde"), 1, 3, 4);
    test(S("abcdefghij"), 5, 6, S("abcde"), 1, 4, 4);
    test(S("abcdefghij"), 5, 6, S("abcde"), 1, 5, 4);
    test(S("abcdefghij"), 5, 6, S("abcde"), 2, 0, 5);
    test(S("abcdefghij"), 5, 6, S("abcde"), 2, 1, 3);
    test(S("abcdefghij"), 5, 6, S("abcde"), 2, 2, 3);
}

template <class S>
void test32()
{
    test(S("abcdefghij"), 5, 6, S("abcde"), 2, 3, 3);
    test(S("abcdefghij"), 5, 6, S("abcde"), 2, 4, 3);
    test(S("abcdefghij"), 5, 6, S("abcde"), 4, 0, 5);
    test(S("abcdefghij"), 5, 6, S("abcde"), 4, 1, 1);
    test(S("abcdefghij"), 5, 6, S("abcde"), 4, 2, 1);
    test(S("abcdefghij"), 5, 6, S("abcde"), 5, 0, 5);
    test(S("abcdefghij"), 5, 6, S("abcde"), 5, 1, 5);
    test(S("abcdefghij"), 5, 6, S("abcde"), 6, 0, 0);
    test(S("abcdefghij"), 5, 6, S("abcdefghij"), 0, 0, 5);
    test(S("abcdefghij"), 5, 6, S("abcdefghij"), 0, 1, 5);
    test(S("abcdefghij"), 5, 6, S("abcdefghij"), 0, 5, 5);
    test(S("abcdefghij"), 5, 6, S("abcdefghij"), 0, 9, 5);
    test(S("abcdefghij"), 5, 6, S("abcdefghij"), 0, 10, 5);
    test(S("abcdefghij"), 5, 6, S("abcdefghij"), 0, 11, 5);
    test(S("abcdefghij"), 5, 6, S("abcdefghij"), 1, 0, 5);
    test(S("abcdefghij"), 5, 6, S("abcdefghij"), 1, 1, 4);
    test(S("abcdefghij"), 5, 6, S("abcdefghij"), 1, 4, 4);
    test(S("abcdefghij"), 5, 6, S("abcdefghij"), 1, 8, 4);
    test(S("abcdefghij"), 5, 6, S("abcdefghij"), 1, 9, 4);
    test(S("abcdefghij"), 5, 6, S("abcdefghij"), 1, 10, 4);
    test(S("abcdefghij"), 5, 6, S("abcdefghij"), 5, 0, 5);
    test(S("abcdefghij"), 5, 6, S("abcdefghij"), 5, 1, 4);
    test(S("abcdefghij"), 5, 6, S("abcdefghij"), 5, 2, 3);
    test(S("abcdefghij"), 5, 6, S("abcdefghij"), 5, 4, 1);
    test(S("abcdefghij"), 5, 6, S("abcdefghij"), 5, 5, 0);
    test(S("abcdefghij"), 5, 6, S("abcdefghij"), 5, 6, 0);
    test(S("abcdefghij"), 5, 6, S("abcdefghij"), 9, 0, 5);
    test(S("abcdefghij"), 5, 6, S("abcdefghij"), 9, 1, -4);
    test(S("abcdefghij"), 5, 6, S("abcdefghij"), 9, 2, -4);
    test(S("abcdefghij"), 5, 6, S("abcdefghij"), 10, 0, 5);
    test(S("abcdefghij"), 5, 6, S("abcdefghij"), 10, 1, 5);
    test(S("abcdefghij"), 5, 6, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghij"), 5, 6, S("abcdefghijklmnopqrst"), 0, 0, 5);
    test(S("abcdefghij"), 5, 6, S("abcdefghijklmnopqrst"), 0, 1, 5);
    test(S("abcdefghij"), 5, 6, S("abcdefghijklmnopqrst"), 0, 10, 5);
    test(S("abcdefghij"), 5, 6, S("abcdefghijklmnopqrst"), 0, 19, 5);
    test(S("abcdefghij"), 5, 6, S("abcdefghijklmnopqrst"), 0, 20, 5);
    test(S("abcdefghij"), 5, 6, S("abcdefghijklmnopqrst"), 0, 21, 5);
    test(S("abcdefghij"), 5, 6, S("abcdefghijklmnopqrst"), 1, 0, 5);
    test(S("abcdefghij"), 5, 6, S("abcdefghijklmnopqrst"), 1, 1, 4);
    test(S("abcdefghij"), 5, 6, S("abcdefghijklmnopqrst"), 1, 9, 4);
    test(S("abcdefghij"), 5, 6, S("abcdefghijklmnopqrst"), 1, 18, 4);
    test(S("abcdefghij"), 5, 6, S("abcdefghijklmnopqrst"), 1, 19, 4);
    test(S("abcdefghij"), 5, 6, S("abcdefghijklmnopqrst"), 1, 20, 4);
    test(S("abcdefghij"), 5, 6, S("abcdefghijklmnopqrst"), 10, 0, 5);
    test(S("abcdefghij"), 5, 6, S("abcdefghijklmnopqrst"), 10, 1, -5);
    test(S("abcdefghij"), 5, 6, S("abcdefghijklmnopqrst"), 10, 5, -5);
    test(S("abcdefghij"), 5, 6, S("abcdefghijklmnopqrst"), 10, 9, -5);
    test(S("abcdefghij"), 5, 6, S("abcdefghijklmnopqrst"), 10, 10, -5);
    test(S("abcdefghij"), 5, 6, S("abcdefghijklmnopqrst"), 10, 11, -5);
    test(S("abcdefghij"), 5, 6, S("abcdefghijklmnopqrst"), 19, 0, 5);
    test(S("abcdefghij"), 5, 6, S("abcdefghijklmnopqrst"), 19, 1, -14);
    test(S("abcdefghij"), 5, 6, S("abcdefghijklmnopqrst"), 19, 2, -14);
    test(S("abcdefghij"), 5, 6, S("abcdefghijklmnopqrst"), 20, 0, 5);
    test(S("abcdefghij"), 5, 6, S("abcdefghijklmnopqrst"), 20, 1, 5);
    test(S("abcdefghij"), 5, 6, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghij"), 9, 0, S(""), 0, 0, 0);
    test(S("abcdefghij"), 9, 0, S(""), 0, 1, 0);
    test(S("abcdefghij"), 9, 0, S(""), 1, 0, 0);
    test(S("abcdefghij"), 9, 0, S("abcde"), 0, 0, 0);
    test(S("abcdefghij"), 9, 0, S("abcde"), 0, 1, -1);
    test(S("abcdefghij"), 9, 0, S("abcde"), 0, 2, -2);
    test(S("abcdefghij"), 9, 0, S("abcde"), 0, 4, -4);
    test(S("abcdefghij"), 9, 0, S("abcde"), 0, 5, -5);
    test(S("abcdefghij"), 9, 0, S("abcde"), 0, 6, -5);
    test(S("abcdefghij"), 9, 0, S("abcde"), 1, 0, 0);
    test(S("abcdefghij"), 9, 0, S("abcde"), 1, 1, -1);
    test(S("abcdefghij"), 9, 0, S("abcde"), 1, 2, -2);
    test(S("abcdefghij"), 9, 0, S("abcde"), 1, 3, -3);
    test(S("abcdefghij"), 9, 0, S("abcde"), 1, 4, -4);
    test(S("abcdefghij"), 9, 0, S("abcde"), 1, 5, -4);
    test(S("abcdefghij"), 9, 0, S("abcde"), 2, 0, 0);
    test(S("abcdefghij"), 9, 0, S("abcde"), 2, 1, -1);
    test(S("abcdefghij"), 9, 0, S("abcde"), 2, 2, -2);
    test(S("abcdefghij"), 9, 0, S("abcde"), 2, 3, -3);
    test(S("abcdefghij"), 9, 0, S("abcde"), 2, 4, -3);
    test(S("abcdefghij"), 9, 0, S("abcde"), 4, 0, 0);
    test(S("abcdefghij"), 9, 0, S("abcde"), 4, 1, -1);
    test(S("abcdefghij"), 9, 0, S("abcde"), 4, 2, -1);
    test(S("abcdefghij"), 9, 0, S("abcde"), 5, 0, 0);
    test(S("abcdefghij"), 9, 0, S("abcde"), 5, 1, 0);
    test(S("abcdefghij"), 9, 0, S("abcde"), 6, 0, 0);
    test(S("abcdefghij"), 9, 0, S("abcdefghij"), 0, 0, 0);
    test(S("abcdefghij"), 9, 0, S("abcdefghij"), 0, 1, -1);
    test(S("abcdefghij"), 9, 0, S("abcdefghij"), 0, 5, -5);
    test(S("abcdefghij"), 9, 0, S("abcdefghij"), 0, 9, -9);
    test(S("abcdefghij"), 9, 0, S("abcdefghij"), 0, 10, -10);
    test(S("abcdefghij"), 9, 0, S("abcdefghij"), 0, 11, -10);
    test(S("abcdefghij"), 9, 0, S("abcdefghij"), 1, 0, 0);
    test(S("abcdefghij"), 9, 0, S("abcdefghij"), 1, 1, -1);
    test(S("abcdefghij"), 9, 0, S("abcdefghij"), 1, 4, -4);
    test(S("abcdefghij"), 9, 0, S("abcdefghij"), 1, 8, -8);
    test(S("abcdefghij"), 9, 0, S("abcdefghij"), 1, 9, -9);
    test(S("abcdefghij"), 9, 0, S("abcdefghij"), 1, 10, -9);
    test(S("abcdefghij"), 9, 0, S("abcdefghij"), 5, 0, 0);
    test(S("abcdefghij"), 9, 0, S("abcdefghij"), 5, 1, -1);
    test(S("abcdefghij"), 9, 0, S("abcdefghij"), 5, 2, -2);
    test(S("abcdefghij"), 9, 0, S("abcdefghij"), 5, 4, -4);
    test(S("abcdefghij"), 9, 0, S("abcdefghij"), 5, 5, -5);
    test(S("abcdefghij"), 9, 0, S("abcdefghij"), 5, 6, -5);
}

template <class S>
void test33()
{
    test(S("abcdefghij"), 9, 0, S("abcdefghij"), 9, 0, 0);
    test(S("abcdefghij"), 9, 0, S("abcdefghij"), 9, 1, -1);
    test(S("abcdefghij"), 9, 0, S("abcdefghij"), 9, 2, -1);
    test(S("abcdefghij"), 9, 0, S("abcdefghij"), 10, 0, 0);
    test(S("abcdefghij"), 9, 0, S("abcdefghij"), 10, 1, 0);
    test(S("abcdefghij"), 9, 0, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghij"), 9, 0, S("abcdefghijklmnopqrst"), 0, 0, 0);
    test(S("abcdefghij"), 9, 0, S("abcdefghijklmnopqrst"), 0, 1, -1);
    test(S("abcdefghij"), 9, 0, S("abcdefghijklmnopqrst"), 0, 10, -10);
    test(S("abcdefghij"), 9, 0, S("abcdefghijklmnopqrst"), 0, 19, -19);
    test(S("abcdefghij"), 9, 0, S("abcdefghijklmnopqrst"), 0, 20, -20);
    test(S("abcdefghij"), 9, 0, S("abcdefghijklmnopqrst"), 0, 21, -20);
    test(S("abcdefghij"), 9, 0, S("abcdefghijklmnopqrst"), 1, 0, 0);
    test(S("abcdefghij"), 9, 0, S("abcdefghijklmnopqrst"), 1, 1, -1);
    test(S("abcdefghij"), 9, 0, S("abcdefghijklmnopqrst"), 1, 9, -9);
    test(S("abcdefghij"), 9, 0, S("abcdefghijklmnopqrst"), 1, 18, -18);
    test(S("abcdefghij"), 9, 0, S("abcdefghijklmnopqrst"), 1, 19, -19);
    test(S("abcdefghij"), 9, 0, S("abcdefghijklmnopqrst"), 1, 20, -19);
    test(S("abcdefghij"), 9, 0, S("abcdefghijklmnopqrst"), 10, 0, 0);
    test(S("abcdefghij"), 9, 0, S("abcdefghijklmnopqrst"), 10, 1, -1);
    test(S("abcdefghij"), 9, 0, S("abcdefghijklmnopqrst"), 10, 5, -5);
    test(S("abcdefghij"), 9, 0, S("abcdefghijklmnopqrst"), 10, 9, -9);
    test(S("abcdefghij"), 9, 0, S("abcdefghijklmnopqrst"), 10, 10, -10);
    test(S("abcdefghij"), 9, 0, S("abcdefghijklmnopqrst"), 10, 11, -10);
    test(S("abcdefghij"), 9, 0, S("abcdefghijklmnopqrst"), 19, 0, 0);
    test(S("abcdefghij"), 9, 0, S("abcdefghijklmnopqrst"), 19, 1, -1);
    test(S("abcdefghij"), 9, 0, S("abcdefghijklmnopqrst"), 19, 2, -1);
    test(S("abcdefghij"), 9, 0, S("abcdefghijklmnopqrst"), 20, 0, 0);
    test(S("abcdefghij"), 9, 0, S("abcdefghijklmnopqrst"), 20, 1, 0);
    test(S("abcdefghij"), 9, 0, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghij"), 9, 1, S(""), 0, 0, 1);
    test(S("abcdefghij"), 9, 1, S(""), 0, 1, 1);
    test(S("abcdefghij"), 9, 1, S(""), 1, 0, 0);
    test(S("abcdefghij"), 9, 1, S("abcde"), 0, 0, 1);
    test(S("abcdefghij"), 9, 1, S("abcde"), 0, 1, 9);
    test(S("abcdefghij"), 9, 1, S("abcde"), 0, 2, 9);
    test(S("abcdefghij"), 9, 1, S("abcde"), 0, 4, 9);
    test(S("abcdefghij"), 9, 1, S("abcde"), 0, 5, 9);
    test(S("abcdefghij"), 9, 1, S("abcde"), 0, 6, 9);
    test(S("abcdefghij"), 9, 1, S("abcde"), 1, 0, 1);
    test(S("abcdefghij"), 9, 1, S("abcde"), 1, 1, 8);
    test(S("abcdefghij"), 9, 1, S("abcde"), 1, 2, 8);
    test(S("abcdefghij"), 9, 1, S("abcde"), 1, 3, 8);
    test(S("abcdefghij"), 9, 1, S("abcde"), 1, 4, 8);
    test(S("abcdefghij"), 9, 1, S("abcde"), 1, 5, 8);
    test(S("abcdefghij"), 9, 1, S("abcde"), 2, 0, 1);
    test(S("abcdefghij"), 9, 1, S("abcde"), 2, 1, 7);
    test(S("abcdefghij"), 9, 1, S("abcde"), 2, 2, 7);
    test(S("abcdefghij"), 9, 1, S("abcde"), 2, 3, 7);
    test(S("abcdefghij"), 9, 1, S("abcde"), 2, 4, 7);
    test(S("abcdefghij"), 9, 1, S("abcde"), 4, 0, 1);
    test(S("abcdefghij"), 9, 1, S("abcde"), 4, 1, 5);
    test(S("abcdefghij"), 9, 1, S("abcde"), 4, 2, 5);
    test(S("abcdefghij"), 9, 1, S("abcde"), 5, 0, 1);
    test(S("abcdefghij"), 9, 1, S("abcde"), 5, 1, 1);
    test(S("abcdefghij"), 9, 1, S("abcde"), 6, 0, 0);
    test(S("abcdefghij"), 9, 1, S("abcdefghij"), 0, 0, 1);
    test(S("abcdefghij"), 9, 1, S("abcdefghij"), 0, 1, 9);
    test(S("abcdefghij"), 9, 1, S("abcdefghij"), 0, 5, 9);
    test(S("abcdefghij"), 9, 1, S("abcdefghij"), 0, 9, 9);
    test(S("abcdefghij"), 9, 1, S("abcdefghij"), 0, 10, 9);
    test(S("abcdefghij"), 9, 1, S("abcdefghij"), 0, 11, 9);
    test(S("abcdefghij"), 9, 1, S("abcdefghij"), 1, 0, 1);
    test(S("abcdefghij"), 9, 1, S("abcdefghij"), 1, 1, 8);
    test(S("abcdefghij"), 9, 1, S("abcdefghij"), 1, 4, 8);
    test(S("abcdefghij"), 9, 1, S("abcdefghij"), 1, 8, 8);
    test(S("abcdefghij"), 9, 1, S("abcdefghij"), 1, 9, 8);
    test(S("abcdefghij"), 9, 1, S("abcdefghij"), 1, 10, 8);
    test(S("abcdefghij"), 9, 1, S("abcdefghij"), 5, 0, 1);
    test(S("abcdefghij"), 9, 1, S("abcdefghij"), 5, 1, 4);
    test(S("abcdefghij"), 9, 1, S("abcdefghij"), 5, 2, 4);
    test(S("abcdefghij"), 9, 1, S("abcdefghij"), 5, 4, 4);
    test(S("abcdefghij"), 9, 1, S("abcdefghij"), 5, 5, 4);
    test(S("abcdefghij"), 9, 1, S("abcdefghij"), 5, 6, 4);
    test(S("abcdefghij"), 9, 1, S("abcdefghij"), 9, 0, 1);
    test(S("abcdefghij"), 9, 1, S("abcdefghij"), 9, 1, 0);
    test(S("abcdefghij"), 9, 1, S("abcdefghij"), 9, 2, 0);
    test(S("abcdefghij"), 9, 1, S("abcdefghij"), 10, 0, 1);
    test(S("abcdefghij"), 9, 1, S("abcdefghij"), 10, 1, 1);
    test(S("abcdefghij"), 9, 1, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghij"), 9, 1, S("abcdefghijklmnopqrst"), 0, 0, 1);
    test(S("abcdefghij"), 9, 1, S("abcdefghijklmnopqrst"), 0, 1, 9);
    test(S("abcdefghij"), 9, 1, S("abcdefghijklmnopqrst"), 0, 10, 9);
    test(S("abcdefghij"), 9, 1, S("abcdefghijklmnopqrst"), 0, 19, 9);
    test(S("abcdefghij"), 9, 1, S("abcdefghijklmnopqrst"), 0, 20, 9);
    test(S("abcdefghij"), 9, 1, S("abcdefghijklmnopqrst"), 0, 21, 9);
    test(S("abcdefghij"), 9, 1, S("abcdefghijklmnopqrst"), 1, 0, 1);
    test(S("abcdefghij"), 9, 1, S("abcdefghijklmnopqrst"), 1, 1, 8);
    test(S("abcdefghij"), 9, 1, S("abcdefghijklmnopqrst"), 1, 9, 8);
    test(S("abcdefghij"), 9, 1, S("abcdefghijklmnopqrst"), 1, 18, 8);
    test(S("abcdefghij"), 9, 1, S("abcdefghijklmnopqrst"), 1, 19, 8);
    test(S("abcdefghij"), 9, 1, S("abcdefghijklmnopqrst"), 1, 20, 8);
    test(S("abcdefghij"), 9, 1, S("abcdefghijklmnopqrst"), 10, 0, 1);
    test(S("abcdefghij"), 9, 1, S("abcdefghijklmnopqrst"), 10, 1, -1);
    test(S("abcdefghij"), 9, 1, S("abcdefghijklmnopqrst"), 10, 5, -1);
    test(S("abcdefghij"), 9, 1, S("abcdefghijklmnopqrst"), 10, 9, -1);
    test(S("abcdefghij"), 9, 1, S("abcdefghijklmnopqrst"), 10, 10, -1);
    test(S("abcdefghij"), 9, 1, S("abcdefghijklmnopqrst"), 10, 11, -1);
    test(S("abcdefghij"), 9, 1, S("abcdefghijklmnopqrst"), 19, 0, 1);
    test(S("abcdefghij"), 9, 1, S("abcdefghijklmnopqrst"), 19, 1, -10);
}

template <class S>
void test34()
{
    test(S("abcdefghij"), 9, 1, S("abcdefghijklmnopqrst"), 19, 2, -10);
    test(S("abcdefghij"), 9, 1, S("abcdefghijklmnopqrst"), 20, 0, 1);
    test(S("abcdefghij"), 9, 1, S("abcdefghijklmnopqrst"), 20, 1, 1);
    test(S("abcdefghij"), 9, 1, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghij"), 9, 2, S(""), 0, 0, 1);
    test(S("abcdefghij"), 9, 2, S(""), 0, 1, 1);
    test(S("abcdefghij"), 9, 2, S(""), 1, 0, 0);
    test(S("abcdefghij"), 9, 2, S("abcde"), 0, 0, 1);
    test(S("abcdefghij"), 9, 2, S("abcde"), 0, 1, 9);
    test(S("abcdefghij"), 9, 2, S("abcde"), 0, 2, 9);
    test(S("abcdefghij"), 9, 2, S("abcde"), 0, 4, 9);
    test(S("abcdefghij"), 9, 2, S("abcde"), 0, 5, 9);
    test(S("abcdefghij"), 9, 2, S("abcde"), 0, 6, 9);
    test(S("abcdefghij"), 9, 2, S("abcde"), 1, 0, 1);
    test(S("abcdefghij"), 9, 2, S("abcde"), 1, 1, 8);
    test(S("abcdefghij"), 9, 2, S("abcde"), 1, 2, 8);
    test(S("abcdefghij"), 9, 2, S("abcde"), 1, 3, 8);
    test(S("abcdefghij"), 9, 2, S("abcde"), 1, 4, 8);
    test(S("abcdefghij"), 9, 2, S("abcde"), 1, 5, 8);
    test(S("abcdefghij"), 9, 2, S("abcde"), 2, 0, 1);
    test(S("abcdefghij"), 9, 2, S("abcde"), 2, 1, 7);
    test(S("abcdefghij"), 9, 2, S("abcde"), 2, 2, 7);
    test(S("abcdefghij"), 9, 2, S("abcde"), 2, 3, 7);
    test(S("abcdefghij"), 9, 2, S("abcde"), 2, 4, 7);
    test(S("abcdefghij"), 9, 2, S("abcde"), 4, 0, 1);
    test(S("abcdefghij"), 9, 2, S("abcde"), 4, 1, 5);
    test(S("abcdefghij"), 9, 2, S("abcde"), 4, 2, 5);
    test(S("abcdefghij"), 9, 2, S("abcde"), 5, 0, 1);
    test(S("abcdefghij"), 9, 2, S("abcde"), 5, 1, 1);
    test(S("abcdefghij"), 9, 2, S("abcde"), 6, 0, 0);
    test(S("abcdefghij"), 9, 2, S("abcdefghij"), 0, 0, 1);
    test(S("abcdefghij"), 9, 2, S("abcdefghij"), 0, 1, 9);
    test(S("abcdefghij"), 9, 2, S("abcdefghij"), 0, 5, 9);
    test(S("abcdefghij"), 9, 2, S("abcdefghij"), 0, 9, 9);
    test(S("abcdefghij"), 9, 2, S("abcdefghij"), 0, 10, 9);
    test(S("abcdefghij"), 9, 2, S("abcdefghij"), 0, 11, 9);
    test(S("abcdefghij"), 9, 2, S("abcdefghij"), 1, 0, 1);
    test(S("abcdefghij"), 9, 2, S("abcdefghij"), 1, 1, 8);
    test(S("abcdefghij"), 9, 2, S("abcdefghij"), 1, 4, 8);
    test(S("abcdefghij"), 9, 2, S("abcdefghij"), 1, 8, 8);
    test(S("abcdefghij"), 9, 2, S("abcdefghij"), 1, 9, 8);
    test(S("abcdefghij"), 9, 2, S("abcdefghij"), 1, 10, 8);
    test(S("abcdefghij"), 9, 2, S("abcdefghij"), 5, 0, 1);
    test(S("abcdefghij"), 9, 2, S("abcdefghij"), 5, 1, 4);
    test(S("abcdefghij"), 9, 2, S("abcdefghij"), 5, 2, 4);
    test(S("abcdefghij"), 9, 2, S("abcdefghij"), 5, 4, 4);
    test(S("abcdefghij"), 9, 2, S("abcdefghij"), 5, 5, 4);
    test(S("abcdefghij"), 9, 2, S("abcdefghij"), 5, 6, 4);
    test(S("abcdefghij"), 9, 2, S("abcdefghij"), 9, 0, 1);
    test(S("abcdefghij"), 9, 2, S("abcdefghij"), 9, 1, 0);
    test(S("abcdefghij"), 9, 2, S("abcdefghij"), 9, 2, 0);
    test(S("abcdefghij"), 9, 2, S("abcdefghij"), 10, 0, 1);
    test(S("abcdefghij"), 9, 2, S("abcdefghij"), 10, 1, 1);
    test(S("abcdefghij"), 9, 2, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghij"), 9, 2, S("abcdefghijklmnopqrst"), 0, 0, 1);
    test(S("abcdefghij"), 9, 2, S("abcdefghijklmnopqrst"), 0, 1, 9);
    test(S("abcdefghij"), 9, 2, S("abcdefghijklmnopqrst"), 0, 10, 9);
    test(S("abcdefghij"), 9, 2, S("abcdefghijklmnopqrst"), 0, 19, 9);
    test(S("abcdefghij"), 9, 2, S("abcdefghijklmnopqrst"), 0, 20, 9);
    test(S("abcdefghij"), 9, 2, S("abcdefghijklmnopqrst"), 0, 21, 9);
    test(S("abcdefghij"), 9, 2, S("abcdefghijklmnopqrst"), 1, 0, 1);
    test(S("abcdefghij"), 9, 2, S("abcdefghijklmnopqrst"), 1, 1, 8);
    test(S("abcdefghij"), 9, 2, S("abcdefghijklmnopqrst"), 1, 9, 8);
    test(S("abcdefghij"), 9, 2, S("abcdefghijklmnopqrst"), 1, 18, 8);
    test(S("abcdefghij"), 9, 2, S("abcdefghijklmnopqrst"), 1, 19, 8);
    test(S("abcdefghij"), 9, 2, S("abcdefghijklmnopqrst"), 1, 20, 8);
    test(S("abcdefghij"), 9, 2, S("abcdefghijklmnopqrst"), 10, 0, 1);
    test(S("abcdefghij"), 9, 2, S("abcdefghijklmnopqrst"), 10, 1, -1);
    test(S("abcdefghij"), 9, 2, S("abcdefghijklmnopqrst"), 10, 5, -1);
    test(S("abcdefghij"), 9, 2, S("abcdefghijklmnopqrst"), 10, 9, -1);
    test(S("abcdefghij"), 9, 2, S("abcdefghijklmnopqrst"), 10, 10, -1);
    test(S("abcdefghij"), 9, 2, S("abcdefghijklmnopqrst"), 10, 11, -1);
    test(S("abcdefghij"), 9, 2, S("abcdefghijklmnopqrst"), 19, 0, 1);
    test(S("abcdefghij"), 9, 2, S("abcdefghijklmnopqrst"), 19, 1, -10);
    test(S("abcdefghij"), 9, 2, S("abcdefghijklmnopqrst"), 19, 2, -10);
    test(S("abcdefghij"), 9, 2, S("abcdefghijklmnopqrst"), 20, 0, 1);
    test(S("abcdefghij"), 9, 2, S("abcdefghijklmnopqrst"), 20, 1, 1);
    test(S("abcdefghij"), 9, 2, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghij"), 10, 0, S(""), 0, 0, 0);
    test(S("abcdefghij"), 10, 0, S(""), 0, 1, 0);
    test(S("abcdefghij"), 10, 0, S(""), 1, 0, 0);
    test(S("abcdefghij"), 10, 0, S("abcde"), 0, 0, 0);
    test(S("abcdefghij"), 10, 0, S("abcde"), 0, 1, -1);
    test(S("abcdefghij"), 10, 0, S("abcde"), 0, 2, -2);
    test(S("abcdefghij"), 10, 0, S("abcde"), 0, 4, -4);
    test(S("abcdefghij"), 10, 0, S("abcde"), 0, 5, -5);
    test(S("abcdefghij"), 10, 0, S("abcde"), 0, 6, -5);
    test(S("abcdefghij"), 10, 0, S("abcde"), 1, 0, 0);
    test(S("abcdefghij"), 10, 0, S("abcde"), 1, 1, -1);
    test(S("abcdefghij"), 10, 0, S("abcde"), 1, 2, -2);
    test(S("abcdefghij"), 10, 0, S("abcde"), 1, 3, -3);
    test(S("abcdefghij"), 10, 0, S("abcde"), 1, 4, -4);
    test(S("abcdefghij"), 10, 0, S("abcde"), 1, 5, -4);
    test(S("abcdefghij"), 10, 0, S("abcde"), 2, 0, 0);
    test(S("abcdefghij"), 10, 0, S("abcde"), 2, 1, -1);
    test(S("abcdefghij"), 10, 0, S("abcde"), 2, 2, -2);
    test(S("abcdefghij"), 10, 0, S("abcde"), 2, 3, -3);
    test(S("abcdefghij"), 10, 0, S("abcde"), 2, 4, -3);
    test(S("abcdefghij"), 10, 0, S("abcde"), 4, 0, 0);
    test(S("abcdefghij"), 10, 0, S("abcde"), 4, 1, -1);
}

template <class S>
void test35()
{
    test(S("abcdefghij"), 10, 0, S("abcde"), 4, 2, -1);
    test(S("abcdefghij"), 10, 0, S("abcde"), 5, 0, 0);
    test(S("abcdefghij"), 10, 0, S("abcde"), 5, 1, 0);
    test(S("abcdefghij"), 10, 0, S("abcde"), 6, 0, 0);
    test(S("abcdefghij"), 10, 0, S("abcdefghij"), 0, 0, 0);
    test(S("abcdefghij"), 10, 0, S("abcdefghij"), 0, 1, -1);
    test(S("abcdefghij"), 10, 0, S("abcdefghij"), 0, 5, -5);
    test(S("abcdefghij"), 10, 0, S("abcdefghij"), 0, 9, -9);
    test(S("abcdefghij"), 10, 0, S("abcdefghij"), 0, 10, -10);
    test(S("abcdefghij"), 10, 0, S("abcdefghij"), 0, 11, -10);
    test(S("abcdefghij"), 10, 0, S("abcdefghij"), 1, 0, 0);
    test(S("abcdefghij"), 10, 0, S("abcdefghij"), 1, 1, -1);
    test(S("abcdefghij"), 10, 0, S("abcdefghij"), 1, 4, -4);
    test(S("abcdefghij"), 10, 0, S("abcdefghij"), 1, 8, -8);
    test(S("abcdefghij"), 10, 0, S("abcdefghij"), 1, 9, -9);
    test(S("abcdefghij"), 10, 0, S("abcdefghij"), 1, 10, -9);
    test(S("abcdefghij"), 10, 0, S("abcdefghij"), 5, 0, 0);
    test(S("abcdefghij"), 10, 0, S("abcdefghij"), 5, 1, -1);
    test(S("abcdefghij"), 10, 0, S("abcdefghij"), 5, 2, -2);
    test(S("abcdefghij"), 10, 0, S("abcdefghij"), 5, 4, -4);
    test(S("abcdefghij"), 10, 0, S("abcdefghij"), 5, 5, -5);
    test(S("abcdefghij"), 10, 0, S("abcdefghij"), 5, 6, -5);
    test(S("abcdefghij"), 10, 0, S("abcdefghij"), 9, 0, 0);
    test(S("abcdefghij"), 10, 0, S("abcdefghij"), 9, 1, -1);
    test(S("abcdefghij"), 10, 0, S("abcdefghij"), 9, 2, -1);
    test(S("abcdefghij"), 10, 0, S("abcdefghij"), 10, 0, 0);
    test(S("abcdefghij"), 10, 0, S("abcdefghij"), 10, 1, 0);
    test(S("abcdefghij"), 10, 0, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghij"), 10, 0, S("abcdefghijklmnopqrst"), 0, 0, 0);
    test(S("abcdefghij"), 10, 0, S("abcdefghijklmnopqrst"), 0, 1, -1);
    test(S("abcdefghij"), 10, 0, S("abcdefghijklmnopqrst"), 0, 10, -10);
    test(S("abcdefghij"), 10, 0, S("abcdefghijklmnopqrst"), 0, 19, -19);
    test(S("abcdefghij"), 10, 0, S("abcdefghijklmnopqrst"), 0, 20, -20);
    test(S("abcdefghij"), 10, 0, S("abcdefghijklmnopqrst"), 0, 21, -20);
    test(S("abcdefghij"), 10, 0, S("abcdefghijklmnopqrst"), 1, 0, 0);
    test(S("abcdefghij"), 10, 0, S("abcdefghijklmnopqrst"), 1, 1, -1);
    test(S("abcdefghij"), 10, 0, S("abcdefghijklmnopqrst"), 1, 9, -9);
    test(S("abcdefghij"), 10, 0, S("abcdefghijklmnopqrst"), 1, 18, -18);
    test(S("abcdefghij"), 10, 0, S("abcdefghijklmnopqrst"), 1, 19, -19);
    test(S("abcdefghij"), 10, 0, S("abcdefghijklmnopqrst"), 1, 20, -19);
    test(S("abcdefghij"), 10, 0, S("abcdefghijklmnopqrst"), 10, 0, 0);
    test(S("abcdefghij"), 10, 0, S("abcdefghijklmnopqrst"), 10, 1, -1);
    test(S("abcdefghij"), 10, 0, S("abcdefghijklmnopqrst"), 10, 5, -5);
    test(S("abcdefghij"), 10, 0, S("abcdefghijklmnopqrst"), 10, 9, -9);
    test(S("abcdefghij"), 10, 0, S("abcdefghijklmnopqrst"), 10, 10, -10);
    test(S("abcdefghij"), 10, 0, S("abcdefghijklmnopqrst"), 10, 11, -10);
    test(S("abcdefghij"), 10, 0, S("abcdefghijklmnopqrst"), 19, 0, 0);
    test(S("abcdefghij"), 10, 0, S("abcdefghijklmnopqrst"), 19, 1, -1);
    test(S("abcdefghij"), 10, 0, S("abcdefghijklmnopqrst"), 19, 2, -1);
    test(S("abcdefghij"), 10, 0, S("abcdefghijklmnopqrst"), 20, 0, 0);
    test(S("abcdefghij"), 10, 0, S("abcdefghijklmnopqrst"), 20, 1, 0);
    test(S("abcdefghij"), 10, 0, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghij"), 10, 1, S(""), 0, 0, 0);
    test(S("abcdefghij"), 10, 1, S(""), 0, 1, 0);
    test(S("abcdefghij"), 10, 1, S(""), 1, 0, 0);
    test(S("abcdefghij"), 10, 1, S("abcde"), 0, 0, 0);
    test(S("abcdefghij"), 10, 1, S("abcde"), 0, 1, -1);
    test(S("abcdefghij"), 10, 1, S("abcde"), 0, 2, -2);
    test(S("abcdefghij"), 10, 1, S("abcde"), 0, 4, -4);
    test(S("abcdefghij"), 10, 1, S("abcde"), 0, 5, -5);
    test(S("abcdefghij"), 10, 1, S("abcde"), 0, 6, -5);
    test(S("abcdefghij"), 10, 1, S("abcde"), 1, 0, 0);
    test(S("abcdefghij"), 10, 1, S("abcde"), 1, 1, -1);
    test(S("abcdefghij"), 10, 1, S("abcde"), 1, 2, -2);
    test(S("abcdefghij"), 10, 1, S("abcde"), 1, 3, -3);
    test(S("abcdefghij"), 10, 1, S("abcde"), 1, 4, -4);
    test(S("abcdefghij"), 10, 1, S("abcde"), 1, 5, -4);
    test(S("abcdefghij"), 10, 1, S("abcde"), 2, 0, 0);
    test(S("abcdefghij"), 10, 1, S("abcde"), 2, 1, -1);
    test(S("abcdefghij"), 10, 1, S("abcde"), 2, 2, -2);
    test(S("abcdefghij"), 10, 1, S("abcde"), 2, 3, -3);
    test(S("abcdefghij"), 10, 1, S("abcde"), 2, 4, -3);
    test(S("abcdefghij"), 10, 1, S("abcde"), 4, 0, 0);
    test(S("abcdefghij"), 10, 1, S("abcde"), 4, 1, -1);
    test(S("abcdefghij"), 10, 1, S("abcde"), 4, 2, -1);
    test(S("abcdefghij"), 10, 1, S("abcde"), 5, 0, 0);
    test(S("abcdefghij"), 10, 1, S("abcde"), 5, 1, 0);
    test(S("abcdefghij"), 10, 1, S("abcde"), 6, 0, 0);
    test(S("abcdefghij"), 10, 1, S("abcdefghij"), 0, 0, 0);
    test(S("abcdefghij"), 10, 1, S("abcdefghij"), 0, 1, -1);
    test(S("abcdefghij"), 10, 1, S("abcdefghij"), 0, 5, -5);
    test(S("abcdefghij"), 10, 1, S("abcdefghij"), 0, 9, -9);
    test(S("abcdefghij"), 10, 1, S("abcdefghij"), 0, 10, -10);
    test(S("abcdefghij"), 10, 1, S("abcdefghij"), 0, 11, -10);
    test(S("abcdefghij"), 10, 1, S("abcdefghij"), 1, 0, 0);
    test(S("abcdefghij"), 10, 1, S("abcdefghij"), 1, 1, -1);
    test(S("abcdefghij"), 10, 1, S("abcdefghij"), 1, 4, -4);
    test(S("abcdefghij"), 10, 1, S("abcdefghij"), 1, 8, -8);
    test(S("abcdefghij"), 10, 1, S("abcdefghij"), 1, 9, -9);
    test(S("abcdefghij"), 10, 1, S("abcdefghij"), 1, 10, -9);
    test(S("abcdefghij"), 10, 1, S("abcdefghij"), 5, 0, 0);
    test(S("abcdefghij"), 10, 1, S("abcdefghij"), 5, 1, -1);
    test(S("abcdefghij"), 10, 1, S("abcdefghij"), 5, 2, -2);
    test(S("abcdefghij"), 10, 1, S("abcdefghij"), 5, 4, -4);
    test(S("abcdefghij"), 10, 1, S("abcdefghij"), 5, 5, -5);
    test(S("abcdefghij"), 10, 1, S("abcdefghij"), 5, 6, -5);
    test(S("abcdefghij"), 10, 1, S("abcdefghij"), 9, 0, 0);
    test(S("abcdefghij"), 10, 1, S("abcdefghij"), 9, 1, -1);
    test(S("abcdefghij"), 10, 1, S("abcdefghij"), 9, 2, -1);
    test(S("abcdefghij"), 10, 1, S("abcdefghij"), 10, 0, 0);
}

template <class S>
void test36()
{
    test(S("abcdefghij"), 10, 1, S("abcdefghij"), 10, 1, 0);
    test(S("abcdefghij"), 10, 1, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghij"), 10, 1, S("abcdefghijklmnopqrst"), 0, 0, 0);
    test(S("abcdefghij"), 10, 1, S("abcdefghijklmnopqrst"), 0, 1, -1);
    test(S("abcdefghij"), 10, 1, S("abcdefghijklmnopqrst"), 0, 10, -10);
    test(S("abcdefghij"), 10, 1, S("abcdefghijklmnopqrst"), 0, 19, -19);
    test(S("abcdefghij"), 10, 1, S("abcdefghijklmnopqrst"), 0, 20, -20);
    test(S("abcdefghij"), 10, 1, S("abcdefghijklmnopqrst"), 0, 21, -20);
    test(S("abcdefghij"), 10, 1, S("abcdefghijklmnopqrst"), 1, 0, 0);
    test(S("abcdefghij"), 10, 1, S("abcdefghijklmnopqrst"), 1, 1, -1);
    test(S("abcdefghij"), 10, 1, S("abcdefghijklmnopqrst"), 1, 9, -9);
    test(S("abcdefghij"), 10, 1, S("abcdefghijklmnopqrst"), 1, 18, -18);
    test(S("abcdefghij"), 10, 1, S("abcdefghijklmnopqrst"), 1, 19, -19);
    test(S("abcdefghij"), 10, 1, S("abcdefghijklmnopqrst"), 1, 20, -19);
    test(S("abcdefghij"), 10, 1, S("abcdefghijklmnopqrst"), 10, 0, 0);
    test(S("abcdefghij"), 10, 1, S("abcdefghijklmnopqrst"), 10, 1, -1);
    test(S("abcdefghij"), 10, 1, S("abcdefghijklmnopqrst"), 10, 5, -5);
    test(S("abcdefghij"), 10, 1, S("abcdefghijklmnopqrst"), 10, 9, -9);
    test(S("abcdefghij"), 10, 1, S("abcdefghijklmnopqrst"), 10, 10, -10);
    test(S("abcdefghij"), 10, 1, S("abcdefghijklmnopqrst"), 10, 11, -10);
    test(S("abcdefghij"), 10, 1, S("abcdefghijklmnopqrst"), 19, 0, 0);
    test(S("abcdefghij"), 10, 1, S("abcdefghijklmnopqrst"), 19, 1, -1);
    test(S("abcdefghij"), 10, 1, S("abcdefghijklmnopqrst"), 19, 2, -1);
    test(S("abcdefghij"), 10, 1, S("abcdefghijklmnopqrst"), 20, 0, 0);
    test(S("abcdefghij"), 10, 1, S("abcdefghijklmnopqrst"), 20, 1, 0);
    test(S("abcdefghij"), 10, 1, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghij"), 11, 0, S(""), 0, 0, 0);
    test(S("abcdefghij"), 11, 0, S(""), 0, 1, 0);
    test(S("abcdefghij"), 11, 0, S(""), 1, 0, 0);
    test(S("abcdefghij"), 11, 0, S("abcde"), 0, 0, 0);
    test(S("abcdefghij"), 11, 0, S("abcde"), 0, 1, 0);
    test(S("abcdefghij"), 11, 0, S("abcde"), 0, 2, 0);
    test(S("abcdefghij"), 11, 0, S("abcde"), 0, 4, 0);
    test(S("abcdefghij"), 11, 0, S("abcde"), 0, 5, 0);
    test(S("abcdefghij"), 11, 0, S("abcde"), 0, 6, 0);
    test(S("abcdefghij"), 11, 0, S("abcde"), 1, 0, 0);
    test(S("abcdefghij"), 11, 0, S("abcde"), 1, 1, 0);
    test(S("abcdefghij"), 11, 0, S("abcde"), 1, 2, 0);
    test(S("abcdefghij"), 11, 0, S("abcde"), 1, 3, 0);
    test(S("abcdefghij"), 11, 0, S("abcde"), 1, 4, 0);
    test(S("abcdefghij"), 11, 0, S("abcde"), 1, 5, 0);
    test(S("abcdefghij"), 11, 0, S("abcde"), 2, 0, 0);
    test(S("abcdefghij"), 11, 0, S("abcde"), 2, 1, 0);
    test(S("abcdefghij"), 11, 0, S("abcde"), 2, 2, 0);
    test(S("abcdefghij"), 11, 0, S("abcde"), 2, 3, 0);
    test(S("abcdefghij"), 11, 0, S("abcde"), 2, 4, 0);
    test(S("abcdefghij"), 11, 0, S("abcde"), 4, 0, 0);
    test(S("abcdefghij"), 11, 0, S("abcde"), 4, 1, 0);
    test(S("abcdefghij"), 11, 0, S("abcde"), 4, 2, 0);
    test(S("abcdefghij"), 11, 0, S("abcde"), 5, 0, 0);
    test(S("abcdefghij"), 11, 0, S("abcde"), 5, 1, 0);
    test(S("abcdefghij"), 11, 0, S("abcde"), 6, 0, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghij"), 0, 0, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghij"), 0, 1, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghij"), 0, 5, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghij"), 0, 9, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghij"), 0, 10, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghij"), 0, 11, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghij"), 1, 0, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghij"), 1, 1, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghij"), 1, 4, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghij"), 1, 8, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghij"), 1, 9, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghij"), 1, 10, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghij"), 5, 0, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghij"), 5, 1, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghij"), 5, 2, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghij"), 5, 4, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghij"), 5, 5, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghij"), 5, 6, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghij"), 9, 0, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghij"), 9, 1, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghij"), 9, 2, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghij"), 10, 0, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghij"), 10, 1, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghijklmnopqrst"), 0, 0, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghijklmnopqrst"), 0, 1, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghijklmnopqrst"), 0, 10, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghijklmnopqrst"), 0, 19, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghijklmnopqrst"), 0, 20, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghijklmnopqrst"), 0, 21, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghijklmnopqrst"), 1, 0, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghijklmnopqrst"), 1, 1, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghijklmnopqrst"), 1, 9, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghijklmnopqrst"), 1, 18, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghijklmnopqrst"), 1, 19, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghijklmnopqrst"), 1, 20, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghijklmnopqrst"), 10, 0, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghijklmnopqrst"), 10, 1, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghijklmnopqrst"), 10, 5, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghijklmnopqrst"), 10, 9, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghijklmnopqrst"), 10, 10, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghijklmnopqrst"), 10, 11, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghijklmnopqrst"), 19, 0, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghijklmnopqrst"), 19, 1, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghijklmnopqrst"), 19, 2, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghijklmnopqrst"), 20, 0, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghijklmnopqrst"), 20, 1, 0);
    test(S("abcdefghij"), 11, 0, S("abcdefghijklmnopqrst"), 21, 0, 0);
}

template <class S>
void test37()
{
    test(S("abcdefghijklmnopqrst"), 0, 0, S(""), 0, 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 0, S(""), 0, 1, 0);
    test(S("abcdefghijklmnopqrst"), 0, 0, S(""), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcde"), 0, 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcde"), 0, 1, -1);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcde"), 0, 2, -2);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcde"), 0, 4, -4);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcde"), 0, 5, -5);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcde"), 0, 6, -5);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcde"), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcde"), 1, 1, -1);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcde"), 1, 2, -2);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcde"), 1, 3, -3);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcde"), 1, 4, -4);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcde"), 1, 5, -4);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcde"), 2, 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcde"), 2, 1, -1);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcde"), 2, 2, -2);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcde"), 2, 3, -3);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcde"), 2, 4, -3);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcde"), 4, 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcde"), 4, 1, -1);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcde"), 4, 2, -1);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcde"), 5, 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcde"), 5, 1, 0);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcde"), 6, 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghij"), 0, 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghij"), 0, 1, -1);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghij"), 0, 5, -5);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghij"), 0, 9, -9);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghij"), 0, 10, -10);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghij"), 0, 11, -10);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghij"), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghij"), 1, 1, -1);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghij"), 1, 4, -4);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghij"), 1, 8, -8);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghij"), 1, 9, -9);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghij"), 1, 10, -9);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghij"), 5, 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghij"), 5, 1, -1);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghij"), 5, 2, -2);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghij"), 5, 4, -4);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghij"), 5, 5, -5);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghij"), 5, 6, -5);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghij"), 9, 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghij"), 9, 1, -1);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghij"), 9, 2, -1);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghij"), 10, 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghij"), 10, 1, 0);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghijklmnopqrst"), 0, 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghijklmnopqrst"), 0, 1, -1);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghijklmnopqrst"), 0, 10, -10);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghijklmnopqrst"), 0, 19, -19);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghijklmnopqrst"), 0, 20, -20);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghijklmnopqrst"), 0, 21, -20);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghijklmnopqrst"), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghijklmnopqrst"), 1, 1, -1);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghijklmnopqrst"), 1, 9, -9);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghijklmnopqrst"), 1, 18, -18);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghijklmnopqrst"), 1, 19, -19);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghijklmnopqrst"), 1, 20, -19);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghijklmnopqrst"), 10, 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghijklmnopqrst"), 10, 1, -1);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghijklmnopqrst"), 10, 5, -5);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghijklmnopqrst"), 10, 9, -9);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghijklmnopqrst"), 10, 10, -10);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghijklmnopqrst"), 10, 11, -10);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghijklmnopqrst"), 19, 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghijklmnopqrst"), 19, 1, -1);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghijklmnopqrst"), 19, 2, -1);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghijklmnopqrst"), 20, 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghijklmnopqrst"), 20, 1, 0);
    test(S("abcdefghijklmnopqrst"), 0, 0, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 1, S(""), 0, 0, 1);
    test(S("abcdefghijklmnopqrst"), 0, 1, S(""), 0, 1, 1);
    test(S("abcdefghijklmnopqrst"), 0, 1, S(""), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcde"), 0, 0, 1);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcde"), 0, 1, 0);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcde"), 0, 2, -1);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcde"), 0, 4, -3);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcde"), 0, 5, -4);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcde"), 0, 6, -4);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcde"), 1, 0, 1);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcde"), 1, 1, -1);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcde"), 1, 2, -1);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcde"), 1, 3, -1);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcde"), 1, 4, -1);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcde"), 1, 5, -1);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcde"), 2, 0, 1);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcde"), 2, 1, -2);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcde"), 2, 2, -2);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcde"), 2, 3, -2);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcde"), 2, 4, -2);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcde"), 4, 0, 1);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcde"), 4, 1, -4);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcde"), 4, 2, -4);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcde"), 5, 0, 1);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcde"), 5, 1, 1);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcde"), 6, 0, 0);
}

template <class S>
void test38()
{
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghij"), 0, 0, 1);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghij"), 0, 1, 0);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghij"), 0, 5, -4);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghij"), 0, 9, -8);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghij"), 0, 10, -9);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghij"), 0, 11, -9);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghij"), 1, 0, 1);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghij"), 1, 1, -1);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghij"), 1, 4, -1);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghij"), 1, 8, -1);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghij"), 1, 9, -1);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghij"), 1, 10, -1);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghij"), 5, 0, 1);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghij"), 5, 1, -5);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghij"), 5, 2, -5);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghij"), 5, 4, -5);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghij"), 5, 5, -5);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghij"), 5, 6, -5);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghij"), 9, 0, 1);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghij"), 9, 1, -9);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghij"), 9, 2, -9);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghij"), 10, 0, 1);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghij"), 10, 1, 1);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghijklmnopqrst"), 0, 0, 1);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghijklmnopqrst"), 0, 1, 0);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghijklmnopqrst"), 0, 10, -9);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghijklmnopqrst"), 0, 19, -18);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghijklmnopqrst"), 0, 20, -19);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghijklmnopqrst"), 0, 21, -19);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghijklmnopqrst"), 1, 0, 1);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghijklmnopqrst"), 1, 1, -1);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghijklmnopqrst"), 1, 9, -1);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghijklmnopqrst"), 1, 18, -1);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghijklmnopqrst"), 1, 19, -1);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghijklmnopqrst"), 1, 20, -1);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghijklmnopqrst"), 10, 0, 1);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghijklmnopqrst"), 10, 1, -10);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghijklmnopqrst"), 10, 5, -10);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghijklmnopqrst"), 10, 9, -10);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghijklmnopqrst"), 10, 10, -10);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghijklmnopqrst"), 10, 11, -10);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghijklmnopqrst"), 19, 0, 1);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghijklmnopqrst"), 19, 1, -19);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghijklmnopqrst"), 19, 2, -19);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghijklmnopqrst"), 20, 0, 1);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghijklmnopqrst"), 20, 1, 1);
    test(S("abcdefghijklmnopqrst"), 0, 1, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 10, S(""), 0, 0, 10);
    test(S("abcdefghijklmnopqrst"), 0, 10, S(""), 0, 1, 10);
    test(S("abcdefghijklmnopqrst"), 0, 10, S(""), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcde"), 0, 0, 10);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcde"), 0, 1, 9);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcde"), 0, 2, 8);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcde"), 0, 4, 6);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcde"), 0, 5, 5);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcde"), 0, 6, 5);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcde"), 1, 0, 10);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcde"), 1, 1, -1);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcde"), 1, 2, -1);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcde"), 1, 3, -1);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcde"), 1, 4, -1);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcde"), 1, 5, -1);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcde"), 2, 0, 10);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcde"), 2, 1, -2);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcde"), 2, 2, -2);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcde"), 2, 3, -2);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcde"), 2, 4, -2);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcde"), 4, 0, 10);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcde"), 4, 1, -4);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcde"), 4, 2, -4);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcde"), 5, 0, 10);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcde"), 5, 1, 10);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcde"), 6, 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghij"), 0, 0, 10);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghij"), 0, 1, 9);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghij"), 0, 5, 5);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghij"), 0, 9, 1);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghij"), 0, 10, 0);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghij"), 0, 11, 0);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghij"), 1, 0, 10);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghij"), 1, 1, -1);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghij"), 1, 4, -1);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghij"), 1, 8, -1);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghij"), 1, 9, -1);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghij"), 1, 10, -1);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghij"), 5, 0, 10);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghij"), 5, 1, -5);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghij"), 5, 2, -5);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghij"), 5, 4, -5);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghij"), 5, 5, -5);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghij"), 5, 6, -5);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghij"), 9, 0, 10);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghij"), 9, 1, -9);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghij"), 9, 2, -9);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghij"), 10, 0, 10);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghij"), 10, 1, 10);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghijklmnopqrst"), 0, 0, 10);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghijklmnopqrst"), 0, 1, 9);
}

template <class S>
void test39()
{
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghijklmnopqrst"), 0, 10, 0);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghijklmnopqrst"), 0, 19, -9);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghijklmnopqrst"), 0, 20, -10);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghijklmnopqrst"), 0, 21, -10);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghijklmnopqrst"), 1, 0, 10);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghijklmnopqrst"), 1, 1, -1);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghijklmnopqrst"), 1, 9, -1);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghijklmnopqrst"), 1, 18, -1);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghijklmnopqrst"), 1, 19, -1);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghijklmnopqrst"), 1, 20, -1);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghijklmnopqrst"), 10, 0, 10);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghijklmnopqrst"), 10, 1, -10);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghijklmnopqrst"), 10, 5, -10);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghijklmnopqrst"), 10, 9, -10);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghijklmnopqrst"), 10, 10, -10);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghijklmnopqrst"), 10, 11, -10);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghijklmnopqrst"), 19, 0, 10);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghijklmnopqrst"), 19, 1, -19);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghijklmnopqrst"), 19, 2, -19);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghijklmnopqrst"), 20, 0, 10);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghijklmnopqrst"), 20, 1, 10);
    test(S("abcdefghijklmnopqrst"), 0, 10, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 19, S(""), 0, 0, 19);
    test(S("abcdefghijklmnopqrst"), 0, 19, S(""), 0, 1, 19);
    test(S("abcdefghijklmnopqrst"), 0, 19, S(""), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcde"), 0, 0, 19);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcde"), 0, 1, 18);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcde"), 0, 2, 17);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcde"), 0, 4, 15);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcde"), 0, 5, 14);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcde"), 0, 6, 14);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcde"), 1, 0, 19);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcde"), 1, 1, -1);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcde"), 1, 2, -1);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcde"), 1, 3, -1);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcde"), 1, 4, -1);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcde"), 1, 5, -1);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcde"), 2, 0, 19);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcde"), 2, 1, -2);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcde"), 2, 2, -2);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcde"), 2, 3, -2);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcde"), 2, 4, -2);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcde"), 4, 0, 19);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcde"), 4, 1, -4);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcde"), 4, 2, -4);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcde"), 5, 0, 19);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcde"), 5, 1, 19);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcde"), 6, 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghij"), 0, 0, 19);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghij"), 0, 1, 18);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghij"), 0, 5, 14);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghij"), 0, 9, 10);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghij"), 0, 10, 9);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghij"), 0, 11, 9);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghij"), 1, 0, 19);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghij"), 1, 1, -1);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghij"), 1, 4, -1);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghij"), 1, 8, -1);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghij"), 1, 9, -1);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghij"), 1, 10, -1);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghij"), 5, 0, 19);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghij"), 5, 1, -5);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghij"), 5, 2, -5);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghij"), 5, 4, -5);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghij"), 5, 5, -5);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghij"), 5, 6, -5);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghij"), 9, 0, 19);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghij"), 9, 1, -9);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghij"), 9, 2, -9);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghij"), 10, 0, 19);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghij"), 10, 1, 19);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghijklmnopqrst"), 0, 0, 19);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghijklmnopqrst"), 0, 1, 18);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghijklmnopqrst"), 0, 10, 9);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghijklmnopqrst"), 0, 19, 0);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghijklmnopqrst"), 0, 20, -1);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghijklmnopqrst"), 0, 21, -1);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghijklmnopqrst"), 1, 0, 19);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghijklmnopqrst"), 1, 1, -1);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghijklmnopqrst"), 1, 9, -1);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghijklmnopqrst"), 1, 18, -1);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghijklmnopqrst"), 1, 19, -1);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghijklmnopqrst"), 1, 20, -1);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghijklmnopqrst"), 10, 0, 19);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghijklmnopqrst"), 10, 1, -10);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghijklmnopqrst"), 10, 5, -10);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghijklmnopqrst"), 10, 9, -10);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghijklmnopqrst"), 10, 10, -10);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghijklmnopqrst"), 10, 11, -10);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghijklmnopqrst"), 19, 0, 19);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghijklmnopqrst"), 19, 1, -19);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghijklmnopqrst"), 19, 2, -19);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghijklmnopqrst"), 20, 0, 19);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghijklmnopqrst"), 20, 1, 19);
    test(S("abcdefghijklmnopqrst"), 0, 19, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 20, S(""), 0, 0, 20);
    test(S("abcdefghijklmnopqrst"), 0, 20, S(""), 0, 1, 20);
    test(S("abcdefghijklmnopqrst"), 0, 20, S(""), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcde"), 0, 0, 20);
}

template <class S>
void test40()
{
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcde"), 0, 1, 19);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcde"), 0, 2, 18);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcde"), 0, 4, 16);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcde"), 0, 5, 15);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcde"), 0, 6, 15);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcde"), 1, 0, 20);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcde"), 1, 1, -1);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcde"), 1, 2, -1);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcde"), 1, 3, -1);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcde"), 1, 4, -1);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcde"), 1, 5, -1);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcde"), 2, 0, 20);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcde"), 2, 1, -2);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcde"), 2, 2, -2);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcde"), 2, 3, -2);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcde"), 2, 4, -2);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcde"), 4, 0, 20);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcde"), 4, 1, -4);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcde"), 4, 2, -4);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcde"), 5, 0, 20);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcde"), 5, 1, 20);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcde"), 6, 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghij"), 0, 0, 20);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghij"), 0, 1, 19);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghij"), 0, 5, 15);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghij"), 0, 9, 11);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghij"), 0, 10, 10);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghij"), 0, 11, 10);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghij"), 1, 0, 20);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghij"), 1, 1, -1);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghij"), 1, 4, -1);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghij"), 1, 8, -1);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghij"), 1, 9, -1);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghij"), 1, 10, -1);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghij"), 5, 0, 20);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghij"), 5, 1, -5);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghij"), 5, 2, -5);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghij"), 5, 4, -5);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghij"), 5, 5, -5);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghij"), 5, 6, -5);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghij"), 9, 0, 20);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghij"), 9, 1, -9);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghij"), 9, 2, -9);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghij"), 10, 0, 20);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghij"), 10, 1, 20);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghijklmnopqrst"), 0, 0, 20);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghijklmnopqrst"), 0, 1, 19);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghijklmnopqrst"), 0, 10, 10);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghijklmnopqrst"), 0, 19, 1);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghijklmnopqrst"), 0, 20, 0);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghijklmnopqrst"), 0, 21, 0);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghijklmnopqrst"), 1, 0, 20);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghijklmnopqrst"), 1, 1, -1);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghijklmnopqrst"), 1, 9, -1);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghijklmnopqrst"), 1, 18, -1);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghijklmnopqrst"), 1, 19, -1);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghijklmnopqrst"), 1, 20, -1);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghijklmnopqrst"), 10, 0, 20);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghijklmnopqrst"), 10, 1, -10);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghijklmnopqrst"), 10, 5, -10);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghijklmnopqrst"), 10, 9, -10);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghijklmnopqrst"), 10, 10, -10);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghijklmnopqrst"), 10, 11, -10);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghijklmnopqrst"), 19, 0, 20);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghijklmnopqrst"), 19, 1, -19);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghijklmnopqrst"), 19, 2, -19);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghijklmnopqrst"), 20, 0, 20);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghijklmnopqrst"), 20, 1, 20);
    test(S("abcdefghijklmnopqrst"), 0, 20, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 21, S(""), 0, 0, 20);
    test(S("abcdefghijklmnopqrst"), 0, 21, S(""), 0, 1, 20);
    test(S("abcdefghijklmnopqrst"), 0, 21, S(""), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcde"), 0, 0, 20);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcde"), 0, 1, 19);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcde"), 0, 2, 18);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcde"), 0, 4, 16);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcde"), 0, 5, 15);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcde"), 0, 6, 15);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcde"), 1, 0, 20);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcde"), 1, 1, -1);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcde"), 1, 2, -1);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcde"), 1, 3, -1);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcde"), 1, 4, -1);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcde"), 1, 5, -1);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcde"), 2, 0, 20);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcde"), 2, 1, -2);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcde"), 2, 2, -2);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcde"), 2, 3, -2);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcde"), 2, 4, -2);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcde"), 4, 0, 20);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcde"), 4, 1, -4);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcde"), 4, 2, -4);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcde"), 5, 0, 20);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcde"), 5, 1, 20);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcde"), 6, 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghij"), 0, 0, 20);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghij"), 0, 1, 19);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghij"), 0, 5, 15);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghij"), 0, 9, 11);
}

template <class S>
void test41()
{
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghij"), 0, 10, 10);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghij"), 0, 11, 10);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghij"), 1, 0, 20);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghij"), 1, 1, -1);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghij"), 1, 4, -1);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghij"), 1, 8, -1);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghij"), 1, 9, -1);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghij"), 1, 10, -1);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghij"), 5, 0, 20);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghij"), 5, 1, -5);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghij"), 5, 2, -5);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghij"), 5, 4, -5);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghij"), 5, 5, -5);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghij"), 5, 6, -5);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghij"), 9, 0, 20);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghij"), 9, 1, -9);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghij"), 9, 2, -9);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghij"), 10, 0, 20);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghij"), 10, 1, 20);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghijklmnopqrst"), 0, 0, 20);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghijklmnopqrst"), 0, 1, 19);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghijklmnopqrst"), 0, 10, 10);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghijklmnopqrst"), 0, 19, 1);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghijklmnopqrst"), 0, 20, 0);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghijklmnopqrst"), 0, 21, 0);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghijklmnopqrst"), 1, 0, 20);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghijklmnopqrst"), 1, 1, -1);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghijklmnopqrst"), 1, 9, -1);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghijklmnopqrst"), 1, 18, -1);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghijklmnopqrst"), 1, 19, -1);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghijklmnopqrst"), 1, 20, -1);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghijklmnopqrst"), 10, 0, 20);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghijklmnopqrst"), 10, 1, -10);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghijklmnopqrst"), 10, 5, -10);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghijklmnopqrst"), 10, 9, -10);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghijklmnopqrst"), 10, 10, -10);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghijklmnopqrst"), 10, 11, -10);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghijklmnopqrst"), 19, 0, 20);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghijklmnopqrst"), 19, 1, -19);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghijklmnopqrst"), 19, 2, -19);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghijklmnopqrst"), 20, 0, 20);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghijklmnopqrst"), 20, 1, 20);
    test(S("abcdefghijklmnopqrst"), 0, 21, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 0, S(""), 0, 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 0, S(""), 0, 1, 0);
    test(S("abcdefghijklmnopqrst"), 1, 0, S(""), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcde"), 0, 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcde"), 0, 1, -1);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcde"), 0, 2, -2);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcde"), 0, 4, -4);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcde"), 0, 5, -5);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcde"), 0, 6, -5);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcde"), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcde"), 1, 1, -1);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcde"), 1, 2, -2);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcde"), 1, 3, -3);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcde"), 1, 4, -4);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcde"), 1, 5, -4);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcde"), 2, 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcde"), 2, 1, -1);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcde"), 2, 2, -2);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcde"), 2, 3, -3);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcde"), 2, 4, -3);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcde"), 4, 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcde"), 4, 1, -1);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcde"), 4, 2, -1);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcde"), 5, 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcde"), 5, 1, 0);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcde"), 6, 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghij"), 0, 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghij"), 0, 1, -1);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghij"), 0, 5, -5);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghij"), 0, 9, -9);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghij"), 0, 10, -10);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghij"), 0, 11, -10);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghij"), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghij"), 1, 1, -1);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghij"), 1, 4, -4);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghij"), 1, 8, -8);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghij"), 1, 9, -9);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghij"), 1, 10, -9);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghij"), 5, 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghij"), 5, 1, -1);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghij"), 5, 2, -2);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghij"), 5, 4, -4);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghij"), 5, 5, -5);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghij"), 5, 6, -5);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghij"), 9, 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghij"), 9, 1, -1);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghij"), 9, 2, -1);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghij"), 10, 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghij"), 10, 1, 0);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghijklmnopqrst"), 0, 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghijklmnopqrst"), 0, 1, -1);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghijklmnopqrst"), 0, 10, -10);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghijklmnopqrst"), 0, 19, -19);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghijklmnopqrst"), 0, 20, -20);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghijklmnopqrst"), 0, 21, -20);
}

template <class S>
void test42()
{
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghijklmnopqrst"), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghijklmnopqrst"), 1, 1, -1);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghijklmnopqrst"), 1, 9, -9);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghijklmnopqrst"), 1, 18, -18);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghijklmnopqrst"), 1, 19, -19);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghijklmnopqrst"), 1, 20, -19);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghijklmnopqrst"), 10, 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghijklmnopqrst"), 10, 1, -1);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghijklmnopqrst"), 10, 5, -5);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghijklmnopqrst"), 10, 9, -9);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghijklmnopqrst"), 10, 10, -10);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghijklmnopqrst"), 10, 11, -10);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghijklmnopqrst"), 19, 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghijklmnopqrst"), 19, 1, -1);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghijklmnopqrst"), 19, 2, -1);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghijklmnopqrst"), 20, 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghijklmnopqrst"), 20, 1, 0);
    test(S("abcdefghijklmnopqrst"), 1, 0, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 1, S(""), 0, 0, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, S(""), 0, 1, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, S(""), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcde"), 0, 0, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcde"), 0, 1, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcde"), 0, 2, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcde"), 0, 4, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcde"), 0, 5, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcde"), 0, 6, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcde"), 1, 0, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcde"), 1, 1, 0);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcde"), 1, 2, -1);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcde"), 1, 3, -2);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcde"), 1, 4, -3);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcde"), 1, 5, -3);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcde"), 2, 0, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcde"), 2, 1, -1);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcde"), 2, 2, -1);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcde"), 2, 3, -1);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcde"), 2, 4, -1);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcde"), 4, 0, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcde"), 4, 1, -3);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcde"), 4, 2, -3);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcde"), 5, 0, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcde"), 5, 1, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcde"), 6, 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghij"), 0, 0, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghij"), 0, 1, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghij"), 0, 5, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghij"), 0, 9, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghij"), 0, 10, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghij"), 0, 11, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghij"), 1, 0, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghij"), 1, 1, 0);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghij"), 1, 4, -3);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghij"), 1, 8, -7);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghij"), 1, 9, -8);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghij"), 1, 10, -8);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghij"), 5, 0, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghij"), 5, 1, -4);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghij"), 5, 2, -4);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghij"), 5, 4, -4);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghij"), 5, 5, -4);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghij"), 5, 6, -4);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghij"), 9, 0, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghij"), 9, 1, -8);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghij"), 9, 2, -8);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghij"), 10, 0, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghij"), 10, 1, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghijklmnopqrst"), 0, 0, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghijklmnopqrst"), 0, 1, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghijklmnopqrst"), 0, 10, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghijklmnopqrst"), 0, 19, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghijklmnopqrst"), 0, 20, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghijklmnopqrst"), 0, 21, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghijklmnopqrst"), 1, 0, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghijklmnopqrst"), 1, 1, 0);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghijklmnopqrst"), 1, 9, -8);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghijklmnopqrst"), 1, 18, -17);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghijklmnopqrst"), 1, 19, -18);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghijklmnopqrst"), 1, 20, -18);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghijklmnopqrst"), 10, 0, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghijklmnopqrst"), 10, 1, -9);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghijklmnopqrst"), 10, 5, -9);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghijklmnopqrst"), 10, 9, -9);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghijklmnopqrst"), 10, 10, -9);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghijklmnopqrst"), 10, 11, -9);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghijklmnopqrst"), 19, 0, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghijklmnopqrst"), 19, 1, -18);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghijklmnopqrst"), 19, 2, -18);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghijklmnopqrst"), 20, 0, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghijklmnopqrst"), 20, 1, 1);
    test(S("abcdefghijklmnopqrst"), 1, 1, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 9, S(""), 0, 0, 9);
    test(S("abcdefghijklmnopqrst"), 1, 9, S(""), 0, 1, 9);
    test(S("abcdefghijklmnopqrst"), 1, 9, S(""), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcde"), 0, 0, 9);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcde"), 0, 1, 1);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcde"), 0, 2, 1);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcde"), 0, 4, 1);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcde"), 0, 5, 1);
}

template <class S>
void test43()
{
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcde"), 0, 6, 1);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcde"), 1, 0, 9);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcde"), 1, 1, 8);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcde"), 1, 2, 7);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcde"), 1, 3, 6);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcde"), 1, 4, 5);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcde"), 1, 5, 5);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcde"), 2, 0, 9);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcde"), 2, 1, -1);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcde"), 2, 2, -1);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcde"), 2, 3, -1);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcde"), 2, 4, -1);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcde"), 4, 0, 9);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcde"), 4, 1, -3);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcde"), 4, 2, -3);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcde"), 5, 0, 9);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcde"), 5, 1, 9);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcde"), 6, 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghij"), 0, 0, 9);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghij"), 0, 1, 1);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghij"), 0, 5, 1);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghij"), 0, 9, 1);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghij"), 0, 10, 1);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghij"), 0, 11, 1);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghij"), 1, 0, 9);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghij"), 1, 1, 8);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghij"), 1, 4, 5);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghij"), 1, 8, 1);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghij"), 1, 9, 0);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghij"), 1, 10, 0);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghij"), 5, 0, 9);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghij"), 5, 1, -4);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghij"), 5, 2, -4);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghij"), 5, 4, -4);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghij"), 5, 5, -4);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghij"), 5, 6, -4);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghij"), 9, 0, 9);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghij"), 9, 1, -8);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghij"), 9, 2, -8);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghij"), 10, 0, 9);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghij"), 10, 1, 9);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghijklmnopqrst"), 0, 0, 9);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghijklmnopqrst"), 0, 1, 1);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghijklmnopqrst"), 0, 10, 1);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghijklmnopqrst"), 0, 19, 1);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghijklmnopqrst"), 0, 20, 1);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghijklmnopqrst"), 0, 21, 1);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghijklmnopqrst"), 1, 0, 9);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghijklmnopqrst"), 1, 1, 8);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghijklmnopqrst"), 1, 9, 0);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghijklmnopqrst"), 1, 18, -9);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghijklmnopqrst"), 1, 19, -10);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghijklmnopqrst"), 1, 20, -10);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghijklmnopqrst"), 10, 0, 9);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghijklmnopqrst"), 10, 1, -9);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghijklmnopqrst"), 10, 5, -9);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghijklmnopqrst"), 10, 9, -9);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghijklmnopqrst"), 10, 10, -9);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghijklmnopqrst"), 10, 11, -9);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghijklmnopqrst"), 19, 0, 9);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghijklmnopqrst"), 19, 1, -18);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghijklmnopqrst"), 19, 2, -18);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghijklmnopqrst"), 20, 0, 9);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghijklmnopqrst"), 20, 1, 9);
    test(S("abcdefghijklmnopqrst"), 1, 9, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 18, S(""), 0, 0, 18);
    test(S("abcdefghijklmnopqrst"), 1, 18, S(""), 0, 1, 18);
    test(S("abcdefghijklmnopqrst"), 1, 18, S(""), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcde"), 0, 0, 18);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcde"), 0, 1, 1);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcde"), 0, 2, 1);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcde"), 0, 4, 1);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcde"), 0, 5, 1);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcde"), 0, 6, 1);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcde"), 1, 0, 18);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcde"), 1, 1, 17);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcde"), 1, 2, 16);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcde"), 1, 3, 15);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcde"), 1, 4, 14);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcde"), 1, 5, 14);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcde"), 2, 0, 18);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcde"), 2, 1, -1);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcde"), 2, 2, -1);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcde"), 2, 3, -1);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcde"), 2, 4, -1);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcde"), 4, 0, 18);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcde"), 4, 1, -3);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcde"), 4, 2, -3);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcde"), 5, 0, 18);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcde"), 5, 1, 18);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcde"), 6, 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghij"), 0, 0, 18);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghij"), 0, 1, 1);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghij"), 0, 5, 1);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghij"), 0, 9, 1);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghij"), 0, 10, 1);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghij"), 0, 11, 1);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghij"), 1, 0, 18);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghij"), 1, 1, 17);
}

template <class S>
void test44()
{
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghij"), 1, 4, 14);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghij"), 1, 8, 10);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghij"), 1, 9, 9);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghij"), 1, 10, 9);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghij"), 5, 0, 18);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghij"), 5, 1, -4);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghij"), 5, 2, -4);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghij"), 5, 4, -4);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghij"), 5, 5, -4);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghij"), 5, 6, -4);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghij"), 9, 0, 18);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghij"), 9, 1, -8);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghij"), 9, 2, -8);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghij"), 10, 0, 18);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghij"), 10, 1, 18);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghijklmnopqrst"), 0, 0, 18);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghijklmnopqrst"), 0, 1, 1);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghijklmnopqrst"), 0, 10, 1);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghijklmnopqrst"), 0, 19, 1);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghijklmnopqrst"), 0, 20, 1);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghijklmnopqrst"), 0, 21, 1);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghijklmnopqrst"), 1, 0, 18);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghijklmnopqrst"), 1, 1, 17);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghijklmnopqrst"), 1, 9, 9);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghijklmnopqrst"), 1, 18, 0);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghijklmnopqrst"), 1, 19, -1);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghijklmnopqrst"), 1, 20, -1);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghijklmnopqrst"), 10, 0, 18);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghijklmnopqrst"), 10, 1, -9);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghijklmnopqrst"), 10, 5, -9);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghijklmnopqrst"), 10, 9, -9);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghijklmnopqrst"), 10, 10, -9);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghijklmnopqrst"), 10, 11, -9);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghijklmnopqrst"), 19, 0, 18);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghijklmnopqrst"), 19, 1, -18);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghijklmnopqrst"), 19, 2, -18);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghijklmnopqrst"), 20, 0, 18);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghijklmnopqrst"), 20, 1, 18);
    test(S("abcdefghijklmnopqrst"), 1, 18, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 19, S(""), 0, 0, 19);
    test(S("abcdefghijklmnopqrst"), 1, 19, S(""), 0, 1, 19);
    test(S("abcdefghijklmnopqrst"), 1, 19, S(""), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcde"), 0, 0, 19);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcde"), 0, 1, 1);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcde"), 0, 2, 1);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcde"), 0, 4, 1);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcde"), 0, 5, 1);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcde"), 0, 6, 1);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcde"), 1, 0, 19);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcde"), 1, 1, 18);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcde"), 1, 2, 17);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcde"), 1, 3, 16);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcde"), 1, 4, 15);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcde"), 1, 5, 15);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcde"), 2, 0, 19);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcde"), 2, 1, -1);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcde"), 2, 2, -1);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcde"), 2, 3, -1);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcde"), 2, 4, -1);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcde"), 4, 0, 19);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcde"), 4, 1, -3);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcde"), 4, 2, -3);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcde"), 5, 0, 19);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcde"), 5, 1, 19);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcde"), 6, 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghij"), 0, 0, 19);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghij"), 0, 1, 1);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghij"), 0, 5, 1);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghij"), 0, 9, 1);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghij"), 0, 10, 1);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghij"), 0, 11, 1);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghij"), 1, 0, 19);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghij"), 1, 1, 18);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghij"), 1, 4, 15);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghij"), 1, 8, 11);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghij"), 1, 9, 10);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghij"), 1, 10, 10);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghij"), 5, 0, 19);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghij"), 5, 1, -4);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghij"), 5, 2, -4);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghij"), 5, 4, -4);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghij"), 5, 5, -4);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghij"), 5, 6, -4);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghij"), 9, 0, 19);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghij"), 9, 1, -8);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghij"), 9, 2, -8);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghij"), 10, 0, 19);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghij"), 10, 1, 19);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghijklmnopqrst"), 0, 0, 19);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghijklmnopqrst"), 0, 1, 1);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghijklmnopqrst"), 0, 10, 1);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghijklmnopqrst"), 0, 19, 1);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghijklmnopqrst"), 0, 20, 1);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghijklmnopqrst"), 0, 21, 1);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghijklmnopqrst"), 1, 0, 19);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghijklmnopqrst"), 1, 1, 18);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghijklmnopqrst"), 1, 9, 10);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghijklmnopqrst"), 1, 18, 1);
}

template <class S>
void test45()
{
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghijklmnopqrst"), 1, 19, 0);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghijklmnopqrst"), 1, 20, 0);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghijklmnopqrst"), 10, 0, 19);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghijklmnopqrst"), 10, 1, -9);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghijklmnopqrst"), 10, 5, -9);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghijklmnopqrst"), 10, 9, -9);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghijklmnopqrst"), 10, 10, -9);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghijklmnopqrst"), 10, 11, -9);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghijklmnopqrst"), 19, 0, 19);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghijklmnopqrst"), 19, 1, -18);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghijklmnopqrst"), 19, 2, -18);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghijklmnopqrst"), 20, 0, 19);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghijklmnopqrst"), 20, 1, 19);
    test(S("abcdefghijklmnopqrst"), 1, 19, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 20, S(""), 0, 0, 19);
    test(S("abcdefghijklmnopqrst"), 1, 20, S(""), 0, 1, 19);
    test(S("abcdefghijklmnopqrst"), 1, 20, S(""), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcde"), 0, 0, 19);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcde"), 0, 1, 1);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcde"), 0, 2, 1);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcde"), 0, 4, 1);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcde"), 0, 5, 1);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcde"), 0, 6, 1);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcde"), 1, 0, 19);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcde"), 1, 1, 18);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcde"), 1, 2, 17);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcde"), 1, 3, 16);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcde"), 1, 4, 15);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcde"), 1, 5, 15);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcde"), 2, 0, 19);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcde"), 2, 1, -1);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcde"), 2, 2, -1);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcde"), 2, 3, -1);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcde"), 2, 4, -1);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcde"), 4, 0, 19);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcde"), 4, 1, -3);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcde"), 4, 2, -3);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcde"), 5, 0, 19);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcde"), 5, 1, 19);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcde"), 6, 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghij"), 0, 0, 19);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghij"), 0, 1, 1);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghij"), 0, 5, 1);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghij"), 0, 9, 1);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghij"), 0, 10, 1);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghij"), 0, 11, 1);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghij"), 1, 0, 19);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghij"), 1, 1, 18);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghij"), 1, 4, 15);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghij"), 1, 8, 11);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghij"), 1, 9, 10);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghij"), 1, 10, 10);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghij"), 5, 0, 19);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghij"), 5, 1, -4);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghij"), 5, 2, -4);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghij"), 5, 4, -4);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghij"), 5, 5, -4);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghij"), 5, 6, -4);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghij"), 9, 0, 19);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghij"), 9, 1, -8);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghij"), 9, 2, -8);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghij"), 10, 0, 19);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghij"), 10, 1, 19);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghijklmnopqrst"), 0, 0, 19);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghijklmnopqrst"), 0, 1, 1);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghijklmnopqrst"), 0, 10, 1);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghijklmnopqrst"), 0, 19, 1);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghijklmnopqrst"), 0, 20, 1);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghijklmnopqrst"), 0, 21, 1);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghijklmnopqrst"), 1, 0, 19);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghijklmnopqrst"), 1, 1, 18);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghijklmnopqrst"), 1, 9, 10);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghijklmnopqrst"), 1, 18, 1);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghijklmnopqrst"), 1, 19, 0);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghijklmnopqrst"), 1, 20, 0);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghijklmnopqrst"), 10, 0, 19);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghijklmnopqrst"), 10, 1, -9);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghijklmnopqrst"), 10, 5, -9);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghijklmnopqrst"), 10, 9, -9);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghijklmnopqrst"), 10, 10, -9);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghijklmnopqrst"), 10, 11, -9);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghijklmnopqrst"), 19, 0, 19);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghijklmnopqrst"), 19, 1, -18);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghijklmnopqrst"), 19, 2, -18);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghijklmnopqrst"), 20, 0, 19);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghijklmnopqrst"), 20, 1, 19);
    test(S("abcdefghijklmnopqrst"), 1, 20, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 0, S(""), 0, 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 0, S(""), 0, 1, 0);
    test(S("abcdefghijklmnopqrst"), 10, 0, S(""), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcde"), 0, 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcde"), 0, 1, -1);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcde"), 0, 2, -2);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcde"), 0, 4, -4);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcde"), 0, 5, -5);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcde"), 0, 6, -5);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcde"), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcde"), 1, 1, -1);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcde"), 1, 2, -2);
}

template <class S>
void test46()
{
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcde"), 1, 3, -3);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcde"), 1, 4, -4);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcde"), 1, 5, -4);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcde"), 2, 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcde"), 2, 1, -1);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcde"), 2, 2, -2);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcde"), 2, 3, -3);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcde"), 2, 4, -3);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcde"), 4, 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcde"), 4, 1, -1);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcde"), 4, 2, -1);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcde"), 5, 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcde"), 5, 1, 0);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcde"), 6, 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghij"), 0, 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghij"), 0, 1, -1);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghij"), 0, 5, -5);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghij"), 0, 9, -9);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghij"), 0, 10, -10);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghij"), 0, 11, -10);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghij"), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghij"), 1, 1, -1);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghij"), 1, 4, -4);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghij"), 1, 8, -8);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghij"), 1, 9, -9);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghij"), 1, 10, -9);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghij"), 5, 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghij"), 5, 1, -1);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghij"), 5, 2, -2);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghij"), 5, 4, -4);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghij"), 5, 5, -5);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghij"), 5, 6, -5);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghij"), 9, 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghij"), 9, 1, -1);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghij"), 9, 2, -1);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghij"), 10, 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghij"), 10, 1, 0);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghijklmnopqrst"), 0, 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghijklmnopqrst"), 0, 1, -1);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghijklmnopqrst"), 0, 10, -10);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghijklmnopqrst"), 0, 19, -19);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghijklmnopqrst"), 0, 20, -20);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghijklmnopqrst"), 0, 21, -20);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghijklmnopqrst"), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghijklmnopqrst"), 1, 1, -1);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghijklmnopqrst"), 1, 9, -9);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghijklmnopqrst"), 1, 18, -18);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghijklmnopqrst"), 1, 19, -19);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghijklmnopqrst"), 1, 20, -19);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghijklmnopqrst"), 10, 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghijklmnopqrst"), 10, 1, -1);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghijklmnopqrst"), 10, 5, -5);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghijklmnopqrst"), 10, 9, -9);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghijklmnopqrst"), 10, 10, -10);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghijklmnopqrst"), 10, 11, -10);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghijklmnopqrst"), 19, 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghijklmnopqrst"), 19, 1, -1);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghijklmnopqrst"), 19, 2, -1);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghijklmnopqrst"), 20, 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghijklmnopqrst"), 20, 1, 0);
    test(S("abcdefghijklmnopqrst"), 10, 0, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 1, S(""), 0, 0, 1);
    test(S("abcdefghijklmnopqrst"), 10, 1, S(""), 0, 1, 1);
    test(S("abcdefghijklmnopqrst"), 10, 1, S(""), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcde"), 0, 0, 1);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcde"), 0, 1, 10);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcde"), 0, 2, 10);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcde"), 0, 4, 10);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcde"), 0, 5, 10);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcde"), 0, 6, 10);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcde"), 1, 0, 1);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcde"), 1, 1, 9);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcde"), 1, 2, 9);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcde"), 1, 3, 9);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcde"), 1, 4, 9);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcde"), 1, 5, 9);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcde"), 2, 0, 1);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcde"), 2, 1, 8);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcde"), 2, 2, 8);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcde"), 2, 3, 8);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcde"), 2, 4, 8);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcde"), 4, 0, 1);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcde"), 4, 1, 6);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcde"), 4, 2, 6);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcde"), 5, 0, 1);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcde"), 5, 1, 1);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcde"), 6, 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghij"), 0, 0, 1);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghij"), 0, 1, 10);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghij"), 0, 5, 10);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghij"), 0, 9, 10);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghij"), 0, 10, 10);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghij"), 0, 11, 10);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghij"), 1, 0, 1);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghij"), 1, 1, 9);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghij"), 1, 4, 9);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghij"), 1, 8, 9);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghij"), 1, 9, 9);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghij"), 1, 10, 9);
}

template <class S>
void test47()
{
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghij"), 5, 0, 1);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghij"), 5, 1, 5);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghij"), 5, 2, 5);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghij"), 5, 4, 5);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghij"), 5, 5, 5);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghij"), 5, 6, 5);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghij"), 9, 0, 1);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghij"), 9, 1, 1);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghij"), 9, 2, 1);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghij"), 10, 0, 1);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghij"), 10, 1, 1);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghijklmnopqrst"), 0, 0, 1);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghijklmnopqrst"), 0, 1, 10);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghijklmnopqrst"), 0, 10, 10);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghijklmnopqrst"), 0, 19, 10);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghijklmnopqrst"), 0, 20, 10);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghijklmnopqrst"), 0, 21, 10);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghijklmnopqrst"), 1, 0, 1);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghijklmnopqrst"), 1, 1, 9);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghijklmnopqrst"), 1, 9, 9);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghijklmnopqrst"), 1, 18, 9);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghijklmnopqrst"), 1, 19, 9);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghijklmnopqrst"), 1, 20, 9);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghijklmnopqrst"), 10, 0, 1);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghijklmnopqrst"), 10, 1, 0);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghijklmnopqrst"), 10, 5, -4);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghijklmnopqrst"), 10, 9, -8);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghijklmnopqrst"), 10, 10, -9);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghijklmnopqrst"), 10, 11, -9);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghijklmnopqrst"), 19, 0, 1);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghijklmnopqrst"), 19, 1, -9);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghijklmnopqrst"), 19, 2, -9);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghijklmnopqrst"), 20, 0, 1);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghijklmnopqrst"), 20, 1, 1);
    test(S("abcdefghijklmnopqrst"), 10, 1, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 5, S(""), 0, 0, 5);
    test(S("abcdefghijklmnopqrst"), 10, 5, S(""), 0, 1, 5);
    test(S("abcdefghijklmnopqrst"), 10, 5, S(""), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcde"), 0, 0, 5);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcde"), 0, 1, 10);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcde"), 0, 2, 10);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcde"), 0, 4, 10);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcde"), 0, 5, 10);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcde"), 0, 6, 10);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcde"), 1, 0, 5);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcde"), 1, 1, 9);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcde"), 1, 2, 9);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcde"), 1, 3, 9);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcde"), 1, 4, 9);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcde"), 1, 5, 9);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcde"), 2, 0, 5);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcde"), 2, 1, 8);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcde"), 2, 2, 8);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcde"), 2, 3, 8);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcde"), 2, 4, 8);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcde"), 4, 0, 5);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcde"), 4, 1, 6);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcde"), 4, 2, 6);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcde"), 5, 0, 5);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcde"), 5, 1, 5);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcde"), 6, 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghij"), 0, 0, 5);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghij"), 0, 1, 10);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghij"), 0, 5, 10);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghij"), 0, 9, 10);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghij"), 0, 10, 10);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghij"), 0, 11, 10);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghij"), 1, 0, 5);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghij"), 1, 1, 9);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghij"), 1, 4, 9);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghij"), 1, 8, 9);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghij"), 1, 9, 9);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghij"), 1, 10, 9);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghij"), 5, 0, 5);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghij"), 5, 1, 5);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghij"), 5, 2, 5);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghij"), 5, 4, 5);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghij"), 5, 5, 5);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghij"), 5, 6, 5);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghij"), 9, 0, 5);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghij"), 9, 1, 1);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghij"), 9, 2, 1);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghij"), 10, 0, 5);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghij"), 10, 1, 5);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghijklmnopqrst"), 0, 0, 5);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghijklmnopqrst"), 0, 1, 10);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghijklmnopqrst"), 0, 10, 10);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghijklmnopqrst"), 0, 19, 10);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghijklmnopqrst"), 0, 20, 10);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghijklmnopqrst"), 0, 21, 10);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghijklmnopqrst"), 1, 0, 5);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghijklmnopqrst"), 1, 1, 9);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghijklmnopqrst"), 1, 9, 9);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghijklmnopqrst"), 1, 18, 9);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghijklmnopqrst"), 1, 19, 9);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghijklmnopqrst"), 1, 20, 9);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghijklmnopqrst"), 10, 0, 5);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghijklmnopqrst"), 10, 1, 4);
}

template <class S>
void test48()
{
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghijklmnopqrst"), 10, 5, 0);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghijklmnopqrst"), 10, 9, -4);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghijklmnopqrst"), 10, 10, -5);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghijklmnopqrst"), 10, 11, -5);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghijklmnopqrst"), 19, 0, 5);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghijklmnopqrst"), 19, 1, -9);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghijklmnopqrst"), 19, 2, -9);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghijklmnopqrst"), 20, 0, 5);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghijklmnopqrst"), 20, 1, 5);
    test(S("abcdefghijklmnopqrst"), 10, 5, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 9, S(""), 0, 0, 9);
    test(S("abcdefghijklmnopqrst"), 10, 9, S(""), 0, 1, 9);
    test(S("abcdefghijklmnopqrst"), 10, 9, S(""), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcde"), 0, 0, 9);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcde"), 0, 1, 10);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcde"), 0, 2, 10);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcde"), 0, 4, 10);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcde"), 0, 5, 10);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcde"), 0, 6, 10);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcde"), 1, 0, 9);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcde"), 1, 1, 9);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcde"), 1, 2, 9);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcde"), 1, 3, 9);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcde"), 1, 4, 9);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcde"), 1, 5, 9);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcde"), 2, 0, 9);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcde"), 2, 1, 8);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcde"), 2, 2, 8);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcde"), 2, 3, 8);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcde"), 2, 4, 8);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcde"), 4, 0, 9);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcde"), 4, 1, 6);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcde"), 4, 2, 6);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcde"), 5, 0, 9);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcde"), 5, 1, 9);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcde"), 6, 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghij"), 0, 0, 9);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghij"), 0, 1, 10);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghij"), 0, 5, 10);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghij"), 0, 9, 10);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghij"), 0, 10, 10);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghij"), 0, 11, 10);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghij"), 1, 0, 9);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghij"), 1, 1, 9);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghij"), 1, 4, 9);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghij"), 1, 8, 9);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghij"), 1, 9, 9);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghij"), 1, 10, 9);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghij"), 5, 0, 9);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghij"), 5, 1, 5);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghij"), 5, 2, 5);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghij"), 5, 4, 5);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghij"), 5, 5, 5);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghij"), 5, 6, 5);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghij"), 9, 0, 9);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghij"), 9, 1, 1);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghij"), 9, 2, 1);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghij"), 10, 0, 9);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghij"), 10, 1, 9);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghijklmnopqrst"), 0, 0, 9);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghijklmnopqrst"), 0, 1, 10);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghijklmnopqrst"), 0, 10, 10);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghijklmnopqrst"), 0, 19, 10);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghijklmnopqrst"), 0, 20, 10);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghijklmnopqrst"), 0, 21, 10);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghijklmnopqrst"), 1, 0, 9);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghijklmnopqrst"), 1, 1, 9);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghijklmnopqrst"), 1, 9, 9);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghijklmnopqrst"), 1, 18, 9);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghijklmnopqrst"), 1, 19, 9);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghijklmnopqrst"), 1, 20, 9);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghijklmnopqrst"), 10, 0, 9);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghijklmnopqrst"), 10, 1, 8);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghijklmnopqrst"), 10, 5, 4);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghijklmnopqrst"), 10, 9, 0);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghijklmnopqrst"), 10, 10, -1);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghijklmnopqrst"), 10, 11, -1);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghijklmnopqrst"), 19, 0, 9);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghijklmnopqrst"), 19, 1, -9);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghijklmnopqrst"), 19, 2, -9);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghijklmnopqrst"), 20, 0, 9);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghijklmnopqrst"), 20, 1, 9);
    test(S("abcdefghijklmnopqrst"), 10, 9, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 10, S(""), 0, 0, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, S(""), 0, 1, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, S(""), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcde"), 0, 0, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcde"), 0, 1, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcde"), 0, 2, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcde"), 0, 4, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcde"), 0, 5, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcde"), 0, 6, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcde"), 1, 0, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcde"), 1, 1, 9);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcde"), 1, 2, 9);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcde"), 1, 3, 9);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcde"), 1, 4, 9);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcde"), 1, 5, 9);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcde"), 2, 0, 10);
}

template <class S>
void test49()
{
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcde"), 2, 1, 8);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcde"), 2, 2, 8);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcde"), 2, 3, 8);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcde"), 2, 4, 8);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcde"), 4, 0, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcde"), 4, 1, 6);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcde"), 4, 2, 6);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcde"), 5, 0, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcde"), 5, 1, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcde"), 6, 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghij"), 0, 0, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghij"), 0, 1, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghij"), 0, 5, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghij"), 0, 9, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghij"), 0, 10, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghij"), 0, 11, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghij"), 1, 0, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghij"), 1, 1, 9);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghij"), 1, 4, 9);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghij"), 1, 8, 9);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghij"), 1, 9, 9);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghij"), 1, 10, 9);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghij"), 5, 0, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghij"), 5, 1, 5);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghij"), 5, 2, 5);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghij"), 5, 4, 5);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghij"), 5, 5, 5);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghij"), 5, 6, 5);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghij"), 9, 0, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghij"), 9, 1, 1);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghij"), 9, 2, 1);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghij"), 10, 0, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghij"), 10, 1, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghijklmnopqrst"), 0, 0, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghijklmnopqrst"), 0, 1, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghijklmnopqrst"), 0, 10, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghijklmnopqrst"), 0, 19, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghijklmnopqrst"), 0, 20, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghijklmnopqrst"), 0, 21, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghijklmnopqrst"), 1, 0, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghijklmnopqrst"), 1, 1, 9);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghijklmnopqrst"), 1, 9, 9);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghijklmnopqrst"), 1, 18, 9);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghijklmnopqrst"), 1, 19, 9);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghijklmnopqrst"), 1, 20, 9);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghijklmnopqrst"), 10, 0, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghijklmnopqrst"), 10, 1, 9);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghijklmnopqrst"), 10, 5, 5);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghijklmnopqrst"), 10, 9, 1);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghijklmnopqrst"), 10, 10, 0);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghijklmnopqrst"), 10, 11, 0);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghijklmnopqrst"), 19, 0, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghijklmnopqrst"), 19, 1, -9);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghijklmnopqrst"), 19, 2, -9);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghijklmnopqrst"), 20, 0, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghijklmnopqrst"), 20, 1, 10);
    test(S("abcdefghijklmnopqrst"), 10, 10, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 11, S(""), 0, 0, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, S(""), 0, 1, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, S(""), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcde"), 0, 0, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcde"), 0, 1, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcde"), 0, 2, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcde"), 0, 4, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcde"), 0, 5, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcde"), 0, 6, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcde"), 1, 0, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcde"), 1, 1, 9);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcde"), 1, 2, 9);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcde"), 1, 3, 9);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcde"), 1, 4, 9);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcde"), 1, 5, 9);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcde"), 2, 0, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcde"), 2, 1, 8);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcde"), 2, 2, 8);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcde"), 2, 3, 8);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcde"), 2, 4, 8);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcde"), 4, 0, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcde"), 4, 1, 6);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcde"), 4, 2, 6);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcde"), 5, 0, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcde"), 5, 1, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcde"), 6, 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghij"), 0, 0, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghij"), 0, 1, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghij"), 0, 5, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghij"), 0, 9, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghij"), 0, 10, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghij"), 0, 11, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghij"), 1, 0, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghij"), 1, 1, 9);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghij"), 1, 4, 9);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghij"), 1, 8, 9);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghij"), 1, 9, 9);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghij"), 1, 10, 9);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghij"), 5, 0, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghij"), 5, 1, 5);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghij"), 5, 2, 5);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghij"), 5, 4, 5);
}

template <class S>
void test50()
{
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghij"), 5, 5, 5);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghij"), 5, 6, 5);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghij"), 9, 0, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghij"), 9, 1, 1);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghij"), 9, 2, 1);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghij"), 10, 0, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghij"), 10, 1, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghijklmnopqrst"), 0, 0, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghijklmnopqrst"), 0, 1, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghijklmnopqrst"), 0, 10, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghijklmnopqrst"), 0, 19, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghijklmnopqrst"), 0, 20, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghijklmnopqrst"), 0, 21, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghijklmnopqrst"), 1, 0, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghijklmnopqrst"), 1, 1, 9);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghijklmnopqrst"), 1, 9, 9);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghijklmnopqrst"), 1, 18, 9);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghijklmnopqrst"), 1, 19, 9);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghijklmnopqrst"), 1, 20, 9);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghijklmnopqrst"), 10, 0, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghijklmnopqrst"), 10, 1, 9);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghijklmnopqrst"), 10, 5, 5);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghijklmnopqrst"), 10, 9, 1);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghijklmnopqrst"), 10, 10, 0);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghijklmnopqrst"), 10, 11, 0);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghijklmnopqrst"), 19, 0, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghijklmnopqrst"), 19, 1, -9);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghijklmnopqrst"), 19, 2, -9);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghijklmnopqrst"), 20, 0, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghijklmnopqrst"), 20, 1, 10);
    test(S("abcdefghijklmnopqrst"), 10, 11, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghijklmnopqrst"), 19, 0, S(""), 0, 0, 0);
    test(S("abcdefghijklmnopqrst"), 19, 0, S(""), 0, 1, 0);
    test(S("abcdefghijklmnopqrst"), 19, 0, S(""), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcde"), 0, 0, 0);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcde"), 0, 1, -1);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcde"), 0, 2, -2);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcde"), 0, 4, -4);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcde"), 0, 5, -5);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcde"), 0, 6, -5);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcde"), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcde"), 1, 1, -1);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcde"), 1, 2, -2);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcde"), 1, 3, -3);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcde"), 1, 4, -4);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcde"), 1, 5, -4);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcde"), 2, 0, 0);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcde"), 2, 1, -1);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcde"), 2, 2, -2);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcde"), 2, 3, -3);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcde"), 2, 4, -3);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcde"), 4, 0, 0);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcde"), 4, 1, -1);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcde"), 4, 2, -1);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcde"), 5, 0, 0);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcde"), 5, 1, 0);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcde"), 6, 0, 0);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghij"), 0, 0, 0);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghij"), 0, 1, -1);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghij"), 0, 5, -5);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghij"), 0, 9, -9);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghij"), 0, 10, -10);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghij"), 0, 11, -10);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghij"), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghij"), 1, 1, -1);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghij"), 1, 4, -4);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghij"), 1, 8, -8);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghij"), 1, 9, -9);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghij"), 1, 10, -9);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghij"), 5, 0, 0);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghij"), 5, 1, -1);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghij"), 5, 2, -2);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghij"), 5, 4, -4);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghij"), 5, 5, -5);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghij"), 5, 6, -5);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghij"), 9, 0, 0);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghij"), 9, 1, -1);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghij"), 9, 2, -1);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghij"), 10, 0, 0);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghij"), 10, 1, 0);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghijklmnopqrst"), 0, 0, 0);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghijklmnopqrst"), 0, 1, -1);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghijklmnopqrst"), 0, 10, -10);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghijklmnopqrst"), 0, 19, -19);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghijklmnopqrst"), 0, 20, -20);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghijklmnopqrst"), 0, 21, -20);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghijklmnopqrst"), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghijklmnopqrst"), 1, 1, -1);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghijklmnopqrst"), 1, 9, -9);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghijklmnopqrst"), 1, 18, -18);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghijklmnopqrst"), 1, 19, -19);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghijklmnopqrst"), 1, 20, -19);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghijklmnopqrst"), 10, 0, 0);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghijklmnopqrst"), 10, 1, -1);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghijklmnopqrst"), 10, 5, -5);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghijklmnopqrst"), 10, 9, -9);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghijklmnopqrst"), 10, 10, -10);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghijklmnopqrst"), 10, 11, -10);
}

template <class S>
void test51()
{
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghijklmnopqrst"), 19, 0, 0);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghijklmnopqrst"), 19, 1, -1);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghijklmnopqrst"), 19, 2, -1);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghijklmnopqrst"), 20, 0, 0);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghijklmnopqrst"), 20, 1, 0);
    test(S("abcdefghijklmnopqrst"), 19, 0, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghijklmnopqrst"), 19, 1, S(""), 0, 0, 1);
    test(S("abcdefghijklmnopqrst"), 19, 1, S(""), 0, 1, 1);
    test(S("abcdefghijklmnopqrst"), 19, 1, S(""), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcde"), 0, 0, 1);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcde"), 0, 1, 19);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcde"), 0, 2, 19);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcde"), 0, 4, 19);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcde"), 0, 5, 19);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcde"), 0, 6, 19);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcde"), 1, 0, 1);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcde"), 1, 1, 18);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcde"), 1, 2, 18);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcde"), 1, 3, 18);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcde"), 1, 4, 18);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcde"), 1, 5, 18);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcde"), 2, 0, 1);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcde"), 2, 1, 17);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcde"), 2, 2, 17);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcde"), 2, 3, 17);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcde"), 2, 4, 17);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcde"), 4, 0, 1);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcde"), 4, 1, 15);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcde"), 4, 2, 15);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcde"), 5, 0, 1);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcde"), 5, 1, 1);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcde"), 6, 0, 0);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghij"), 0, 0, 1);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghij"), 0, 1, 19);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghij"), 0, 5, 19);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghij"), 0, 9, 19);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghij"), 0, 10, 19);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghij"), 0, 11, 19);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghij"), 1, 0, 1);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghij"), 1, 1, 18);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghij"), 1, 4, 18);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghij"), 1, 8, 18);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghij"), 1, 9, 18);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghij"), 1, 10, 18);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghij"), 5, 0, 1);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghij"), 5, 1, 14);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghij"), 5, 2, 14);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghij"), 5, 4, 14);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghij"), 5, 5, 14);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghij"), 5, 6, 14);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghij"), 9, 0, 1);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghij"), 9, 1, 10);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghij"), 9, 2, 10);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghij"), 10, 0, 1);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghij"), 10, 1, 1);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghijklmnopqrst"), 0, 0, 1);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghijklmnopqrst"), 0, 1, 19);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghijklmnopqrst"), 0, 10, 19);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghijklmnopqrst"), 0, 19, 19);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghijklmnopqrst"), 0, 20, 19);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghijklmnopqrst"), 0, 21, 19);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghijklmnopqrst"), 1, 0, 1);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghijklmnopqrst"), 1, 1, 18);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghijklmnopqrst"), 1, 9, 18);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghijklmnopqrst"), 1, 18, 18);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghijklmnopqrst"), 1, 19, 18);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghijklmnopqrst"), 1, 20, 18);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghijklmnopqrst"), 10, 0, 1);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghijklmnopqrst"), 10, 1, 9);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghijklmnopqrst"), 10, 5, 9);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghijklmnopqrst"), 10, 9, 9);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghijklmnopqrst"), 10, 10, 9);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghijklmnopqrst"), 10, 11, 9);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghijklmnopqrst"), 19, 0, 1);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghijklmnopqrst"), 19, 1, 0);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghijklmnopqrst"), 19, 2, 0);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghijklmnopqrst"), 20, 0, 1);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghijklmnopqrst"), 20, 1, 1);
    test(S("abcdefghijklmnopqrst"), 19, 1, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghijklmnopqrst"), 19, 2, S(""), 0, 0, 1);
    test(S("abcdefghijklmnopqrst"), 19, 2, S(""), 0, 1, 1);
    test(S("abcdefghijklmnopqrst"), 19, 2, S(""), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcde"), 0, 0, 1);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcde"), 0, 1, 19);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcde"), 0, 2, 19);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcde"), 0, 4, 19);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcde"), 0, 5, 19);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcde"), 0, 6, 19);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcde"), 1, 0, 1);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcde"), 1, 1, 18);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcde"), 1, 2, 18);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcde"), 1, 3, 18);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcde"), 1, 4, 18);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcde"), 1, 5, 18);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcde"), 2, 0, 1);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcde"), 2, 1, 17);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcde"), 2, 2, 17);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcde"), 2, 3, 17);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcde"), 2, 4, 17);
}

template <class S>
void test52()
{
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcde"), 4, 0, 1);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcde"), 4, 1, 15);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcde"), 4, 2, 15);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcde"), 5, 0, 1);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcde"), 5, 1, 1);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcde"), 6, 0, 0);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghij"), 0, 0, 1);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghij"), 0, 1, 19);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghij"), 0, 5, 19);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghij"), 0, 9, 19);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghij"), 0, 10, 19);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghij"), 0, 11, 19);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghij"), 1, 0, 1);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghij"), 1, 1, 18);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghij"), 1, 4, 18);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghij"), 1, 8, 18);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghij"), 1, 9, 18);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghij"), 1, 10, 18);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghij"), 5, 0, 1);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghij"), 5, 1, 14);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghij"), 5, 2, 14);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghij"), 5, 4, 14);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghij"), 5, 5, 14);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghij"), 5, 6, 14);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghij"), 9, 0, 1);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghij"), 9, 1, 10);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghij"), 9, 2, 10);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghij"), 10, 0, 1);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghij"), 10, 1, 1);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghijklmnopqrst"), 0, 0, 1);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghijklmnopqrst"), 0, 1, 19);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghijklmnopqrst"), 0, 10, 19);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghijklmnopqrst"), 0, 19, 19);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghijklmnopqrst"), 0, 20, 19);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghijklmnopqrst"), 0, 21, 19);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghijklmnopqrst"), 1, 0, 1);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghijklmnopqrst"), 1, 1, 18);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghijklmnopqrst"), 1, 9, 18);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghijklmnopqrst"), 1, 18, 18);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghijklmnopqrst"), 1, 19, 18);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghijklmnopqrst"), 1, 20, 18);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghijklmnopqrst"), 10, 0, 1);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghijklmnopqrst"), 10, 1, 9);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghijklmnopqrst"), 10, 5, 9);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghijklmnopqrst"), 10, 9, 9);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghijklmnopqrst"), 10, 10, 9);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghijklmnopqrst"), 10, 11, 9);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghijklmnopqrst"), 19, 0, 1);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghijklmnopqrst"), 19, 1, 0);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghijklmnopqrst"), 19, 2, 0);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghijklmnopqrst"), 20, 0, 1);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghijklmnopqrst"), 20, 1, 1);
    test(S("abcdefghijklmnopqrst"), 19, 2, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 0, S(""), 0, 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 0, S(""), 0, 1, 0);
    test(S("abcdefghijklmnopqrst"), 20, 0, S(""), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcde"), 0, 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcde"), 0, 1, -1);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcde"), 0, 2, -2);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcde"), 0, 4, -4);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcde"), 0, 5, -5);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcde"), 0, 6, -5);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcde"), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcde"), 1, 1, -1);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcde"), 1, 2, -2);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcde"), 1, 3, -3);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcde"), 1, 4, -4);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcde"), 1, 5, -4);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcde"), 2, 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcde"), 2, 1, -1);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcde"), 2, 2, -2);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcde"), 2, 3, -3);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcde"), 2, 4, -3);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcde"), 4, 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcde"), 4, 1, -1);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcde"), 4, 2, -1);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcde"), 5, 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcde"), 5, 1, 0);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcde"), 6, 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghij"), 0, 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghij"), 0, 1, -1);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghij"), 0, 5, -5);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghij"), 0, 9, -9);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghij"), 0, 10, -10);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghij"), 0, 11, -10);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghij"), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghij"), 1, 1, -1);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghij"), 1, 4, -4);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghij"), 1, 8, -8);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghij"), 1, 9, -9);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghij"), 1, 10, -9);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghij"), 5, 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghij"), 5, 1, -1);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghij"), 5, 2, -2);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghij"), 5, 4, -4);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghij"), 5, 5, -5);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghij"), 5, 6, -5);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghij"), 9, 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghij"), 9, 1, -1);
}

template <class S>
void test53()
{
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghij"), 9, 2, -1);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghij"), 10, 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghij"), 10, 1, 0);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghijklmnopqrst"), 0, 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghijklmnopqrst"), 0, 1, -1);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghijklmnopqrst"), 0, 10, -10);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghijklmnopqrst"), 0, 19, -19);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghijklmnopqrst"), 0, 20, -20);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghijklmnopqrst"), 0, 21, -20);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghijklmnopqrst"), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghijklmnopqrst"), 1, 1, -1);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghijklmnopqrst"), 1, 9, -9);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghijklmnopqrst"), 1, 18, -18);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghijklmnopqrst"), 1, 19, -19);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghijklmnopqrst"), 1, 20, -19);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghijklmnopqrst"), 10, 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghijklmnopqrst"), 10, 1, -1);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghijklmnopqrst"), 10, 5, -5);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghijklmnopqrst"), 10, 9, -9);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghijklmnopqrst"), 10, 10, -10);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghijklmnopqrst"), 10, 11, -10);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghijklmnopqrst"), 19, 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghijklmnopqrst"), 19, 1, -1);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghijklmnopqrst"), 19, 2, -1);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghijklmnopqrst"), 20, 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghijklmnopqrst"), 20, 1, 0);
    test(S("abcdefghijklmnopqrst"), 20, 0, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 1, S(""), 0, 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 1, S(""), 0, 1, 0);
    test(S("abcdefghijklmnopqrst"), 20, 1, S(""), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcde"), 0, 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcde"), 0, 1, -1);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcde"), 0, 2, -2);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcde"), 0, 4, -4);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcde"), 0, 5, -5);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcde"), 0, 6, -5);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcde"), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcde"), 1, 1, -1);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcde"), 1, 2, -2);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcde"), 1, 3, -3);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcde"), 1, 4, -4);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcde"), 1, 5, -4);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcde"), 2, 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcde"), 2, 1, -1);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcde"), 2, 2, -2);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcde"), 2, 3, -3);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcde"), 2, 4, -3);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcde"), 4, 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcde"), 4, 1, -1);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcde"), 4, 2, -1);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcde"), 5, 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcde"), 5, 1, 0);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcde"), 6, 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghij"), 0, 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghij"), 0, 1, -1);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghij"), 0, 5, -5);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghij"), 0, 9, -9);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghij"), 0, 10, -10);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghij"), 0, 11, -10);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghij"), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghij"), 1, 1, -1);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghij"), 1, 4, -4);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghij"), 1, 8, -8);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghij"), 1, 9, -9);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghij"), 1, 10, -9);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghij"), 5, 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghij"), 5, 1, -1);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghij"), 5, 2, -2);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghij"), 5, 4, -4);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghij"), 5, 5, -5);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghij"), 5, 6, -5);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghij"), 9, 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghij"), 9, 1, -1);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghij"), 9, 2, -1);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghij"), 10, 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghij"), 10, 1, 0);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghijklmnopqrst"), 0, 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghijklmnopqrst"), 0, 1, -1);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghijklmnopqrst"), 0, 10, -10);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghijklmnopqrst"), 0, 19, -19);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghijklmnopqrst"), 0, 20, -20);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghijklmnopqrst"), 0, 21, -20);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghijklmnopqrst"), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghijklmnopqrst"), 1, 1, -1);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghijklmnopqrst"), 1, 9, -9);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghijklmnopqrst"), 1, 18, -18);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghijklmnopqrst"), 1, 19, -19);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghijklmnopqrst"), 1, 20, -19);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghijklmnopqrst"), 10, 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghijklmnopqrst"), 10, 1, -1);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghijklmnopqrst"), 10, 5, -5);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghijklmnopqrst"), 10, 9, -9);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghijklmnopqrst"), 10, 10, -10);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghijklmnopqrst"), 10, 11, -10);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghijklmnopqrst"), 19, 0, 0);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghijklmnopqrst"), 19, 1, -1);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghijklmnopqrst"), 19, 2, -1);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghijklmnopqrst"), 20, 0, 0);
}

template <class S>
void test54()
{
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghijklmnopqrst"), 20, 1, 0);
    test(S("abcdefghijklmnopqrst"), 20, 1, S("abcdefghijklmnopqrst"), 21, 0, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S(""), 0, 0, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S(""), 0, 1, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S(""), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcde"), 0, 0, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcde"), 0, 1, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcde"), 0, 2, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcde"), 0, 4, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcde"), 0, 5, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcde"), 0, 6, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcde"), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcde"), 1, 1, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcde"), 1, 2, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcde"), 1, 3, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcde"), 1, 4, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcde"), 1, 5, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcde"), 2, 0, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcde"), 2, 1, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcde"), 2, 2, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcde"), 2, 3, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcde"), 2, 4, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcde"), 4, 0, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcde"), 4, 1, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcde"), 4, 2, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcde"), 5, 0, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcde"), 5, 1, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcde"), 6, 0, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghij"), 0, 0, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghij"), 0, 1, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghij"), 0, 5, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghij"), 0, 9, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghij"), 0, 10, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghij"), 0, 11, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghij"), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghij"), 1, 1, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghij"), 1, 4, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghij"), 1, 8, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghij"), 1, 9, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghij"), 1, 10, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghij"), 5, 0, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghij"), 5, 1, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghij"), 5, 2, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghij"), 5, 4, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghij"), 5, 5, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghij"), 5, 6, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghij"), 9, 0, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghij"), 9, 1, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghij"), 9, 2, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghij"), 10, 0, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghij"), 10, 1, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghij"), 11, 0, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghijklmnopqrst"), 0, 0, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghijklmnopqrst"), 0, 1, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghijklmnopqrst"), 0, 10, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghijklmnopqrst"), 0, 19, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghijklmnopqrst"), 0, 20, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghijklmnopqrst"), 0, 21, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghijklmnopqrst"), 1, 0, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghijklmnopqrst"), 1, 1, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghijklmnopqrst"), 1, 9, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghijklmnopqrst"), 1, 18, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghijklmnopqrst"), 1, 19, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghijklmnopqrst"), 1, 20, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghijklmnopqrst"), 10, 0, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghijklmnopqrst"), 10, 1, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghijklmnopqrst"), 10, 5, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghijklmnopqrst"), 10, 9, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghijklmnopqrst"), 10, 10, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghijklmnopqrst"), 10, 11, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghijklmnopqrst"), 19, 0, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghijklmnopqrst"), 19, 1, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghijklmnopqrst"), 19, 2, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghijklmnopqrst"), 20, 0, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghijklmnopqrst"), 20, 1, 0);
    test(S("abcdefghijklmnopqrst"), 21, 0, S("abcdefghijklmnopqrst"), 21, 0, 0);
}

template<class S>
void test55()
{
    test_npos(S(""), 0, 0, S(""), 0, 0);
    test_npos(S(""), 0, 0, S("abcde"), 0, -5);
    test_npos(S("abcde"), 0, 0, S("abcdefghij"), 0, -10);
    test_npos(S("abcde"), 0, 0, S("abcdefghij"), 1, -9);
    test_npos(S("abcde"), 0, 0, S("abcdefghij"), 5, -5);
}

int main(int, char**)
{
    {
    typedef std::string S;
    test0<S>();
    test1<S>();
    test2<S>();
    test3<S>();
    test4<S>();
    test5<S>();
    test6<S>();
    test7<S>();
    test8<S>();
    test9<S>();
    test10<S>();
    test11<S>();
    test12<S>();
    test13<S>();
    test14<S>();
    test15<S>();
    test16<S>();
    test17<S>();
    test18<S>();
    test19<S>();
    test20<S>();
    test21<S>();
    test22<S>();
    test23<S>();
    test24<S>();
    test25<S>();
    test26<S>();
    test27<S>();
    test28<S>();
    test29<S>();
    test30<S>();
    test31<S>();
    test32<S>();
    test33<S>();
    test34<S>();
    test35<S>();
    test36<S>();
    test37<S>();
    test38<S>();
    test39<S>();
    test40<S>();
    test41<S>();
    test42<S>();
    test43<S>();
    test44<S>();
    test45<S>();
    test46<S>();
    test47<S>();
    test48<S>();
    test49<S>();
    test50<S>();
    test51<S>();
    test52<S>();
    test53<S>();
    test54<S>();
    test55<S>();
    }
#if TEST_STD_VER >= 11
    {
    typedef std::basic_string<char, std::char_traits<char>, min_allocator<char>> S;
    test0<S>();
    test1<S>();
    test2<S>();
    test3<S>();
    test4<S>();
    test5<S>();
    test6<S>();
    test7<S>();
    test8<S>();
    test9<S>();
    test10<S>();
    test11<S>();
    test12<S>();
    test13<S>();
    test14<S>();
    test15<S>();
    test16<S>();
    test17<S>();
    test18<S>();
    test19<S>();
    test20<S>();
    test21<S>();
    test22<S>();
    test23<S>();
    test24<S>();
    test25<S>();
    test26<S>();
    test27<S>();
    test28<S>();
    test29<S>();
    test30<S>();
    test31<S>();
    test32<S>();
    test33<S>();
    test34<S>();
    test35<S>();
    test36<S>();
    test37<S>();
    test38<S>();
    test39<S>();
    test40<S>();
    test41<S>();
    test42<S>();
    test43<S>();
    test44<S>();
    test45<S>();
    test46<S>();
    test47<S>();
    test48<S>();
    test49<S>();
    test50<S>();
    test51<S>();
    test52<S>();
    test53<S>();
    test54<S>();
    test55<S>();
    }
#endif

  return 0;
}
